"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.CachedMethods = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.Distribution = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.additionalBehaviors = [];
        this.boundOrigins = [];
        this.originGroups = [];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        const originId = this.addOrigin(props.defaultBehavior.origin);
        this.defaultBehavior = new cache_behavior_1.CacheBehavior(originId, { pathPattern: '*', ...props.defaultBehavior });
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_a = props.errorResponses) !== null && _a !== void 0 ? _a : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig: {
                enabled: true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                originGroups: core_1.Lazy.anyValue({ produce: () => this.renderOriginGroups() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
                customErrorResponses: this.renderErrorResponses(),
                priceClass: (_b = props.priceClass) !== null && _b !== void 0 ? _b : undefined,
            } });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        const originId = this.addOrigin(origin);
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior(originId, { pathPattern, ...behaviorOptions }));
    }
    addOrigin(origin, isFailoverOrigin = false) {
        const existingOrigin = this.boundOrigins.find(boundOrigin => boundOrigin.origin === origin);
        if (existingOrigin) {
            return existingOrigin.originId;
        }
        else {
            const originIndex = this.boundOrigins.length + 1;
            const scope = new core_1.Construct(this, `Origin${originIndex}`);
            const originId = scope.node.uniqueId;
            const originBindConfig = origin.bind(scope, { originId });
            this.boundOrigins.push({ origin, originId, ...originBindConfig });
            if (originBindConfig.failoverConfig) {
                if (isFailoverOrigin) {
                    throw new Error('An Origin cannot use an Origin with its own failover configuration as its fallback origin!');
                }
                const failoverOriginId = this.addOrigin(originBindConfig.failoverConfig.failoverOrigin, true);
                this.addOriginGroup(originBindConfig.failoverConfig.statusCodes, originId, failoverOriginId);
            }
            return originId;
        }
    }
    addOriginGroup(statusCodes, originId, failoverOriginId) {
        statusCodes = statusCodes !== null && statusCodes !== void 0 ? statusCodes : [500, 502, 503, 504];
        if (statusCodes.length === 0) {
            throw new Error('fallbackStatusCodes cannot be empty');
        }
        const groupIndex = this.originGroups.length + 1;
        this.originGroups.push({
            failoverCriteria: {
                statusCodes: {
                    items: statusCodes,
                    quantity: statusCodes.length,
                },
            },
            id: new core_1.Construct(this, `OriginGroup${groupIndex}`).node.uniqueId,
            members: {
                items: [
                    { originId },
                    { originId: failoverOriginId },
                ],
                quantity: 2,
            },
        });
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.boundOrigins.forEach(boundOrigin => {
            if (boundOrigin.originProperty) {
                renderedOrigins.push(boundOrigin.originProperty);
            }
        });
        return renderedOrigins;
    }
    renderOriginGroups() {
        return this.originGroups.length === 0
            ? undefined
            : {
                items: this.originGroups,
                quantity: this.originGroups.length,
            };
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2018,
        };
    }
}
exports.Distribution = Distribution;
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The HTTP methods that the Behavior will cache requests on.
 */
class CachedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.CachedMethods = CachedMethods;
/** HEAD and GET */
CachedMethods.CACHE_GET_HEAD = new CachedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
CachedMethods.CACHE_GET_HEAD_OPTIONS = new CachedMethods(['GET', 'HEAD', 'OPTIONS']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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