"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphQLApi = exports.IamResource = exports.SchemaDefinition = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const fs_1 = require("fs");
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const data_source_1 = require("./data-source");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * Enum containing the different modes of schema definition
 */
var SchemaDefinition;
(function (SchemaDefinition) {
    /**
     * Define schema through functions like addType, addQuery, etc.
     */
    SchemaDefinition["CODE"] = "CODE";
    /**
     * Define schema in a file, i.e. schema.graphql
     */
    SchemaDefinition["FILE"] = "FILE";
})(SchemaDefinition = exports.SchemaDefinition || (exports.SchemaDefinition = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: this.formatAdditionalAuthenticationProviders(props),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schemaMode = props.schemaDefinition;
        if (defaultAuthorizationType === AuthorizationType.API_KEY || ((_h = (_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) === null || _h === void 0 ? void 0 : _h.some((authMode) => authMode.authorizationType === AuthorizationType.API_KEY))) {
            const apiKeyConfig = ((_k = (_j = props.authorizationConfig) === null || _j === void 0 ? void 0 : _j.defaultAuthorization) === null || _k === void 0 ? void 0 : _k.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this._apiKey = this.createAPIKey(apiKeyConfig);
        }
        this.schema = this.defineSchema(props.schemaDefinitionFile);
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new data_source_1.NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new data_source_1.DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name, description, endpoint) {
        return new data_source_1.HttpDataSource(this, `${name}DS`, {
            api: this,
            description,
            endpoint,
            name,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new data_source_1.LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        return key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
    formatAdditionalAuthenticationProviders(props) {
        var _a;
        const authModes = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.additionalAuthorizationModes;
        return authModes ? this.formatAdditionalAuthorizationModes(authModes) : undefined;
    }
    /**
     * Sets schema defintiion to input if schema mode is configured with SchemaDefinition.CODE
     *
     * @param definition string that is the graphql representation of schema
     * @experimental temporary
     */
    updateDefinition(definition) {
        if (this.schemaMode != SchemaDefinition.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.schema.definition = definition;
    }
    /**
     * Define schema based on props configuration
     * @param file the file name/s3 location of Schema
     */
    defineSchema(file) {
        let definition;
        if (this.schemaMode == SchemaDefinition.FILE && !file) {
            throw new Error('schemaDefinitionFile must be configured if using FILE definition mode.');
        }
        else if (this.schemaMode == SchemaDefinition.FILE && file) {
            definition = fs_1.readFileSync(file).toString('UTF-8');
        }
        else if (this.schemaMode == SchemaDefinition.CODE && !file) {
            definition = '';
        }
        else if (this.schemaMode == SchemaDefinition.CODE && file) {
            throw new Error('definition mode CODE is incompatible with file definition. Change mode to FILE/S3 or unconfigure schemaDefinitionFile');
        }
        return new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
}
exports.GraphQLApi = GraphQLApi;
//# sourceMappingURL=data:application/json;base64,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