#!/usr/bin/env python3
#
#  __main__.py
"""
CLI entry point.
"""
#
#  Copyright © 2021 Dominic Davis-Foster <dominic@davis-foster.co.uk>
#
#  Permission is hereby granted, free of charge, to any person obtaining a copy
#  of this software and associated documentation files (the "Software"), to deal
#  in the Software without restriction, including without limitation the rights
#  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
#  copies of the Software, and to permit persons to whom the Software is
#  furnished to do so, subject to the following conditions:
#
#  The above copyright notice and this permission notice shall be included in all
#  copies or substantial portions of the Software.
#
#  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
#  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
#  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
#  DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
#  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
#  OR OTHER DEALINGS IN THE SOFTWARE.
#

# stdlib
import os.path
import sys
from typing import Optional

# 3rd party
import click
from consolekit import click_command
from consolekit.options import auto_default_argument, auto_default_option, flag_option
from domdf_python_tools.typing import PathLike

__all__ = ["main"]


def _configure(ctx, _, filename):
	# 3rd party
	import dom_toml

	# this package
	from simple503.config import Simple503ConfigParser

	if os.path.isfile(filename):
		config_file_contents = dom_toml.load(filename)

		if "simple503" in config_file_contents:
			ctx.default_map = Simple503ConfigParser().parse(
					config_file_contents["simple503"],
					set_defaults=False,
					)


@flag_option("--cleanup", help="Cleanup files generated by simple503 in the target directory, and exit.")
@click.option(
		"--config",
		type=click.Path(dir_okay=False),
		default="simple503.toml",
		callback=_configure,
		is_eager=True,
		expose_value=False,
		help="Read option defaults from the specified TOML file.",
		show_default=True,
		)
@flag_option(
		"-c/-C",
		"--copy/--no-copy",
		help="Copy files from the source to the destination, rather than moving them.",
		)
@flag_option(
		"-s/-S",
		"--sort/--no-sort",
		help="Sort the wheel files into per-project base directories.",
		)
@auto_default_option(
		"-B",
		"--base-url",
		type=click.STRING,
		help="The base URL for the Python package repository.",
		show_default=True,
		)
@auto_default_argument("target", type=click.STRING)
@click.argument("origin", type=click.STRING)
@click_command()
def main(
		origin: PathLike,
		target: Optional[PathLike] = None,
		base_url: str = '/',
		sort: bool = False,
		copy: bool = False,
		cleanup: bool = False,
		):
	"""
	Generate a PEP 503 Python package repository in TARGET for the wheels in ORIGIN.
	"""

	if cleanup:
		# this package
		from simple503 import cleanup as do_cleanup

		do_cleanup(target or origin)

	else:
		# this package
		from simple503 import make_simple

		make_simple(origin, target, base_url=base_url, sort=sort, copy=copy)


if __name__ == "__main__":
	sys.exit(main())
