"""@package solution_vis
Visualize the positive definiteness of the Hessian and contour plots of the objective function.
"""
import numpy as np
import os
import pandas as pd
# import matplotlib.pyplot as plt
from .mpl_import import *
import matplotlib.lines as mlines

from .vc_parameter_identification import vc_get_hessian
from .posdef_check import posdef_check
from .uvc_model import uvc_get_hessian, error_single_test_uvc
from .RESSPyLab import errorTest_scl


def hess_posdef_range(data, x, x_prct_variation, num_samples, model_type='Original'):
    """ Evaluates if the Hessian is positive definite in the provided range for each parameter.
    :param list data: Set of pd.DataFrames with stress-strain data.
    :param np.ndarray x: Parameters for Voce-Chaboche model.
    :param dict x_prct_variation: Defines the sample range relative to the initial value.
    :param int num_samples: Number of data points in each parameter range.
    :param str model_type: Indicates the Voce-Chaboche model to use, either 'Original' or 'Updated'.
    :return pd.DataFrame: Indicates if the Hessian is positive definite for each parameter value in the range.

    Notes:
        - Keys for x_prct_variation are the integer indices of the parameters, the values are arrays of length 2
        where the first value is the lower bound and the second value is the upper bound. E.g., [0.9, 1.1] indicates
        90% of the value and 110% of the value.
    """
    pos_def_history = dict()
    for i, x_vary in x_prct_variation.items():
        x_range = np.linspace(x_vary[0] * x[i], x_vary[1] * x[i], num_samples)
        pos_def = []
        for xj in x_range:
            x_new = x.copy()
            x_new[i] = xj
            if model_type == 'Original':
                hess = vc_get_hessian(x_new, data)
            else:
                hess = uvc_get_hessian(x_new, data)
            pos_def.append(posdef_check(hess))
        pos_def_history[i] = pos_def
    df = pd.DataFrame(pos_def_history)
    df.columns = [str(c) for c in df.columns]
    return df


def plot_posdef(x_prct_variation, x_labels, pos_def_history, plot_dir):
    """ Plots the results for positive definite check for each parameter.
    :param dict x_prct_variation: Defines the sample range relative to the initial value.
    :param dict x_labels: Identifies the parameters in the model.
    :param pd.DataFrame pos_def_history: Indicates if the Hessian is positive definite for each parameter value in
    the range.
    :param str plot_dir: Directory to save the .pdf plots indicating if the Hessian is positive definite.

    Notes:
        - Keys for x_prct_variation are the integer indices of the parameters, the values are arrays of length 2
        where the first value is the lower bound and the second value is the upper bound.
        - Keys in x_labels must match those in x_prct_variation.
        - pos_def_history should be generated by hess_posdef_range().
    """
    for i in x_prct_variation.keys():
        pos_def = pos_def_history[str(i)]

        def signer(xi):
            if xi is True:
                return 1
            else:
                return -1

        hess_sign = [signer(xj) for xj in pos_def]
        num_samples = len(pos_def)
        x_vary = x_prct_variation[i]
        x_range = np.linspace(x_vary[0], x_vary[1], num_samples)
        plt.figure(figsize=(1.5, 1.5))
        plt.plot(x_range, hess_sign, 'k')
        plt.ylabel(r'sign[min[$\lambda$]]')
        plt.xlabel(r'Prct. Ini. ' + x_labels[i])
        plt.tight_layout()
        f_name = os.path.join(plot_dir, '_'.join(['eig_plot', str(i)]) + '.pdf')
        plt.savefig(f_name, bbox_inches='tight')


def gen_function_surfaces(data, x, x_prct_variation, num_samples, model_type='Original'):
    """ Generates sets of surfaces for the unique pairs of parameters.
    :param list data: Set of pd.DataFrames with stress-strain data.
    :param np.ndarray x: Parameters for Voce-Chaboche model.
    :param dict x_prct_variation: Defines the sample range relative to the initial value.
    :param int num_samples: Number of data points in each parameter range.
    :param str model_type: Indicates the Voce-Chaboche model to use, either 'Original' or 'Updated'.
    :returns list: Ordered list of surfaces formed by evaluating the objective function over the sampled range.

    Notes:
        - Keys for x_prct_variation are the integer indices of the parameters, the values are arrays of length 2
        where the first value is the lower bound and the second value is the upper bound.
        - Only evaluates the (i, j) parameter combinations where i > j, otherwise fills in the entries with None.
    """
    function_surfaces = []
    for i, x_vary_i in x_prct_variation.items():
        # Choose the first parameter (i)
        f_results_i = []
        x_i_range = np.linspace(x_vary_i[0] * x[i], x_vary_i[1] * x[i], num_samples)
        for j, x_vary_j in x_prct_variation.items():
            if i <= j:
                # Don't do anything on the lower triangle
                f_results_i.append(None)
            else:
                # Now the second parameter (j)
                x_j_range = np.linspace(x_vary_j[0] * x[j], x_vary_j[1] * x[j], num_samples)
                res_mat = np.zeros((num_samples, num_samples))
                for i2, xi2 in enumerate(x_i_range):
                    for j2, xj2 in enumerate(x_j_range):
                        x_new = x.copy()
                        x_new[[i, j]] = [xi2, xj2]
                        for d in data:
                            if model_type == 'Original':
                                res_mat[i2, j2] += errorTest_scl(x_new, d)
                            else:
                                res_mat[i2, j2] += error_single_test_uvc(x_new, d)
                f_results_i.append(res_mat)
        function_surfaces.append(f_results_i)
    return function_surfaces


def plot_function_surfaces(x, x_prct_variation, num_samples, function_values, plot_dir, x_labels,
                           model_type='Original'):
    """ Creates a contour plot of the objective function surfaces for each unique parameter pair.
    :param np.ndarray x: Parameters for Voce-Chaboche model.
    :param dict x_prct_variation: Defines the sample range relative to the initial value.
    :param int num_samples: Number of data points in each parameter range.
    :param list function_values: Objective function surfaces for each unique pair of parameters.
    :param str plot_dir: Directory to save the surface plot.
    :param dict x_labels: Identifies the parameters in the model.
    :param str model_type: Indicates the Voce-Chaboche model to use, either 'Original' or 'Updated'.

    Notes:
        - Keys for x_prct_variation are the integer indices of the parameters, the values are arrays of length 2
        where the first value is the lower bound and the second value is the upper bound.
        - function_values should be generated from gen_function_surfaces()
    """

    def get_initial_cep_constraint(xx, x_vary, ind):
        """ Returns the g1 constraint values for varying x """
        x_new = xx.copy()
        f = []

        def constr(x2):
            n_backstresses = int(len(x2) - 6) / 2
            g = x2[4] * x2[5] - x2[2] * x2[3]
            for ii in range(0, n_backstresses):
                ck_ind = 6 + 2 * ii
                g += -x2[ck_ind]
            return g

        for xi in x_vary:
            x_new[ind] = xi
            f.append(constr(x_new))
        return f

    def get_initial_dcep_constraint(xx, x_vary, ind):
        """ Returns the g2 constraint values for varying x """
        x_new = xx.copy()
        f = []

        def constr2(x2):
            n_backstresses = int((len(x2) - 6) / 2)
            g = x2[3] ** 2 * x2[2] - x2[5] ** 2 * x2[4]
            for ii in range(0, n_backstresses):
                gk_ind = 7 + 2 * ii
                ck_ind = 6 + 2 * ii
                g += x2[ck_ind] * x2[gk_ind]
            return g

        for xi in x_vary:
            x_new[ind] = xi
            f.append(constr2(x_new))
        return f

    n_params = len(x_prct_variation)
    # Don't plot when j == i
    n_figs = n_params - 1
    # Should hopefully fit on an A4 paper
    fig = plt.figure(figsize=(6, 9))
    for i, param_i in enumerate(x_prct_variation):
        for j, param_j in enumerate(x_prct_variation):
            if j < i and (j != n_figs):
                x_vary_i = x_prct_variation[param_i]
                x_vary_j = x_prct_variation[param_j]
                x_i_range = np.linspace(x_vary_i[0] * x[param_i], x_vary_i[1] * x[param_i], num_samples)
                x_j_range = np.linspace(x_vary_j[0] * x[param_j], x_vary_j[1] * x[param_j], num_samples)

                ax = plt.subplot(n_figs, n_figs, i + n_figs * j)
                ax.contourf(x_i_range, x_j_range, function_values[i][j], cmap='Greys')
                ax.plot([x[param_i]], [x[param_j]], 'ko')

                if model_type == 'Updated':
                    lim_xi = get_initial_cep_constraint(x, x_i_range, param_i)
                    lim_xj = get_initial_cep_constraint(x, x_j_range, param_j)
                    lim2_xi = get_initial_dcep_constraint(x, x_i_range, param_i)
                    lim2_xj = get_initial_dcep_constraint(x, x_j_range, param_j)
                    ax.plot(x_i_range, lim_xi, 'k--')
                    ax.plot(lim_xj, x_j_range, 'k--')
                    ax.plot(x_i_range, lim2_xi, 'k-.')
                    ax.plot(lim2_xj, x_j_range, 'k-.')

                ax.set_ylim([min(x_j_range), max(x_j_range)])
                ax.set_xlim([min(x_i_range), max(x_i_range)])

                # Only label the subplots on the diagonal
                if j == i - 1:
                    ax.set_xlabel(x_labels[param_i])
                    ax.set_ylabel(x_labels[param_j])
                    ax.tick_params(direction='in')
                else:
                    ax.tick_params(direction='in', labelbottom=False, labelleft=False)

    # Explicitly make the legend
    legend_lines = [mlines.Line2D([], [], color='black', ls='None', marker='o', label=r'Solution Point')]
    if model_type == 'Updated':
        legend_lines.append(mlines.Line2D([], [], color='black', ls='--', label=r'$g_1$ Constraint Limits'))
        legend_lines.append(mlines.Line2D([], [], color='black', ls='-.', label=r'$g_2$ Constraint Limits'))
    fig.legend(handles=legend_lines, loc='lower left')
    plt.tight_layout()
    plt.subplots_adjust(wspace=0, hspace=0.5)
    f_name = os.path.join(plot_dir, 'combined_surf_plot.pdf')
    plt.savefig(f_name, bbox_inches='tight')
    return fig


def save_surf(x_prct_variation, surf, save_dir):
    """ Saves the objective function surfaces to files.
    :param dict x_prct_variation: Defines the sample range relative to the initial value.
    :param list surf: Objective function surfaces for each unique pair of parameters.
    :param str save_dir: Directory to save all the surface files.

    Notes:
        - Keys for x_prct_variation are the integer indices of the parameters, the values are arrays of length 2
        where the first value is the lower bound and the second value is the upper bound.
        - The surfaces are saved as save_dir/surf_[i]_[j].csv, where i and j are indices of the parameters.
    """
    for i, param_i in enumerate(x_prct_variation):
        for j, param_j in enumerate(x_prct_variation):
            if j < i:
                f_name = os.path.join(save_dir, '_'.join(['surf', str(param_i), str(param_j)]) + '.csv')
                np.savetxt(f_name, surf[i][j])


def solution_visualizations(data, x, x_variation, num_pts, x_labels, model_type, save_dir, plot_dir):
    """ Visualizations for positive definiteness of the Hessian and contour plots of the objective function.
    :param list data: Set of pd.DataFrames with stress-strain data.
    :param np.ndarray x: Parameters for Voce-Chaboche model.
    :param dict x_variation: Defines the sample range relative to the initial value.
    :param int num_pts: Number of data points in each parameter range.
    :param dict x_labels: Identifies the parameters in the model.
    :param str model_type: Indicates the Voce-Chaboche model to use, either 'Original' or 'Updated'.
    :param str save_dir: Directory to save result files.
    :param str plot_dir: Directory to save plots.

    Notes:
        - Keys for x_prct_variation are the integer indices of the parameters, the values are arrays of length 2
        where the first value is the lower bound and the second value is the upper bound.
    """
    # Generate the data
    print ('Checking positive definiteness of the Hessian...')
    pos_def_hessian = hess_posdef_range(data, x, x_variation, num_pts, model_type)
    pos_def_hessian.to_csv(os.path.join(save_dir, 'hessian_results.csv'))
    print ('Generating function surfaces...')
    surf = gen_function_surfaces(data, x, x_variation, num_pts, model_type)
    save_surf(x_variation, surf, save_dir)

    plot_posdef(x_variation, x_labels, pos_def_hessian, plot_dir)
    plot_function_surfaces(x, x_variation, num_pts, surf, plot_dir, x_labels, model_type)
    print ('Finished!')
    return [pos_def_hessian, surf]
