"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cr = require("@aws-cdk/custom-resources");
const path = require("path");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
let PriceMap = new Map([
    ['m6g.medium', '0.0385'],
    ['m6g.large', '0.077'],
    ['t4g.nano', '0.0042'],
    ['t4g.micro', '0.0084'],
    ['t4g.small', '0.0168'],
    ['t4g.medium', '0.0336'],
    ['t4g.large', '0.0672'],
    ['t4g.xlarge', '0.1344'],
    ['t4g.2xlarge', '0.2688'],
]);
/**
 * Represents the k3sCluster construct
 */
class Cluster extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        // VPC configuration
        const vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file 
        const k3sBucket = new s3.Bucket(this, 'k3sBucket', {
            removalPolicy: (_b = props.bucketRemovalPolicy) !== null && _b !== void 0 ? _b : cdk.RemovalPolicy.RETAIN,
        });
        // Delete S3 Object CustomResource
        if (props.bucketRemovalPolicy === cdk.RemovalPolicy.DESTROY) {
            const onEvent = new lambda.Function(this, 'onEventHandler', {
                runtime: lambda.Runtime.PYTHON_3_8,
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-handler')),
                handler: 'index.on_event',
            });
            const deleteS3ObjectProvider = new cr.Provider(this, 'deleteS3ObjectProvider', {
                onEventHandler: onEvent,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            const CRdeleteS3ObjectProvider = new cdk.CustomResource(this, 'CRdeleteS3ObjectProvider', {
                serviceToken: deleteS3ObjectProvider.serviceToken,
                properties: {
                    Bucket: k3sBucket.bucketName,
                },
            });
            CRdeleteS3ObjectProvider.node.addDependency(k3sBucket);
            k3sBucket.grantDelete(onEvent);
            k3sBucket.grantReadWrite(onEvent);
        }
        // control plane node Security Group      
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', {
            vpc,
            securityGroupName: 'k3s-controlplane-SG',
            allowAllOutbound: true,
        });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group      
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', {
            vpc,
            securityGroupName: 'k3-worker-SG',
            allowAllOutbound: true,
        });
        // for this prototype the workers are being placed in a public subnet 
        // ideally they should land on a private subnet 
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only 
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used 
        this.controlPlaneInstanceType = (_c = props.controlPlaneInstanceType) !== null && _c !== void 0 ? _c : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_d = props.workerInstanceType) !== null && _d !== void 0 ? _d : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_e = props.workerMinCapacity) !== null && _e !== void 0 ? _e : 3,
            spotPrice: props.spotWorkerNodes ? PriceMap.get(this.workerInstanceType.toString()) : undefined,
        });
        workerAsg.addUserData(`
       #!/bin/bash
       LOGFILE='/var/log/k3s.log'
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
       chmod +x k3s
       echo the bucket name is ${k3sBucket.bucketName} 
       aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
       (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
       --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64
 */
class AmiProvider {
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
/**
 * The VPC provider to create or import the VPC
 */
class VpcProvider {
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
//# sourceMappingURL=data:application/json;base64,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