"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericAsyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_lambda_event_sources_1 = require("aws-cdk-lib/aws-lambda-event-sources");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sns = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const sqs = require("aws-cdk-lib/aws-sqs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
var textractSync_1 = require("./textractSync");
Object.defineProperty(exports, "TextractGenericSyncSfnTask", { enumerable: true, get: function () { return textractSync_1.TextractGenericSyncSfnTask; } });
var textractDecider_1 = require("./textractDecider");
Object.defineProperty(exports, "TextractPOCDecider", { enumerable: true, get: function () { return textractDecider_1.TextractPOCDecider; } });
var textractClassificationConfigurator_1 = require("./textractClassificationConfigurator");
Object.defineProperty(exports, "TextractClassificationConfigurator", { enumerable: true, get: function () { return textractClassificationConfigurator_1.TextractClassificationConfigurator; } });
var textractOutputConfigToJSON_1 = require("./textractOutputConfigToJSON");
Object.defineProperty(exports, "TextractAsyncToJSON", { enumerable: true, get: function () { return textractOutputConfigToJSON_1.TextractAsyncToJSON; } });
var textractGenerateCSV_1 = require("./textractGenerateCSV");
Object.defineProperty(exports, "TextractGenerateCSV", { enumerable: true, get: function () { return textractGenerateCSV_1.TextractGenerateCSV; } });
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
class TextractGenericAsyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        const taskTokenTable = new dynamodb.Table(this, 'TextractTaskTokenTable', {
            partitionKey: {
                name: 'ID',
                type: dynamodb.AttributeType.STRING,
            },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            timeToLiveAttribute: 'ttltimestamp',
        });
        this.taskTokenTableName = taskTokenTable.tableName;
        const duration = 30;
        this.textractAsyncSQS = new sqs.Queue(this, 'StartTPSRequests', {
            visibilityTimeout: aws_cdk_lib_1.Duration.seconds(6 * duration),
        });
        this.textractPutOnSQSFunction = new lambda.DockerImageFunction(this, 'PutOnSQS', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/put_on_sqs/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(duration),
            environment: {
                LOG_LEVEL: lambdaLogLevel,
                SQS_QUEUE_URL: this.textractAsyncSQS.queueUrl,
            },
        });
        this.textractPutOnSQSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sqs:SendMessage'], resources: [this.textractAsyncSQS.queueArn] }));
        this.putOnSQSLambdaLogGroup = this.textractPutOnSQSFunction.logGroup;
        const putOnSQSFunctionInvoke = new tasks.LambdaInvoke(this, 'PutOnSQSFunctionInvoke', {
            lambdaFunction: this.textractPutOnSQSFunction,
        });
        this.textractAsyncSNSRole = new iam.Role(this, 'TextractAsyncSNSRole', {
            assumedBy: new iam.ServicePrincipal('textract.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSQSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSNSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonTextractFullAccess')],
        });
        this.textractAsyncSNS = new sns.Topic(this, 'TextractAsyncSNS');
        this.textractAsyncCallFunction = new lambda.DockerImageFunction(this, 'TextractAsyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async/')),
            memorySize: 128,
            environment: {
                NOTIFICATION_SNS: this.textractAsyncSNS.topicArn,
                NOTIFICATION_ROLE_ARN: this.textractAsyncSNSRole.roleArn,
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                SQS_QUEUE_URL: this.textractAsyncSQS.queueName,
                TEXTRACT_API: textractAPI,
            },
        });
        this.textractAsyncCallFunction.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.textractAsyncSQS));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        // this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: [this.textractAsyncSNS.topicArn] }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [taskTokenTable.tableArn] }));
        this.startTextractLambdaLogGroup = this.textractAsyncCallFunction.logGroup;
        this.textractAsyncReceiveSNSFunction = new lambda.DockerImageFunction(this, 'TextractAsyncSNSFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async_sns_listener/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        this.textractAsyncSNS.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.textractAsyncReceiveSNSFunction));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:GetItem'], resources: ['*'] }));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:Put*', 's3:List*'],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/${props.s3TempOutputPrefix}/*`],
        }));
        this.receiveStartSNSLambdaLogGroup = this.textractAsyncReceiveSNSFunction.logGroup;
        const workflow_chain = sfn.Chain.start(putOnSQSFunctionInvoke);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        const appName = this.node.tryGetContext('appName');
        const customMetricNamespace = 'TextractConstructGenericAsync';
        // OPERATIONAL
        const asyncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Duration',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                .whereString('message', '=', `textract_async_${textractAPI}_job_duration_in_ms:`),
            metricValue: '$durationMs',
        });
        const asyncDurationMetric = asyncDurationMetricFilter.metric({ statistic: 'avg' });
        const asyncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
            metricValue: '$pages',
        });
        const asyncNumberPagesMetric = asyncNumberPagesMetricFilter.metric({ statistic: 'sum' });
        // TODO: expose all filters as properties of Construct, so users can build their own metrics
        const asyncJobFinshedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsFinishedFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsFinished',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
            metricValue: '1',
        });
        const asyncJobFinshedMetric = asyncJobFinshedMetricFilter.metric({ statistic: 'sum' });
        const asyncJobStartedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsStartedFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsStarted',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message')
                .whereString('message', '=', `textract_async_${textractAPI}_job_started`),
            metricValue: '1',
        });
        const asyncJobStartedMetric = asyncJobStartedMetricFilter.metric({ statistic: 'sum' });
        const asyncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_send_to_process:`),
            metricValue: '$pages',
        });
        const asyncNumberPagesSendMetric = asyncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
        // OPERATIONAL STOP
        // CALCUATED OPERATIONAL METRICS
        const pagesPerSecond = new cloudwatch.MathExpression({
            expression: 'pages / (duration / 1000)',
            usingMetrics: {
                pages: asyncNumberPagesMetric,
                duration: asyncDurationMetric,
            },
        });
        const openJobs = new cloudwatch.MathExpression({
            expression: 'startedJobs - finishedJobs',
            usingMetrics: {
                startedJobs: asyncJobStartedMetric,
                finishedJobs: asyncJobFinshedMetric,
            },
        });
        // CALCUATED OPERATIONAL METRICS STOP
        const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Errors',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
            metricValue: '1',
        });
        const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'LimitExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
            metricValue: '1',
        });
        const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ThrottlingException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
            metricValue: '1',
        });
        const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ProvisionedThroughputExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
            metricValue: '1',
        });
        const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'InternalServerError',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
            metricValue: '1',
        });
        const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
        const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
        const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
        const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
        const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
        const textractStartDocumentTextThrottle = new cloudwatch.Metric({
            namespace: 'AWS/Textract',
            metricName: 'ThrottledCount',
            dimensionsMap: {
                Operation: 'StartDocumentTextDetection',
            },
            statistic: 'sum',
        });
        // DASHBOARD LAMBDA
        const dashboardWidth = 24;
        // const widgetStandardHeight=9;
        // DASHBOARD
        const dashboard = new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
            end: 'end',
            periodOverride: cloudwatch.PeriodOverride.AUTO,
            start: 'start',
            widgets: [
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'OpenJobs', left: [openJobs], width: Math.floor(dashboardWidth / 2) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [asyncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [asyncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [asyncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'JobsStartedAndFinished', left: [asyncJobFinshedMetric], right: [asyncJobStartedMetric], width: Math.floor(dashboardWidth / 4) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractAsyncCallFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Errors', left: [this.textractAsyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Invocations', left: [this.textractAsyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Throttles', left: [this.textractAsyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractAsyncReceiveSNSFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Errors', left: [this.textractAsyncReceiveSNSFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Invocations', left: [this.textractAsyncReceiveSNSFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Throttles', left: [this.textractAsyncReceiveSNSFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# SQS ', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesSent', left: [this.textractAsyncSQS.metricNumberOfMessagesSent()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesDeleted', left: [this.textractAsyncSQS.metricNumberOfMessagesDeleted()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesVisible', left: [this.textractAsyncSQS.metricApproximateNumberOfMessagesVisible()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesNotVisible', left: [this.textractAsyncSQS.metricApproximateNumberOfMessagesNotVisible()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsAgeOldestMessage', left: [this.textractAsyncSQS.metricApproximateAgeOfOldestMessage()] }),
                ],
            ],
        });
        this.dashboardName = dashboard.dashboardName;
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericAsyncSfnTask = TextractGenericAsyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericAsyncSfnTask[_a] = { fqn: "schadem-cdk-construct-textract-generic-async.TextractGenericAsyncSfnTask", version: "0.0.4" };
TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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