# This file is part of aiootp, an asynchronous pseudo-one-time-pad based
# crypto and anonymity library.
#
# Licensed under the AGPLv3: https://www.gnu.org/licenses/agpl-3.0.html
# Copyright © 2019-2021 Gonzo Investigatory Journalism Agency, LLC
#            <gonzo.development@protonmail.ch>
#           © 2019-2021 Richard Machado <rmlibre@riseup.net>
# All rights reserved.
#


__all__ = [
    "bits",
    "primes",
    "WORD_LIST",
    "ASCII_TABLE",
    "ASCII_TABLE_128",
    "BYTES_TABLE",
    "BASE_36_TABLE",
    "BASE_38_TABLE",
    "BASE_64_TABLE",
    "URL_SAFE_TABLE",
    "ONION_CHAR_TABLE",
    "ASCII_ALPHANUMERIC",
    "BasePrimeGroups",
]


ASCII_TABLE = """ !"#$%&'()*+,-./0123456789:;<=>?@"""
ASCII_TABLE += "ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`"
ASCII_TABLE += "abcdefghijklmnopqrstuvwxyz{|}~"


ASCII_TABLE_128 = """\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b"""
ASCII_TABLE_128 += """\x0c\r\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17"""
ASCII_TABLE_128 += """\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f !"#$%&\'()*+,"""
ASCII_TABLE_128 += """-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ"""
ASCII_TABLE_128 += """[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\x7f"""


ASCII_ALPHANUMERIC = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
ASCII_ALPHANUMERIC += "abcdefghijklmnopqrstuvwxyz"


BASE_36_TABLE = "0123456789abcdefghijklmnopqrstuvwxyz"


BASE_38_TABLE = "-0123456789_abcdefghijklmnopqrstuvwxyz"


BASE_64_TABLE = "+/0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
BASE_64_TABLE += "abcdefghijklmnopqrstuvwxyz"


ONION_CHAR_TABLE = "234567abcdefghijklmnopqrstuvwxyz"


URL_SAFE_TABLE = "-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmn"
URL_SAFE_TABLE += "opqrstuvwxyz"


BYTES_TABLE = [
    b"\x00",
    b"\x01",
    b"\x02",
    b"\x03",
    b"\x04",
    b"\x05",
    b"\x06",
    b"\x07",
    b"\x08",
    b"\t",
    b"\n",
    b"\x0b",
    b"\x0c",
    b"\r",
    b"\x0e",
    b"\x0f",
    b"\x10",
    b"\x11",
    b"\x12",
    b"\x13",
    b"\x14",
    b"\x15",
    b"\x16",
    b"\x17",
    b"\x18",
    b"\x19",
    b"\x1a",
    b"\x1b",
    b"\x1c",
    b"\x1d",
    b"\x1e",
    b"\x1f",
    b" ",
    b"!",
    b'"',
    b"#",
    b"$",
    b"%",
    b"&",
    b"'",
    b"(",
    b")",
    b"*",
    b"+",
    b",",
    b"-",
    b".",
    b"/",
    b"0",
    b"1",
    b"2",
    b"3",
    b"4",
    b"5",
    b"6",
    b"7",
    b"8",
    b"9",
    b":",
    b";",
    b"<",
    b"=",
    b">",
    b"?",
    b"@",
    b"A",
    b"B",
    b"C",
    b"D",
    b"E",
    b"F",
    b"G",
    b"H",
    b"I",
    b"J",
    b"K",
    b"L",
    b"M",
    b"N",
    b"O",
    b"P",
    b"Q",
    b"R",
    b"S",
    b"T",
    b"U",
    b"V",
    b"W",
    b"X",
    b"Y",
    b"Z",
    b"[",
    b"\\",
    b"]",
    b"^",
    b"_",
    b"`",
    b"a",
    b"b",
    b"c",
    b"d",
    b"e",
    b"f",
    b"g",
    b"h",
    b"i",
    b"j",
    b"k",
    b"l",
    b"m",
    b"n",
    b"o",
    b"p",
    b"q",
    b"r",
    b"s",
    b"t",
    b"u",
    b"v",
    b"w",
    b"x",
    b"y",
    b"z",
    b"{",
    b"|",
    b"}",
    b"~",
    b"\x7f",
    b"\x80",
    b"\x81",
    b"\x82",
    b"\x83",
    b"\x84",
    b"\x85",
    b"\x86",
    b"\x87",
    b"\x88",
    b"\x89",
    b"\x8a",
    b"\x8b",
    b"\x8c",
    b"\x8d",
    b"\x8e",
    b"\x8f",
    b"\x90",
    b"\x91",
    b"\x92",
    b"\x93",
    b"\x94",
    b"\x95",
    b"\x96",
    b"\x97",
    b"\x98",
    b"\x99",
    b"\x9a",
    b"\x9b",
    b"\x9c",
    b"\x9d",
    b"\x9e",
    b"\x9f",
    b"\xa0",
    b"\xa1",
    b"\xa2",
    b"\xa3",
    b"\xa4",
    b"\xa5",
    b"\xa6",
    b"\xa7",
    b"\xa8",
    b"\xa9",
    b"\xaa",
    b"\xab",
    b"\xac",
    b"\xad",
    b"\xae",
    b"\xaf",
    b"\xb0",
    b"\xb1",
    b"\xb2",
    b"\xb3",
    b"\xb4",
    b"\xb5",
    b"\xb6",
    b"\xb7",
    b"\xb8",
    b"\xb9",
    b"\xba",
    b"\xbb",
    b"\xbc",
    b"\xbd",
    b"\xbe",
    b"\xbf",
    b"\xc0",
    b"\xc1",
    b"\xc2",
    b"\xc3",
    b"\xc4",
    b"\xc5",
    b"\xc6",
    b"\xc7",
    b"\xc8",
    b"\xc9",
    b"\xca",
    b"\xcb",
    b"\xcc",
    b"\xcd",
    b"\xce",
    b"\xcf",
    b"\xd0",
    b"\xd1",
    b"\xd2",
    b"\xd3",
    b"\xd4",
    b"\xd5",
    b"\xd6",
    b"\xd7",
    b"\xd8",
    b"\xd9",
    b"\xda",
    b"\xdb",
    b"\xdc",
    b"\xdd",
    b"\xde",
    b"\xdf",
    b"\xe0",
    b"\xe1",
    b"\xe2",
    b"\xe3",
    b"\xe4",
    b"\xe5",
    b"\xe6",
    b"\xe7",
    b"\xe8",
    b"\xe9",
    b"\xea",
    b"\xeb",
    b"\xec",
    b"\xed",
    b"\xee",
    b"\xef",
    b"\xf0",
    b"\xf1",
    b"\xf2",
    b"\xf3",
    b"\xf4",
    b"\xf5",
    b"\xf6",
    b"\xf7",
    b"\xf8",
    b"\xf9",
    b"\xfa",
    b"\xfb",
    b"\xfc",
    b"\xfd",
    b"\xfe",
    b"\xff",
]


bits = {
    # A static table of powers of 2, for efficient, more readable logic
    # requiring bit and entropy calculations. Each entry is named after
    # how many bits the value is.
    0: 0x1,
    1: 0x2,
    2: 0x4,
    3: 0x8,
    4: 0x10,
    5: 0x20,
    6: 0x40,
    7: 0x80,
    8: 0x100,
    9: 0x200,
    10: 0x400,
    11: 0x800,
    12: 0x1000,
    13: 0x2000,
    14: 0x4000,
    15: 0x8000,
    16: 0x10000,
    17: 0x20000,
    18: 0x40000,
    19: 0x80000,
    20: 0x100000,
    21: 0x200000,
    22: 0x400000,
    23: 0x800000,
    24: 0x1000000,
    25: 0x2000000,
    26: 0x4000000,
    27: 0x8000000,
    28: 0x10000000,
    29: 0x20000000,
    30: 0x40000000,
    31: 0x80000000,
    32: 0x100000000,
    33: 0x200000000,
    34: 0x400000000,
    35: 0x800000000,
    36: 0x1000000000,
    37: 0x2000000000,
    38: 0x4000000000,
    39: 0x8000000000,
    40: 0x10000000000,
    41: 0x20000000000,
    42: 0x40000000000,
    43: 0x80000000000,
    44: 0x100000000000,
    45: 0x200000000000,
    46: 0x400000000000,
    47: 0x800000000000,
    48: 0x1000000000000,
    49: 0x2000000000000,
    50: 0x4000000000000,
    51: 0x8000000000000,
    52: 0x10000000000000,
    53: 0x20000000000000,
    54: 0x40000000000000,
    55: 0x80000000000000,
    56: 0x100000000000000,
    57: 0x200000000000000,
    58: 0x400000000000000,
    59: 0x800000000000000,
    60: 0x1000000000000000,
    61: 0x2000000000000000,
    62: 0x4000000000000000,
    63: 0x8000000000000000,
    64: 0x10000000000000000,
    65: 0x20000000000000000,
    66: 0x40000000000000000,
    67: 0x80000000000000000,
    68: 0x100000000000000000,
    69: 0x200000000000000000,
    70: 0x400000000000000000,
    71: 0x800000000000000000,
    72: 0x1000000000000000000,
    73: 0x2000000000000000000,
    74: 0x4000000000000000000,
    75: 0x8000000000000000000,
    76: 0x10000000000000000000,
    77: 0x20000000000000000000,
    78: 0x40000000000000000000,
    79: 0x80000000000000000000,
    80: 0x100000000000000000000,
    81: 0x200000000000000000000,
    82: 0x400000000000000000000,
    83: 0x800000000000000000000,
    84: 0x1000000000000000000000,
    85: 0x2000000000000000000000,
    86: 0x4000000000000000000000,
    87: 0x8000000000000000000000,
    88: 0x10000000000000000000000,
    89: 0x20000000000000000000000,
    90: 0x40000000000000000000000,
    91: 0x80000000000000000000000,
    92: 0x100000000000000000000000,
    93: 0x200000000000000000000000,
    94: 0x400000000000000000000000,
    95: 0x800000000000000000000000,
    96: 0x1000000000000000000000000,
    97: 0x2000000000000000000000000,
    98: 0x4000000000000000000000000,
    99: 0x8000000000000000000000000,
    100: 0x10000000000000000000000000,
    101: 0x20000000000000000000000000,
    102: 0x40000000000000000000000000,
    103: 0x80000000000000000000000000,
    104: 0x100000000000000000000000000,
    105: 0x200000000000000000000000000,
    106: 0x400000000000000000000000000,
    107: 0x800000000000000000000000000,
    108: 0x1000000000000000000000000000,
    109: 0x2000000000000000000000000000,
    110: 0x4000000000000000000000000000,
    111: 0x8000000000000000000000000000,
    112: 0x10000000000000000000000000000,
    113: 0x20000000000000000000000000000,
    114: 0x40000000000000000000000000000,
    115: 0x80000000000000000000000000000,
    116: 0x100000000000000000000000000000,
    117: 0x200000000000000000000000000000,
    118: 0x400000000000000000000000000000,
    119: 0x800000000000000000000000000000,
    120: 0x1000000000000000000000000000000,
    121: 0x2000000000000000000000000000000,
    122: 0x4000000000000000000000000000000,
    123: 0x8000000000000000000000000000000,
    124: 0x10000000000000000000000000000000,
    125: 0x20000000000000000000000000000000,
    126: 0x40000000000000000000000000000000,
    127: 0x80000000000000000000000000000000,
    128: 0x100000000000000000000000000000000,
    129: 0x200000000000000000000000000000000,
    130: 0x400000000000000000000000000000000,
    131: 0x800000000000000000000000000000000,
    132: 0x1000000000000000000000000000000000,
    133: 0x2000000000000000000000000000000000,
    134: 0x4000000000000000000000000000000000,
    135: 0x8000000000000000000000000000000000,
    136: 0x10000000000000000000000000000000000,
    137: 0x20000000000000000000000000000000000,
    138: 0x40000000000000000000000000000000000,
    139: 0x80000000000000000000000000000000000,
    140: 0x100000000000000000000000000000000000,
    141: 0x200000000000000000000000000000000000,
    142: 0x400000000000000000000000000000000000,
    143: 0x800000000000000000000000000000000000,
    144: 0x1000000000000000000000000000000000000,
    145: 0x2000000000000000000000000000000000000,
    146: 0x4000000000000000000000000000000000000,
    147: 0x8000000000000000000000000000000000000,
    148: 0x10000000000000000000000000000000000000,
    149: 0x20000000000000000000000000000000000000,
    150: 0x40000000000000000000000000000000000000,
    151: 0x80000000000000000000000000000000000000,
    152: 0x100000000000000000000000000000000000000,
    153: 0x200000000000000000000000000000000000000,
    154: 0x400000000000000000000000000000000000000,
    155: 0x800000000000000000000000000000000000000,
    156: 0x1000000000000000000000000000000000000000,
    157: 0x2000000000000000000000000000000000000000,
    158: 0x4000000000000000000000000000000000000000,
    159: 0x8000000000000000000000000000000000000000,
    160: 0x10000000000000000000000000000000000000000,
    161: 0x20000000000000000000000000000000000000000,
    162: 0x40000000000000000000000000000000000000000,
    163: 0x80000000000000000000000000000000000000000,
    164: 0x100000000000000000000000000000000000000000,
    165: 0x200000000000000000000000000000000000000000,
    166: 0x400000000000000000000000000000000000000000,
    167: 0x800000000000000000000000000000000000000000,
    168: 0x1000000000000000000000000000000000000000000,
    169: 0x2000000000000000000000000000000000000000000,
    170: 0x4000000000000000000000000000000000000000000,
    171: 0x8000000000000000000000000000000000000000000,
    172: 0x10000000000000000000000000000000000000000000,
    173: 0x20000000000000000000000000000000000000000000,
    174: 0x40000000000000000000000000000000000000000000,
    175: 0x80000000000000000000000000000000000000000000,
    176: 0x100000000000000000000000000000000000000000000,
    177: 0x200000000000000000000000000000000000000000000,
    178: 0x400000000000000000000000000000000000000000000,
    179: 0x800000000000000000000000000000000000000000000,
    180: 0x1000000000000000000000000000000000000000000000,
    181: 0x2000000000000000000000000000000000000000000000,
    182: 0x4000000000000000000000000000000000000000000000,
    183: 0x8000000000000000000000000000000000000000000000,
    184: 0x10000000000000000000000000000000000000000000000,
    185: 0x20000000000000000000000000000000000000000000000,
    186: 0x40000000000000000000000000000000000000000000000,
    187: 0x80000000000000000000000000000000000000000000000,
    188: 0x100000000000000000000000000000000000000000000000,
    189: 0x200000000000000000000000000000000000000000000000,
    190: 0x400000000000000000000000000000000000000000000000,
    191: 0x800000000000000000000000000000000000000000000000,
    192: 0x1000000000000000000000000000000000000000000000000,
    193: 0x2000000000000000000000000000000000000000000000000,
    194: 0x4000000000000000000000000000000000000000000000000,
    195: 0x8000000000000000000000000000000000000000000000000,
    196: 0x10000000000000000000000000000000000000000000000000,
    197: 0x20000000000000000000000000000000000000000000000000,
    198: 0x40000000000000000000000000000000000000000000000000,
    199: 0x80000000000000000000000000000000000000000000000000,
    200: 0x100000000000000000000000000000000000000000000000000,
    201: 0x200000000000000000000000000000000000000000000000000,
    202: 0x400000000000000000000000000000000000000000000000000,
    203: 0x800000000000000000000000000000000000000000000000000,
    204: 0x1000000000000000000000000000000000000000000000000000,
    205: 0x2000000000000000000000000000000000000000000000000000,
    206: 0x4000000000000000000000000000000000000000000000000000,
    207: 0x8000000000000000000000000000000000000000000000000000,
    208: 0x10000000000000000000000000000000000000000000000000000,
    209: 0x20000000000000000000000000000000000000000000000000000,
    210: 0x40000000000000000000000000000000000000000000000000000,
    211: 0x80000000000000000000000000000000000000000000000000000,
    212: 0x100000000000000000000000000000000000000000000000000000,
    213: 0x200000000000000000000000000000000000000000000000000000,
    214: 0x400000000000000000000000000000000000000000000000000000,
    215: 0x800000000000000000000000000000000000000000000000000000,
    216: 0x1000000000000000000000000000000000000000000000000000000,
    217: 0x2000000000000000000000000000000000000000000000000000000,
    218: 0x4000000000000000000000000000000000000000000000000000000,
    219: 0x8000000000000000000000000000000000000000000000000000000,
    220: 0x10000000000000000000000000000000000000000000000000000000,
    221: 0x20000000000000000000000000000000000000000000000000000000,
    222: 0x40000000000000000000000000000000000000000000000000000000,
    223: 0x80000000000000000000000000000000000000000000000000000000,
    224: 0x100000000000000000000000000000000000000000000000000000000,
    225: 0x200000000000000000000000000000000000000000000000000000000,
    226: 0x400000000000000000000000000000000000000000000000000000000,
    227: 0x800000000000000000000000000000000000000000000000000000000,
    228: 0x1000000000000000000000000000000000000000000000000000000000,
    229: 0x2000000000000000000000000000000000000000000000000000000000,
    230: 0x4000000000000000000000000000000000000000000000000000000000,
    231: 0x8000000000000000000000000000000000000000000000000000000000,
    232: 0x10000000000000000000000000000000000000000000000000000000000,
    233: 0x20000000000000000000000000000000000000000000000000000000000,
    234: 0x40000000000000000000000000000000000000000000000000000000000,
    235: 0x80000000000000000000000000000000000000000000000000000000000,
    236: 0x100000000000000000000000000000000000000000000000000000000000,
    237: 0x200000000000000000000000000000000000000000000000000000000000,
    238: 0x400000000000000000000000000000000000000000000000000000000000,
    239: 0x800000000000000000000000000000000000000000000000000000000000,
    240: 0x1000000000000000000000000000000000000000000000000000000000000,
    241: 0x2000000000000000000000000000000000000000000000000000000000000,
    242: 0x4000000000000000000000000000000000000000000000000000000000000,
    243: 0x8000000000000000000000000000000000000000000000000000000000000,
    244: 0x10000000000000000000000000000000000000000000000000000000000000,
    245: 0x20000000000000000000000000000000000000000000000000000000000000,
    246: 0x40000000000000000000000000000000000000000000000000000000000000,
    247: 0x80000000000000000000000000000000000000000000000000000000000000,
    248: 0x100000000000000000000000000000000000000000000000000000000000000,
    249: 0x200000000000000000000000000000000000000000000000000000000000000,
    250: 0x400000000000000000000000000000000000000000000000000000000000000,
    251: 0x800000000000000000000000000000000000000000000000000000000000000,
    252: 0x1000000000000000000000000000000000000000000000000000000000000000,
    253: 0x2000000000000000000000000000000000000000000000000000000000000000,
    254: 0x4000000000000000000000000000000000000000000000000000000000000000,
    255: 0x8000000000000000000000000000000000000000000000000000000000000000,
    256: 0x10000000000000000000000000000000000000000000000000000000000000000,
    257: 0x20000000000000000000000000000000000000000000000000000000000000000,
    383: 0x800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    384: 0x1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    385: 0x2000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    511: 0x80000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    512: 0x100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    513: 0x200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    1023: 0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    1024: 0x10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    1025: 0x20000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    1535: 0x800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    1536: 0x1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    1537: 0x2000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    2047: 0x80000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    2048: 0x100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    2049: 0x200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    2559: 0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    2560: 0x10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    2561: 0x20000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    3071: 0x800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    3072: 0x1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    3073: 0x2000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    3583: 0x80000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    3584: 0x100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    3585: 0x200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    4095: 0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    4096: 0x10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
    4097: 0x20000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,
}


primes = {
    # A static table of primes for efficient and/or deterministic
    # functionality requiring prime numbers. The primes are organized
    # so there's two primes in a list in each element of the dictionary.
    # The first & last element of each list contains the first & last
    # prime of bit length ``key``.
    2: [0x2, 0x3],
    3: [0x5, 0x7],
    4: [0xb, 0xd],
    5: [0x13, 0x1f],
    6: [0x25, 0x3d],
    7: [0x43, 0x7f],
    8: [0x83, 0xfb],
    9: [0x101, 0x1fd],
    10: [0x209, 0x3fd],
    11: [0x407, 0x7f7],
    12: [0x805, 0xffd],
    13: [0x1003, 0x1fff],
    14: [0x2011, 0x3ffd],
    15: [0x401b, 0x7fed],
    16: [0x8003, 0xfff1],
    17: [0x10001, 0x1ffff],
    18: [0x2001d, 0x3fffb],
    19: [0x40003, 0x7ffff],
    20: [0x80015, 0xffffd],
    21: [0x100007, 0x1ffff7],
    22: [0x200011, 0x3ffffd],
    23: [0x40000f, 0x7ffff1],
    24: [0x800009, 0xfffffd],
    25: [0x100002b, 0x1ffffd9],
    26: [0x2000023, 0x3fffffb],
    27: [0x400000f, 0x7ffffd9],
    28: [0x800001d, 0xfffffc7],
    29: [0x10000003, 0x1ffffffd],
    30: [0x2000000b, 0x3fffffdd],
    31: [0x40000003, 0x7fffffff],
    32: [0x8000000b, 0xfffffffb],
    33: [0x10000000f, 0x1fffffff7],
    34: [0x200000011, 0x3ffffffd7],
    35: [0x400000019, 0x7ffffffe1],
    36: [0x800000035, 0xffffffffb],
    37: [0x100000001f, 0x1fffffffe7],
    38: [0x2000000009, 0x3fffffffd3],
    39: [0x4000000007, 0x7ffffffff9],
    40: [0x8000000017, 0xffffffffa9],
    41: [0x1000000000f, 0x1ffffffffeb],
    42: [0x2000000001b, 0x3fffffffff5],
    43: [0x4000000000f, 0x7ffffffffc7],
    44: [0x8000000001d, 0xfffffffffef],
    45: [0x100000000007, 0x1fffffffffc9],
    46: [0x20000000003b, 0x3fffffffffeb],
    47: [0x40000000000f, 0x7fffffffff8d],
    48: [0x800000000005, 0xffffffffffc5],
    49: [0x1000000000015, 0x1ffffffffffaf],
    50: [0x2000000000045, 0x3ffffffffffe5],
    51: [0x4000000000037, 0x7ffffffffff7f],
    52: [0x8000000000015, 0xfffffffffffd1],
    53: [0x10000000000015, 0x1fffffffffff91],
    54: [0x20000000000005, 0x3fffffffffffdf],
    55: [0x4000000000009f, 0x7fffffffffffc9],
    56: [0x80000000000003, 0xfffffffffffffb],
    57: [0x100000000000051, 0x1fffffffffffff3],
    58: [0x200000000000009, 0x3ffffffffffffe5],
    59: [0x400000000000045, 0x7ffffffffffffc9],
    60: [0x800000000000083, 0xfffffffffffffa3],
    61: [0x1000000000000021, 0x1fffffffffffffff],
    62: [0x200000000000000f, 0x3fffffffffffffc7],
    63: [0x4000000000000087, 0x7fffffffffffffe7],
    64: [0x800000000000001d, 0xffffffffffffffc5],
    65: [0x1000000000000000d, 0x1ffffffffffffffcf],
    66: [0x20000000000000083, 0x3fffffffffffffffb],
    67: [0x40000000000000009, 0x7ffffffffffffffed],
    68: [0x80000000000000003, 0xfffffffffffffffe9],
    69: [0x100000000000000021, 0x1fffffffffffffffed],
    70: [0x20000000000000001d, 0x3fffffffffffffffdd],
    71: [0x400000000000000019, 0x7fffffffffffffff19],
    72: [0x80000000000000000b, 0xffffffffffffffffa3],
    73: [0x100000000000000000f, 0x1ffffffffffffffffbb],
    74: [0x200000000000000001d, 0x3ffffffffffffffffdd],
    75: [0x4000000000000000025, 0x7ffffffffffffffff9f],
    76: [0x8000000000000000021, 0xffffffffffffffffff1],
    77: [0x1000000000000000000f, 0x1fffffffffffffffffdf],
    78: [0x2000000000000000000b, 0x3ffffffffffffffffff5],
    79: [0x40000000000000000007, 0x7fffffffffffffffffbd],
    80: [0x80000000000000000017, 0xffffffffffffffffffbf],
    81: [0x10000000000000000000d, 0x1ffffffffffffffffffcd],
    82: [0x200000000000000000011, 0x3ffffffffffffffffffc7],
    83: [0x400000000000000000009, 0x7ffffffffffffffffffc9],
    84: [0x80000000000000000004b, 0xfffffffffffffffffffdd],
    85: [0x1000000000000000000003, 0x1fffffffffffffffffffed],
    86: [0x20000000000000000000ab, 0x3fffffffffffffffffffdd],
    87: [0x400000000000000000001b, 0x7fffffffffffffffffffbd],
    88: [0x8000000000000000000027, 0xfffffffffffffffffffed5],
    89: [0x10000000000000000000007, 0x1ffffffffffffffffffffff],
    90: [0x2000000000000000000001d, 0x3ffffffffffffffffffffdf],
    91: [0x40000000000000000000085, 0x7ffffffffffffffffffffd3],
    92: [0x8000000000000000000003b, 0xfffffffffffffffffffffad],
    93: [0x100000000000000000000019, 0x1fffffffffffffffffffffe7],
    94: [0x200000000000000000000069, 0x3ffffffffffffffffffffffd],
    95: [0x400000000000000000000081, 0x7ffffffffffffffffffffff1],
    96: [0x800000000000000000000009, 0xffffffffffffffffffffffef],
    97: [0x100000000000000000000003d, 0x1ffffffffffffffffffffff73],
    98: [0x2000000000000000000000069, 0x3ffffffffffffffffffffffcd],
    99: [0x4000000000000000000000007, 0x7ffffffffffffffffffffff8d],
    100: [0x80000000000000000000000ff, 0xffffffffffffffffffffffff1],
    101: [0x10000000000000000000000115, 0x1fffffffffffffffffffffffbb],
    102: [0x20000000000000000000000051, 0x3fffffffffffffffffffffffdf],
    103: [0x4000000000000000000000010b, 0x7fffffffffffffffffffffff9f],
    104: [0x80000000000000000000000051, 0xffffffffffffffffffffffffef],
    105: [0x10000000000000000000000006f, 0x1fffffffffffffffffffffffff3],
    106: [0x200000000000000000000000027, 0x3ffffffffffffffffffffffff8b],
    107: [0x400000000000000000000000063, 0x7ffffffffffffffffffffffffff],
    108: [0x800000000000000000000000027, 0xfffffffffffffffffffffffffc5],
    109: [0x1000000000000000000000000021, 0x1fffffffffffffffffffffffffe1],
    110: [0x2000000000000000000000000093, 0x3fffffffffffffffffffffffffeb],
    111: [0x400000000000000000000000001b, 0x7fffffffffffffffffffffffffdb],
    112: [0x8000000000000000000000000033, 0xffffffffffffffffffffffffffb5],
    113: [0x10000000000000000000000000019, 0x1ffffffffffffffffffffffffff7b],
    114: [0x20000000000000000000000000119, 0x3fffffffffffffffffffffffffff5],
    115: [0x4000000000000000000000000002b, 0x7ffffffffffffffffffffffffffbd],
    116: [0x80000000000000000000000000047, 0xffffffffffffffffffffffffffffd],
    117: [
        0x100000000000000000000000000021,
        0x1ffffffffffffffffffffffffffee9,
    ],
    118: [
        0x20000000000000000000000000001d,
        0x3ffffffffffffffffffffffffffffb,
    ],
    119: [
        0x400000000000000000000000000019,
        0x7fffffffffffffffffffffffffffbb,
    ],
    120: [
        0x800000000000000000000000000009,
        0xffffffffffffffffffffffffffff89,
    ],
    121: [
        0x10000000000000000000000000001c3,
        0x1ffffffffffffffffffffffffffffb7,
    ],
    122: [
        0x2000000000000000000000000000029,
        0x3fffffffffffffffffffffffffffffd,
    ],
    123: [
        0x4000000000000000000000000000115,
        0x7ffffffffffffffffffffffffffffbd,
    ],
    124: [
        0x80000000000000000000000000000a5,
        0xfffffffffffffffffffffffffffffc5,
    ],
    125: [
        0x10000000000000000000000000000043,
        0x1ffffffffffffffffffffffffffffff7,
    ],
    126: [
        0x2000000000000000000000000000001b,
        0x3fffffffffffffffffffffffffffff77,
    ],
    127: [
        0x40000000000000000000000000000007,
        0x7fffffffffffffffffffffffffffffff,
    ],
    128: [
        0x8000000000000000000000000000001d,
        0xffffffffffffffffffffffffffffff61,
    ],
    129: [
        0x100000000000000000000000000000033,
        0x1ffffffffffffffffffffffffffffffe7,
    ],
    130: [
        0x200000000000000000000000000000011,
        0x3fffffffffffffffffffffffffffffffb,
    ],
    131: [
        0x4000000000000000000000000000000a9,
        0x7ffffffffffffffffffffffffffffffbb,
    ],
    132: [
        0x800000000000000000000000000000027,
        0xffffffffffffffffffffffffffffffea5,
    ],
    133: [
        0x1000000000000000000000000000000043,
        0x1fffffffffffffffffffffffffffffff9d,
    ],
    134: [
        0x200000000000000000000000000000001b,
        0x3fffffffffffffffffffffffffffffffd3,
    ],
    135: [
        0x400000000000000000000000000000001b,
        0x7fffffffffffffffffffffffffffffffd3,
    ],
    136: [
        0x8000000000000000000000000000000021,
        0xffffffffffffffffffffffffffffffff8f,
    ],
    137: [
        0x10000000000000000000000000000000055,
        0x1fffffffffffffffffffffffffffffffff3,
    ],
    138: [
        0x2000000000000000000000000000000009b,
        0x3ffffffffffffffffffffffffffffffff97,
    ],
    139: [
        0x40000000000000000000000000000000057,
        0x7ffffffffffffffffffffffffffffffff45,
    ],
    140: [
        0x8000000000000000000000000000000009b,
        0xfffffffffffffffffffffffffffffffffe5,
    ],
    141: [
        0x100000000000000000000000000000000025,
        0x1ffffffffffffffffffffffffffffffffff7,
    ],
    142: [
        0x200000000000000000000000000000000005,
        0x3fffffffffffffffffffffffffffffffff91,
    ],
    143: [
        0x4000000000000000000000000000000000d9,
        0x7fffffffffffffffffffffffffffffffffbb,
    ],
    144: [
        0x800000000000000000000000000000000005,
        0xffffffffffffffffffffffffffffffffffad,
    ],
    145: [
        0x10000000000000000000000000000000000af,
        0x1ffffffffffffffffffffffffffffffffff69,
    ],
    146: [
        0x200000000000000000000000000000000001b,
        0x3ffffffffffffffffffffffffffffffffff67,
    ],
    147: [
        0x4000000000000000000000000000000000055,
        0x7ffffffffffffffffffffffffffffffffff6f,
    ],
    148: [
        0x8000000000000000000000000000000000033,
        0xfffffffffffffffffffffffffffffffffff59,
    ],
    149: [
        0x1000000000000000000000000000000000005b,
        0x1fffffffffffffffffffffffffffffffffffe1,
    ],
    150: [
        0x20000000000000000000000000000000000045,
        0x3ffffffffffffffffffffffffffffffffffffd,
    ],
    151: [
        0x40000000000000000000000000000000000093,
        0x7fffffffffffffffffffffffffffffffffff3d,
    ],
    152: [
        0x8000000000000000000000000000000000002d,
        0xffffffffffffffffffffffffffffffffffffef,
    ],
    153: [
        0x1000000000000000000000000000000000000fd,
        0x1ffffffffffffffffffffffffffffffffffffbb,
    ],
    154: [
        0x20000000000000000000000000000000000005f,
        0x3ffffffffffffffffffffffffffffffffffff0d,
    ],
    155: [
        0x40000000000000000000000000000000000001b,
        0x7ffffffffffffffffffffffffffffffffffffe1,
    ],
    156: [
        0x80000000000000000000000000000000000000f,
        0xfffffffffffffffffffffffffffffffffffff71,
    ],
    157: [
        0x100000000000000000000000000000000000002d,
        0x1fffffffffffffffffffffffffffffffffffffed,
    ],
    158: [
        0x2000000000000000000000000000000000000045,
        0x3ffffffffffffffffffffffffffffffffffffff1,
    ],
    159: [
        0x4000000000000000000000000000000000000061,
        0x7fffffffffffffffffffffffffffffffffffffa5,
    ],
    160: [
        0x800000000000000000000000000000000000012b,
        0xffffffffffffffffffffffffffffffffffffffd1,
    ],
    161: [
        0x10000000000000000000000000000000000000007,
        0x1ffffffffffffffffffffffffffffffffffffff61,
    ],
    162: [
        0x2000000000000000000000000000000000000006b,
        0x3ffffffffffffffffffffffffffffffffffffff9b,
    ],
    163: [
        0x40000000000000000000000000000000000000013,
        0x7ffffffffffffffffffffffffffffffffffffffc9,
    ],
    164: [
        0x80000000000000000000000000000000000000015,
        0xfffffffffffffffffffffffffffffffffffffffc1,
    ],
    165: [
        0x100000000000000000000000000000000000000075,
        0x1fffffffffffffffffffffffffffffffffffffffe7,
    ],
    166: [
        0x20000000000000000000000000000000000000008d,
        0x3ffffffffffffffffffffffffffffffffffffffffb,
    ],
    167: [
        0x400000000000000000000000000000000000000055,
        0x7fffffffffffffffffffffffffffffffffffffff79,
    ],
    168: [
        0x800000000000000000000000000000000000000053,
        0xfffffffffffffffffffffffffffffffffffffffeff,
    ],
    169: [
        0x1000000000000000000000000000000000000000057,
        0x1fffffffffffffffffffffffffffffffffffffffd7d,
    ],
    170: [
        0x2000000000000000000000000000000000000000093,
        0x3ffffffffffffffffffffffffffffffffffffffff71,
    ],
    171: [
        0x4000000000000000000000000000000000000000031,
        0x7ffffffffffffffffffffffffffffffffffffffffed,
    ],
    172: [
        0x8000000000000000000000000000000000000000081,
        0xfffffffffffffffffffffffffffffffffffffffffa1,
    ],
    173: [
        0x10000000000000000000000000000000000000000069,
        0x1fffffffffffffffffffffffffffffffffffffffffc9,
    ],
    174: [
        0x2000000000000000000000000000000000000000004d,
        0x3ffffffffffffffffffffffffffffffffffffffffffd,
    ],
    175: [
        0x40000000000000000000000000000000000000000007,
        0x7fffffffffffffffffffffffffffffffffffffffff1b,
    ],
    176: [
        0x80000000000000000000000000000000000000000009,
        0xffffffffffffffffffffffffffffffffffffffffff17,
    ],
    177: [
        0x1000000000000000000000000000000000000000001ab,
        0x1fffffffffffffffffffffffffffffffffffffffffead,
    ],
    178: [
        0x20000000000000000000000000000000000000000004b,
        0x3ffffffffffffffffffffffffffffffffffffffffffd7,
    ],
    179: [
        0x400000000000000000000000000000000000000000057,
        0x7ffffffffffffffffffffffffffffffffffffffffffcf,
    ],
    180: [
        0x800000000000000000000000000000000000000000135,
        0xfffffffffffffffffffffffffffffffffffffffffffd1,
    ],
    181: [
        0x100000000000000000000000000000000000000000000f,
        0x1fffffffffffffffffffffffffffffffffffffffffff5b,
    ],
    182: [
        0x20000000000000000000000000000000000000000000a5,
        0x3fffffffffffffffffffffffffffffffffffffffffff5f,
    ],
    183: [
        0x4000000000000000000000000000000000000000000031,
        0x7fffffffffffffffffffffffffffffffffffffffffff6d,
    ],
    184: [
        0x80000000000000000000000000000000000000000000d7,
        0xffffffffffffffffffffffffffffffffffffffffffffdf,
    ],
    185: [
        0x1000000000000000000000000000000000000000000001b,
        0x1fffffffffffffffffffffffffffffffffffffffffffed1,
    ],
    186: [
        0x2000000000000000000000000000000000000000000009f,
        0x3fffffffffffffffffffffffffffffffffffffffffffe8d,
    ],
    187: [
        0x400000000000000000000000000000000000000000000cd,
        0x7ffffffffffffffffffffffffffffffffffffffffffffab,
    ],
    188: [
        0x8000000000000000000000000000000000000000000012f,
        0xfffffffffffffffffffffffffffffffffffffffffffff83,
    ],
    189: [
        0x100000000000000000000000000000000000000000000039,
        0x1fffffffffffffffffffffffffffffffffffffffffffffe7,
    ],
    190: [
        0x200000000000000000000000000000000000000000000023,
        0x3ffffffffffffffffffffffffffffffffffffffffffffff5,
    ],
    191: [
        0x400000000000000000000000000000000000000000000081,
        0x7fffffffffffffffffffffffffffffffffffffffffffffed,
    ],
    192: [
        0x800000000000000000000000000000000000000000000005,
        0xffffffffffffffffffffffffffffffffffffffffffffff13,
    ],
    193: [
        0x1000000000000000000000000000000000000000000000085,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffe1,
    ],
    194: [
        0x2000000000000000000000000000000000000000000000041,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffdf,
    ],
    195: [
        0x400000000000000000000000000000000000000000000001b,
        0x7ffffffffffffffffffffffffffffffffffffffffffffff79,
    ],
    196: [
        0x8000000000000000000000000000000000000000000000023,
        0xffffffffffffffffffffffffffffffffffffffffffffffff1,
    ],
    197: [
        0x10000000000000000000000000000000000000000000000015,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffb5,
    ],
    198: [
        0x2000000000000000000000000000000000000000000000006b,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffef,
    ],
    199: [
        0x4000000000000000000000000000000000000000000000000f,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffcf,
    ],
    200: [
        0x80000000000000000000000000000000000000000000000065,
        0xffffffffffffffffffffffffffffffffffffffffffffffffb5,
    ],
    201: [
        0x1000000000000000000000000000000000000000000000000eb,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffc9,
    ],
    202: [
        0x20000000000000000000000000000000000000000000000015f,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffff49,
    ],
    203: [
        0x400000000000000000000000000000000000000000000000043,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffff61,
    ],
    204: [
        0x80000000000000000000000000000000000000000000000000f,
        0xfffffffffffffffffffffffffffffffffffffffffffffffff59,
    ],
    205: [
        0x1000000000000000000000000000000000000000000000000007,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffaf,
    ],
    206: [
        0x2000000000000000000000000000000000000000000000000245,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffffffb,
    ],
    207: [
        0x4000000000000000000000000000000000000000000000000021,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffa5,
    ],
    208: [
        0x80000000000000000000000000000000000000000000000000cb,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffed5,
    ],
    209: [
        0x10000000000000000000000000000000000000000000000000177,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffdf,
    ],
    210: [
        0x2000000000000000000000000000000000000000000000000002f,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffffffd1,
    ],
    211: [
        0x40000000000000000000000000000000000000000000000000021,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffffff51,
    ],
    212: [
        0x80000000000000000000000000000000000000000000000000047,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffe9,
    ],
    213: [
        0x100000000000000000000000000000000000000000000000000039,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffd,
    ],
    214: [
        0x20000000000000000000000000000000000000000000000000004b,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffffff47,
    ],
    215: [
        0x400000000000000000000000000000000000000000000000000007,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffff63,
    ],
    216: [
        0x8000000000000000000000000000000000000000000000000000fb,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffe87,
    ],
    217: [
        0x10000000000000000000000000000000000000000000000000001a7,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffc3,
    ],
    218: [
        0x2000000000000000000000000000000000000000000000000000081,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffffffffdf,
    ],
    219: [
        0x40000000000000000000000000000000000000000000000000000a3,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffffffff87,
    ],
    220: [
        0x80000000000000000000000000000000000000000000000000000b9,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffb3,
    ],
    221: [
        0x100000000000000000000000000000000000000000000000000000d9,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffffd,
    ],
    222: [
        0x20000000000000000000000000000000000000000000000000000051,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffffffff8b,
    ],
    223: [
        0x40000000000000000000000000000000000000000000000000000031,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffff15,
    ],
    224: [
        0x800000000000000000000000000000000000000000000000000000bd,
        0xffffffffffffffffffffffffffffffffffffffffffffffffffffffc1,
    ],
    225: [
        0x1000000000000000000000000000000000000000000000000000002df,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffffcf,
    ],
    226: [
        0x200000000000000000000000000000000000000000000000000000077,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffffffffffb,
    ],
    227: [
        0x4000000000000000000000000000000000000000000000000000002df,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffe6b,
    ],
    228: [
        0x8000000000000000000000000000000000000000000000000000001e3,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffa3,
    ],
    229: [
        0x1000000000000000000000000000000000000000000000000000000003,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffa5,
    ],
    230: [
        0x20000000000000000000000000000000000000000000000000000000f9,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffffffffffe5,
    ],
    231: [
        0x4000000000000000000000000000000000000000000000000000000043,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffff5b,
    ],
    232: [
        0x8000000000000000000000000000000000000000000000000000000069,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffdc9,
    ],
    233: [
        0x10000000000000000000000000000000000000000000000000000000165,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffd,
    ],
    234: [
        0x200000000000000000000000000000000000000000000000000000001af,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffffffffffffad,
    ],
    235: [
        0x4000000000000000000000000000000000000000000000000000000002b,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffff1,
    ],
    236: [
        0x80000000000000000000000000000000000000000000000000000000051,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffff2f,
    ],
    237: [
        0x100000000000000000000000000000000000000000000000000000000019,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffff4b,
    ],
    238: [
        0x2000000000000000000000000000000000000000000000000000000000f9,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffffffffffff5f,
    ],
    239: [
        0x400000000000000000000000000000000000000000000000000000000043,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffa9,
    ],
    240: [
        0x80000000000000000000000000000000000000000000000000000000001d,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffe2d,
    ],
    241: [
        0x1000000000000000000000000000000000000000000000000000000000073,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd9,
    ],
    242: [
        0x2000000000000000000000000000000000000000000000000000000000105,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc1,
    ],
    243: [
        0x4000000000000000000000000000000000000000000000000000000000045,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffff7,
    ],
    244: [
        0x800000000000000000000000000000000000000000000000000000000003b,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffff43,
    ],
    245: [
        0x10000000000000000000000000000000000000000000000000000000000085,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffff5d,
    ],
    246: [
        0x2000000000000000000000000000000000000000000000000000000000013b,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffffffffffffff95,
    ],
    247: [
        0x40000000000000000000000000000000000000000000000000000000000151,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffaf,
    ],
    248: [
        0x8000000000000000000000000000000000000000000000000000000000003f,
        0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff13,
    ],
    249: [
        0x100000000000000000000000000000000000000000000000000000000000051,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffb5,
    ],
    250: [
        0x200000000000000000000000000000000000000000000000000000000000077,
        0x3ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff31,
    ],
    251: [
        0x400000000000000000000000000000000000000000000000000000000000019,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff7,
    ],
    252: [
        0x800000000000000000000000000000000000000000000000000000000000041,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff7f,
    ],
    253: [
        0x10000000000000000000000000000000000000000000000000000000000001a5,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffeef,
    ],
    254: [
        0x2000000000000000000000000000000000000000000000000000000000000027,
        0x3fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0b,
    ],
    255: [
        0x400000000000000000000000000000000000000000000000000000000000004f,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffed,
    ],
    256: [
        0x800000000000000000000000000000000000000000000000000000000000005f,
        0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff43,
    ],
    257: [
        0x10000000000000000000000000000000000000000000000000000000000000129,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffa3,
    ],
    383: [
        0x4000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000ff,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe1,
    ],
    384: [
        0x800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000171,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffec3,
    ],
    385: [
        0x10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e7,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffef7,
    ],
    511: [
        0x4000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000f,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff45,
    ],
    512: [
        0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006f,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffdc7,
    ],
    513: [
        0x10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004b,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe43,
    ],
    1023: [
        0x40000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000005a3,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe97,
    ],
    1024: [
        0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000483,
        0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff97,
    ],
    1025: [
        0x10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000283,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff58f,
    ],
    1535: [
        0x400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000109,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffadb,
    ],
    1536: [
        0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002bb,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff283,
    ],
    1537: [
        0x100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004b,
        0x1ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff51,
    ],
    2047: [
        0x40000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001033,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffab,
    ],
    2048: [
        0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000077f,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff9eb,
    ],
    2049: [
        0x1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003d5,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffa57,
    ],
    2559: [
        0x4000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000387,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd3b,
    ],
    2560: [
        0x80000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008eb,
        0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffb5,
    ],
    2561: [
        0x10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000387,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd99,
    ],
    3071: [
        0x4000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000c3,
        0x7fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff45,
    ],
    3072: [
        0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008f3,
        0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd1,
    ],
    3073: [
        0x100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000032d,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffcb7,
    ],
    3583: [
        0x40000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000927,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff3e7,
    ],
    3584: [
        0x800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010a3,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe53,
    ],
    3585: [
        0x100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000015,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe65,
    ],
    4095: [
        0x40000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001e9,
        0x7ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff9b1,
    ],
    4096: [
        0x8000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000243,
        0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff60b,
    ],
    4097: [
        0x100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000006e1,
        0x1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffcf1,
    ],
}


class BasePrimeGroups:
    """
    A collection of prime moduli & each of their respective primitive
    roots organized by bit length.
    """

    GENERATOR_32: int = 2
    GENERATOR_40: int = 3
    GENERATOR_48: int = 3
    GENERATOR_56: int = 2
    GENERATOR_64: int = 3
    GENERATOR_128: int = 11
    GENERATOR_192: int = 2
    GENERATOR_256: int = 2
    GENERATOR_384: int = 3
    GENERATOR_512: int = 7
    DH_GENERATOR_4096_GROUP_16: int = 2
    MOD_32: int = 0xfffffffb
    MOD_40: int = 0xfffffffc29
    MOD_48: int = 0xfffffffffab5
    MOD_56: int = 0xfffffffffff833
    MOD_64: int = 0xfffffffffffff9b9
    MOD_128: int = 0xffffffffffffffffffffffffffffe807
    MOD_192: int = 0xfffffffffffffffffffffffffffffffffffffffffffffa85
    MOD_256: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8dcb
    MOD_384: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff5e31
    MOD_512: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff9203
    DH_MOD_4096_GROUP_16: int = 0xffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca18217c32905e462e36ce3be39e772c180e86039b2783a2ec07a28fb5c55df06f4c52c9de2bcbf6955817183995497cea956ae515d2261898fa051015728e5a8aaac42dad33170d04507a33a85521abdf1cba64ecfb850458dbef0a8aea71575d060c7db3970f85a6e1e4c7abf5ae8cdb0933d71e8c94e04a25619dcee3d2261ad2ee6bf12ffa06d98a0864d87602733ec86a64521f2b18177b200cbbe117577a615d6c770988c0bad946e208e24fa074e5ab3143db5bfce0fd108e4b82d120a92108011a723c12a787e6d788719a10bdba5b2699c327186af4e23c1a946834b6150bda2583e9ca2ad44ce8dbbbc2db04de8ef92e8efc141fbecaa6287c59474e6bc05d99b2964fa090c3a2233ba186515be7ed1f612970cee2d7afb81bdd762170481cd0069127d5b05aa993b4ea988d8fddc186ffb7dc90a6c08f4df435c934063199ffffffffffffffff


WORD_LIST = [
    "abandon",
    "ability",
    "able",
    "about",
    "above",
    "absent",
    "absorb",
    "abstract",
    "absurd",
    "abuse",
    "access",
    "accident",
    "account",
    "accuse",
    "achieve",
    "acid",
    "acoustic",
    "acquire",
    "across",
    "act",
    "action",
    "actor",
    "actress",
    "actual",
    "adapt",
    "add",
    "addict",
    "address",
    "adjust",
    "admit",
    "adult",
    "advance",
    "advice",
    "aerobic",
    "affair",
    "afford",
    "afraid",
    "again",
    "age",
    "agent",
    "agree",
    "ahead",
    "aim",
    "air",
    "airport",
    "aisle",
    "alarm",
    "album",
    "alcohol",
    "alert",
    "alien",
    "all",
    "alley",
    "allow",
    "almost",
    "alone",
    "alpha",
    "already",
    "also",
    "alter",
    "always",
    "amateur",
    "amazing",
    "among",
    "amount",
    "amused",
    "analyst",
    "anchor",
    "ancient",
    "anger",
    "angle",
    "angry",
    "animal",
    "ankle",
    "announce",
    "annual",
    "another",
    "answer",
    "antenna",
    "antique",
    "anxiety",
    "any",
    "apart",
    "apology",
    "appear",
    "apple",
    "approve",
    "april",
    "arch",
    "arctic",
    "area",
    "arena",
    "argue",
    "arm",
    "armed",
    "armor",
    "army",
    "around",
    "arrange",
    "arrest",
    "arrive",
    "arrow",
    "art",
    "artefact",
    "artist",
    "artwork",
    "ask",
    "aspect",
    "assault",
    "asset",
    "assist",
    "assume",
    "asthma",
    "athlete",
    "atom",
    "attack",
    "attend",
    "attitude",
    "attract",
    "auction",
    "audit",
    "august",
    "aunt",
    "author",
    "auto",
    "autumn",
    "average",
    "avocado",
    "avoid",
    "awake",
    "aware",
    "away",
    "awesome",
    "awful",
    "awkward",
    "axis",
    "baby",
    "bachelor",
    "bacon",
    "badge",
    "bag",
    "balance",
    "balcony",
    "ball",
    "bamboo",
    "banana",
    "banner",
    "bar",
    "barely",
    "bargain",
    "barrel",
    "base",
    "basic",
    "basket",
    "battle",
    "beach",
    "bean",
    "beauty",
    "because",
    "become",
    "beef",
    "before",
    "begin",
    "behave",
    "behind",
    "believe",
    "below",
    "belt",
    "bench",
    "benefit",
    "best",
    "betray",
    "better",
    "between",
    "beyond",
    "bicycle",
    "bid",
    "bike",
    "bind",
    "biology",
    "bird",
    "birth",
    "bitter",
    "black",
    "blade",
    "blame",
    "blanket",
    "blast",
    "bleak",
    "bless",
    "blind",
    "blood",
    "blossom",
    "blouse",
    "blue",
    "blur",
    "blush",
    "board",
    "boat",
    "body",
    "boil",
    "bomb",
    "bone",
    "bonus",
    "book",
    "boost",
    "border",
    "boring",
    "borrow",
    "boss",
    "bottom",
    "bounce",
    "box",
    "boy",
    "bracket",
    "brain",
    "brand",
    "brass",
    "brave",
    "bread",
    "breeze",
    "brick",
    "bridge",
    "brief",
    "bright",
    "bring",
    "brisk",
    "broccoli",
    "broken",
    "bronze",
    "broom",
    "brother",
    "brown",
    "brush",
    "bubble",
    "buddy",
    "budget",
    "buffalo",
    "build",
    "bulb",
    "bulk",
    "bullet",
    "bundle",
    "bunker",
    "burden",
    "burger",
    "burst",
    "bus",
    "business",
    "busy",
    "butter",
    "buyer",
    "buzz",
    "cabbage",
    "cabin",
    "cable",
    "cactus",
    "cage",
    "cake",
    "call",
    "calm",
    "camera",
    "camp",
    "can",
    "canal",
    "cancel",
    "candy",
    "cannon",
    "canoe",
    "canvas",
    "canyon",
    "capable",
    "capital",
    "captain",
    "car",
    "carbon",
    "card",
    "cargo",
    "carpet",
    "carry",
    "cart",
    "case",
    "cash",
    "casino",
    "castle",
    "casual",
    "cat",
    "catalog",
    "catch",
    "category",
    "cattle",
    "caught",
    "cause",
    "caution",
    "cave",
    "ceiling",
    "celery",
    "cement",
    "census",
    "century",
    "cereal",
    "certain",
    "chair",
    "chalk",
    "champion",
    "change",
    "chaos",
    "chapter",
    "charge",
    "chase",
    "chat",
    "cheap",
    "check",
    "cheese",
    "chef",
    "cherry",
    "chest",
    "chicken",
    "chief",
    "child",
    "chimney",
    "choice",
    "choose",
    "chronic",
    "chuckle",
    "chunk",
    "churn",
    "cigar",
    "cinnamon",
    "circle",
    "citizen",
    "city",
    "civil",
    "claim",
    "clap",
    "clarify",
    "claw",
    "clay",
    "clean",
    "clerk",
    "clever",
    "click",
    "client",
    "cliff",
    "climb",
    "clinic",
    "clip",
    "clock",
    "clog",
    "close",
    "cloth",
    "cloud",
    "clown",
    "club",
    "clump",
    "cluster",
    "clutch",
    "coach",
    "coast",
    "coconut",
    "code",
    "coffee",
    "coil",
    "coin",
    "collect",
    "color",
    "column",
    "combine",
    "come",
    "comfort",
    "comic",
    "common",
    "company",
    "concert",
    "conduct",
    "confirm",
    "congress",
    "connect",
    "consider",
    "control",
    "convince",
    "cook",
    "cool",
    "copper",
    "copy",
    "coral",
    "core",
    "corn",
    "correct",
    "cost",
    "cotton",
    "couch",
    "country",
    "couple",
    "course",
    "cousin",
    "cover",
    "coyote",
    "crack",
    "cradle",
    "craft",
    "cram",
    "crane",
    "crash",
    "crater",
    "crawl",
    "crazy",
    "cream",
    "credit",
    "creek",
    "crew",
    "cricket",
    "crime",
    "crisp",
    "critic",
    "crop",
    "cross",
    "crouch",
    "crowd",
    "crucial",
    "cruel",
    "cruise",
    "crumble",
    "crunch",
    "crush",
    "cry",
    "crystal",
    "cube",
    "culture",
    "cup",
    "cupboard",
    "curious",
    "current",
    "curtain",
    "curve",
    "cushion",
    "custom",
    "cute",
    "cycle",
    "dad",
    "damage",
    "damp",
    "dance",
    "danger",
    "daring",
    "dash",
    "daughter",
    "dawn",
    "day",
    "deal",
    "debate",
    "debris",
    "decade",
    "december",
    "decide",
    "decline",
    "decorate",
    "decrease",
    "deer",
    "defense",
    "define",
    "defy",
    "degree",
    "delay",
    "deliver",
    "demand",
    "demise",
    "denial",
    "dentist",
    "deny",
    "depart",
    "depend",
    "deposit",
    "depth",
    "deputy",
    "derive",
    "describe",
    "desert",
    "design",
    "desk",
    "despair",
    "destroy",
    "detail",
    "detect",
    "develop",
    "device",
    "devote",
    "diagram",
    "dial",
    "diamond",
    "diary",
    "dice",
    "diesel",
    "diet",
    "differ",
    "digital",
    "dignity",
    "dilemma",
    "dinner",
    "dinosaur",
    "direct",
    "dirt",
    "disagree",
    "discover",
    "disease",
    "dish",
    "dismiss",
    "disorder",
    "display",
    "distance",
    "divert",
    "divide",
    "divorce",
    "dizzy",
    "doctor",
    "document",
    "dog",
    "doll",
    "dolphin",
    "domain",
    "donate",
    "donkey",
    "donor",
    "door",
    "dose",
    "double",
    "dove",
    "draft",
    "dragon",
    "drama",
    "drastic",
    "draw",
    "dream",
    "dress",
    "drift",
    "drill",
    "drink",
    "drip",
    "drive",
    "drop",
    "drum",
    "dry",
    "duck",
    "dumb",
    "dune",
    "during",
    "dust",
    "dutch",
    "duty",
    "dwarf",
    "dynamic",
    "eager",
    "eagle",
    "early",
    "earn",
    "earth",
    "easily",
    "east",
    "easy",
    "echo",
    "ecology",
    "economy",
    "edge",
    "edit",
    "educate",
    "effort",
    "egg",
    "eight",
    "either",
    "elbow",
    "elder",
    "electric",
    "elegant",
    "element",
    "elephant",
    "elevator",
    "elite",
    "else",
    "embark",
    "embody",
    "embrace",
    "emerge",
    "emotion",
    "employ",
    "empower",
    "empty",
    "enable",
    "enact",
    "end",
    "endless",
    "endorse",
    "enemy",
    "energy",
    "enforce",
    "engage",
    "engine",
    "enhance",
    "enjoy",
    "enlist",
    "enough",
    "enrich",
    "enroll",
    "ensure",
    "enter",
    "entire",
    "entry",
    "envelope",
    "episode",
    "equal",
    "equip",
    "era",
    "erase",
    "erode",
    "erosion",
    "error",
    "erupt",
    "escape",
    "essay",
    "essence",
    "estate",
    "eternal",
    "ethics",
    "evidence",
    "evil",
    "evoke",
    "evolve",
    "exact",
    "example",
    "excess",
    "exchange",
    "excite",
    "exclude",
    "excuse",
    "execute",
    "exercise",
    "exhaust",
    "exhibit",
    "exile",
    "exist",
    "exit",
    "exotic",
    "expand",
    "expect",
    "expire",
    "explain",
    "expose",
    "express",
    "extend",
    "extra",
    "eye",
    "eyebrow",
    "fabric",
    "face",
    "faculty",
    "fade",
    "faint",
    "faith",
    "fall",
    "false",
    "fame",
    "family",
    "famous",
    "fan",
    "fancy",
    "fantasy",
    "farm",
    "fashion",
    "fat",
    "fatal",
    "father",
    "fatigue",
    "fault",
    "favorite",
    "feature",
    "february",
    "federal",
    "fee",
    "feed",
    "feel",
    "female",
    "fence",
    "festival",
    "fetch",
    "fever",
    "few",
    "fiber",
    "fiction",
    "field",
    "figure",
    "file",
    "film",
    "filter",
    "final",
    "find",
    "fine",
    "finger",
    "finish",
    "fire",
    "firm",
    "first",
    "fiscal",
    "fish",
    "fit",
    "fitness",
    "fix",
    "flag",
    "flame",
    "flash",
    "flat",
    "flavor",
    "flee",
    "flight",
    "flip",
    "float",
    "flock",
    "floor",
    "flower",
    "fluid",
    "flush",
    "fly",
    "foam",
    "focus",
    "fog",
    "foil",
    "fold",
    "follow",
    "food",
    "foot",
    "force",
    "forest",
    "forget",
    "fork",
    "fortune",
    "forum",
    "forward",
    "fossil",
    "foster",
    "found",
    "fox",
    "fragile",
    "frame",
    "frequent",
    "fresh",
    "friend",
    "fringe",
    "frog",
    "front",
    "frost",
    "frown",
    "frozen",
    "fruit",
    "fuel",
    "fun",
    "funny",
    "furnace",
    "fury",
    "future",
    "gadget",
    "gain",
    "galaxy",
    "gallery",
    "game",
    "gap",
    "garage",
    "garbage",
    "garden",
    "garlic",
    "garment",
    "gas",
    "gasp",
    "gate",
    "gather",
    "gauge",
    "gaze",
    "general",
    "genius",
    "genre",
    "gentle",
    "genuine",
    "gesture",
    "ghost",
    "giant",
    "gift",
    "giggle",
    "ginger",
    "giraffe",
    "girl",
    "give",
    "glad",
    "glance",
    "glare",
    "glass",
    "glide",
    "glimpse",
    "globe",
    "gloom",
    "glory",
    "glove",
    "glow",
    "glue",
    "goat",
    "goddess",
    "gold",
    "good",
    "goose",
    "gorilla",
    "gospel",
    "gossip",
    "govern",
    "gown",
    "grab",
    "grace",
    "grain",
    "grant",
    "grape",
    "grass",
    "gravity",
    "great",
    "green",
    "grid",
    "grief",
    "grit",
    "grocery",
    "group",
    "grow",
    "grunt",
    "guard",
    "guess",
    "guide",
    "guilt",
    "guitar",
    "gun",
    "gym",
    "habit",
    "hair",
    "half",
    "hammer",
    "hamster",
    "hand",
    "happy",
    "harbor",
    "hard",
    "harsh",
    "harvest",
    "hat",
    "have",
    "hawk",
    "hazard",
    "head",
    "health",
    "heart",
    "heavy",
    "hedgehog",
    "height",
    "hello",
    "helmet",
    "help",
    "hen",
    "hero",
    "hidden",
    "high",
    "hill",
    "hint",
    "hip",
    "hire",
    "history",
    "hobby",
    "hockey",
    "hold",
    "hole",
    "holiday",
    "hollow",
    "home",
    "honey",
    "hood",
    "hope",
    "horn",
    "horror",
    "horse",
    "hospital",
    "host",
    "hotel",
    "hour",
    "hover",
    "hub",
    "huge",
    "human",
    "humble",
    "humor",
    "hundred",
    "hungry",
    "hunt",
    "hurdle",
    "hurry",
    "hurt",
    "husband",
    "hybrid",
    "ice",
    "icon",
    "idea",
    "identify",
    "idle",
    "ignore",
    "ill",
    "illegal",
    "illness",
    "image",
    "imitate",
    "immense",
    "immune",
    "impact",
    "impose",
    "improve",
    "impulse",
    "inch",
    "include",
    "income",
    "increase",
    "index",
    "indicate",
    "indoor",
    "industry",
    "infant",
    "inflict",
    "inform",
    "inhale",
    "inherit",
    "initial",
    "inject",
    "injury",
    "inmate",
    "inner",
    "innocent",
    "input",
    "inquiry",
    "insane",
    "insect",
    "inside",
    "inspire",
    "install",
    "intact",
    "interest",
    "into",
    "invest",
    "invite",
    "involve",
    "iron",
    "island",
    "isolate",
    "issue",
    "item",
    "ivory",
    "jacket",
    "jaguar",
    "jar",
    "jazz",
    "jealous",
    "jeans",
    "jelly",
    "jewel",
    "job",
    "join",
    "joke",
    "journey",
    "joy",
    "judge",
    "juice",
    "jump",
    "jungle",
    "junior",
    "junk",
    "just",
    "kangaroo",
    "keen",
    "keep",
    "ketchup",
    "key",
    "kick",
    "kid",
    "kidney",
    "kind",
    "kingdom",
    "kiss",
    "kit",
    "kitchen",
    "kite",
    "kitten",
    "kiwi",
    "knee",
    "knife",
    "knock",
    "know",
    "lab",
    "label",
    "labor",
    "ladder",
    "lady",
    "lake",
    "lamp",
    "language",
    "laptop",
    "large",
    "later",
    "latin",
    "laugh",
    "laundry",
    "lava",
    "law",
    "lawn",
    "lawsuit",
    "layer",
    "lazy",
    "leader",
    "leaf",
    "learn",
    "leave",
    "lecture",
    "left",
    "leg",
    "legal",
    "legend",
    "leisure",
    "lemon",
    "lend",
    "length",
    "lens",
    "leopard",
    "lesson",
    "letter",
    "level",
    "liar",
    "liberty",
    "library",
    "license",
    "life",
    "lift",
    "light",
    "like",
    "limb",
    "limit",
    "link",
    "lion",
    "liquid",
    "list",
    "little",
    "live",
    "lizard",
    "load",
    "loan",
    "lobster",
    "local",
    "lock",
    "logic",
    "lonely",
    "long",
    "loop",
    "lottery",
    "loud",
    "lounge",
    "love",
    "loyal",
    "lucky",
    "luggage",
    "lumber",
    "lunar",
    "lunch",
    "luxury",
    "lyrics",
    "machine",
    "mad",
    "magic",
    "magnet",
    "maid",
    "mail",
    "main",
    "major",
    "make",
    "mammal",
    "man",
    "manage",
    "mandate",
    "mango",
    "mansion",
    "manual",
    "maple",
    "marble",
    "march",
    "margin",
    "marine",
    "market",
    "marriage",
    "mask",
    "mass",
    "master",
    "match",
    "material",
    "math",
    "matrix",
    "matter",
    "maximum",
    "maze",
    "meadow",
    "mean",
    "measure",
    "meat",
    "mechanic",
    "medal",
    "media",
    "melody",
    "melt",
    "member",
    "memory",
    "mention",
    "menu",
    "mercy",
    "merge",
    "merit",
    "merry",
    "mesh",
    "message",
    "metal",
    "method",
    "middle",
    "midnight",
    "milk",
    "million",
    "mimic",
    "mind",
    "minimum",
    "minor",
    "minute",
    "miracle",
    "mirror",
    "misery",
    "miss",
    "mistake",
    "mix",
    "mixed",
    "mixture",
    "mobile",
    "model",
    "modify",
    "mom",
    "moment",
    "monitor",
    "monkey",
    "monster",
    "month",
    "moon",
    "moral",
    "more",
    "morning",
    "mosquito",
    "mother",
    "motion",
    "motor",
    "mountain",
    "mouse",
    "move",
    "movie",
    "much",
    "muffin",
    "mule",
    "multiply",
    "muscle",
    "museum",
    "mushroom",
    "music",
    "must",
    "mutual",
    "myself",
    "mystery",
    "myth",
    "naive",
    "name",
    "napkin",
    "narrow",
    "nasty",
    "nation",
    "nature",
    "near",
    "neck",
    "need",
    "negative",
    "neglect",
    "neither",
    "nephew",
    "nerve",
    "nest",
    "net",
    "network",
    "neutral",
    "never",
    "news",
    "next",
    "nice",
    "night",
    "noble",
    "noise",
    "nominee",
    "noodle",
    "normal",
    "north",
    "nose",
    "notable",
    "note",
    "nothing",
    "notice",
    "novel",
    "now",
    "nuclear",
    "number",
    "nurse",
    "nut",
    "oak",
    "obey",
    "object",
    "oblige",
    "obscure",
    "observe",
    "obtain",
    "obvious",
    "occur",
    "ocean",
    "october",
    "odor",
    "off",
    "offer",
    "office",
    "often",
    "oil",
    "okay",
    "old",
    "olive",
    "olympic",
    "omit",
    "once",
    "one",
    "onion",
    "online",
    "only",
    "open",
    "opera",
    "opinion",
    "oppose",
    "option",
    "orange",
    "orbit",
    "orchard",
    "order",
    "ordinary",
    "organ",
    "orient",
    "original",
    "orphan",
    "ostrich",
    "other",
    "outdoor",
    "outer",
    "output",
    "outside",
    "oval",
    "oven",
    "over",
    "own",
    "owner",
    "oxygen",
    "oyster",
    "ozone",
    "pact",
    "paddle",
    "page",
    "pair",
    "palace",
    "palm",
    "panda",
    "panel",
    "panic",
    "panther",
    "paper",
    "parade",
    "parent",
    "park",
    "parrot",
    "party",
    "pass",
    "patch",
    "path",
    "patient",
    "patrol",
    "pattern",
    "pause",
    "pave",
    "payment",
    "peace",
    "peanut",
    "pear",
    "peasant",
    "pelican",
    "pen",
    "penalty",
    "pencil",
    "people",
    "pepper",
    "perfect",
    "permit",
    "person",
    "pet",
    "phone",
    "photo",
    "phrase",
    "physical",
    "piano",
    "picnic",
    "picture",
    "piece",
    "pig",
    "pigeon",
    "pill",
    "pilot",
    "pink",
    "pioneer",
    "pipe",
    "pistol",
    "pitch",
    "pizza",
    "place",
    "planet",
    "plastic",
    "plate",
    "play",
    "please",
    "pledge",
    "pluck",
    "plug",
    "plunge",
    "poem",
    "poet",
    "point",
    "polar",
    "pole",
    "police",
    "pond",
    "pony",
    "pool",
    "popular",
    "portion",
    "position",
    "possible",
    "post",
    "potato",
    "pottery",
    "poverty",
    "powder",
    "power",
    "practice",
    "praise",
    "predict",
    "prefer",
    "prepare",
    "present",
    "pretty",
    "prevent",
    "price",
    "pride",
    "primary",
    "print",
    "priority",
    "prison",
    "private",
    "prize",
    "problem",
    "process",
    "produce",
    "profit",
    "program",
    "project",
    "promote",
    "proof",
    "property",
    "prosper",
    "protect",
    "proud",
    "provide",
    "public",
    "pudding",
    "pull",
    "pulp",
    "pulse",
    "pumpkin",
    "punch",
    "pupil",
    "puppy",
    "purchase",
    "purity",
    "purpose",
    "purse",
    "push",
    "put",
    "puzzle",
    "pyramid",
    "quality",
    "quantum",
    "quarter",
    "question",
    "quick",
    "quit",
    "quiz",
    "quote",
    "rabbit",
    "raccoon",
    "race",
    "rack",
    "radar",
    "radio",
    "rail",
    "rain",
    "raise",
    "rally",
    "ramp",
    "ranch",
    "random",
    "range",
    "rapid",
    "rare",
    "rate",
    "rather",
    "raven",
    "raw",
    "razor",
    "ready",
    "real",
    "reason",
    "rebel",
    "rebuild",
    "recall",
    "receive",
    "recipe",
    "record",
    "recycle",
    "reduce",
    "reflect",
    "reform",
    "refuse",
    "region",
    "regret",
    "regular",
    "reject",
    "relax",
    "release",
    "relief",
    "rely",
    "remain",
    "remember",
    "remind",
    "remove",
    "render",
    "renew",
    "rent",
    "reopen",
    "repair",
    "repeat",
    "replace",
    "report",
    "require",
    "rescue",
    "resemble",
    "resist",
    "resource",
    "response",
    "result",
    "retire",
    "retreat",
    "return",
    "reunion",
    "reveal",
    "review",
    "reward",
    "rhythm",
    "rib",
    "ribbon",
    "rice",
    "rich",
    "ride",
    "ridge",
    "rifle",
    "right",
    "rigid",
    "ring",
    "riot",
    "ripple",
    "risk",
    "ritual",
    "rival",
    "river",
    "road",
    "roast",
    "robot",
    "robust",
    "rocket",
    "romance",
    "roof",
    "rookie",
    "room",
    "rose",
    "rotate",
    "rough",
    "round",
    "route",
    "royal",
    "rubber",
    "rude",
    "rug",
    "rule",
    "run",
    "runway",
    "rural",
    "sad",
    "saddle",
    "sadness",
    "safe",
    "sail",
    "salad",
    "salmon",
    "salon",
    "salt",
    "salute",
    "same",
    "sample",
    "sand",
    "satisfy",
    "satoshi",
    "sauce",
    "sausage",
    "save",
    "say",
    "scale",
    "scan",
    "scare",
    "scatter",
    "scene",
    "scheme",
    "school",
    "science",
    "scissors",
    "scorpion",
    "scout",
    "scrap",
    "screen",
    "script",
    "scrub",
    "sea",
    "search",
    "season",
    "seat",
    "second",
    "secret",
    "section",
    "security",
    "seed",
    "seek",
    "segment",
    "select",
    "sell",
    "seminar",
    "senior",
    "sense",
    "sentence",
    "series",
    "service",
    "session",
    "settle",
    "setup",
    "seven",
    "shadow",
    "shaft",
    "shallow",
    "share",
    "shed",
    "shell",
    "sheriff",
    "shield",
    "shift",
    "shine",
    "ship",
    "shiver",
    "shock",
    "shoe",
    "shoot",
    "shop",
    "short",
    "shoulder",
    "shove",
    "shrimp",
    "shrug",
    "shuffle",
    "shy",
    "sibling",
    "sick",
    "side",
    "siege",
    "sight",
    "sign",
    "silent",
    "silk",
    "silly",
    "silver",
    "similar",
    "simple",
    "since",
    "sing",
    "siren",
    "sister",
    "situate",
    "six",
    "size",
    "skate",
    "sketch",
    "ski",
    "skill",
    "skin",
    "skirt",
    "skull",
    "slab",
    "slam",
    "sleep",
    "slender",
    "slice",
    "slide",
    "slight",
    "slim",
    "slogan",
    "slot",
    "slow",
    "slush",
    "small",
    "smart",
    "smile",
    "smoke",
    "smooth",
    "snack",
    "snake",
    "snap",
    "sniff",
    "snow",
    "soap",
    "soccer",
    "social",
    "sock",
    "soda",
    "soft",
    "solar",
    "soldier",
    "solid",
    "solution",
    "solve",
    "someone",
    "song",
    "soon",
    "sorry",
    "sort",
    "soul",
    "sound",
    "soup",
    "source",
    "south",
    "space",
    "spare",
    "spatial",
    "spawn",
    "speak",
    "special",
    "speed",
    "spell",
    "spend",
    "sphere",
    "spice",
    "spider",
    "spike",
    "spin",
    "spirit",
    "split",
    "spoil",
    "sponsor",
    "spoon",
    "sport",
    "spot",
    "spray",
    "spread",
    "spring",
    "spy",
    "square",
    "squeeze",
    "squirrel",
    "stable",
    "stadium",
    "staff",
    "stage",
    "stairs",
    "stamp",
    "stand",
    "start",
    "state",
    "stay",
    "steak",
    "steel",
    "stem",
    "step",
    "stereo",
    "stick",
    "still",
    "sting",
    "stock",
    "stomach",
    "stone",
    "stool",
    "story",
    "stove",
    "strategy",
    "street",
    "strike",
    "strong",
    "struggle",
    "student",
    "stuff",
    "stumble",
    "style",
    "subject",
    "submit",
    "subway",
    "success",
    "such",
    "sudden",
    "suffer",
    "sugar",
    "suggest",
    "suit",
    "summer",
    "sun",
    "sunny",
    "sunset",
    "super",
    "supply",
    "supreme",
    "sure",
    "surface",
    "surge",
    "surprise",
    "surround",
    "survey",
    "suspect",
    "sustain",
    "swallow",
    "swamp",
    "swap",
    "swarm",
    "swear",
    "sweet",
    "swift",
    "swim",
    "swing",
    "switch",
    "sword",
    "symbol",
    "symptom",
    "syrup",
    "system",
    "table",
    "tackle",
    "tag",
    "tail",
    "talent",
    "talk",
    "tank",
    "tape",
    "target",
    "task",
    "taste",
    "tattoo",
    "taxi",
    "teach",
    "team",
    "tell",
    "ten",
    "tenant",
    "tennis",
    "tent",
    "term",
    "test",
    "text",
    "thank",
    "that",
    "theme",
    "then",
    "theory",
    "there",
    "they",
    "thing",
    "this",
    "thought",
    "three",
    "thrive",
    "throw",
    "thumb",
    "thunder",
    "ticket",
    "tide",
    "tiger",
    "tilt",
    "timber",
    "time",
    "tiny",
    "tip",
    "tired",
    "tissue",
    "title",
    "toast",
    "tobacco",
    "today",
    "toddler",
    "toe",
    "together",
    "toilet",
    "token",
    "tomato",
    "tomorrow",
    "tone",
    "tongue",
    "tonight",
    "tool",
    "tooth",
    "top",
    "topic",
    "topple",
    "torch",
    "tornado",
    "tortoise",
    "toss",
    "total",
    "tourist",
    "toward",
    "tower",
    "town",
    "toy",
    "track",
    "trade",
    "traffic",
    "tragic",
    "train",
    "transfer",
    "trap",
    "trash",
    "travel",
    "tray",
    "treat",
    "tree",
    "trend",
    "trial",
    "tribe",
    "trick",
    "trigger",
    "trim",
    "trip",
    "trophy",
    "trouble",
    "truck",
    "true",
    "truly",
    "trumpet",
    "trust",
    "truth",
    "try",
    "tube",
    "tuition",
    "tumble",
    "tuna",
    "tunnel",
    "turkey",
    "turn",
    "turtle",
    "twelve",
    "twenty",
    "twice",
    "twin",
    "twist",
    "two",
    "type",
    "typical",
    "ugly",
    "umbrella",
    "unable",
    "unaware",
    "uncle",
    "uncover",
    "under",
    "undo",
    "unfair",
    "unfold",
    "unhappy",
    "uniform",
    "unique",
    "unit",
    "universe",
    "unknown",
    "unlock",
    "until",
    "unusual",
    "unveil",
    "update",
    "upgrade",
    "uphold",
    "upon",
    "upper",
    "upset",
    "urban",
    "urge",
    "usage",
    "use",
    "used",
    "useful",
    "useless",
    "usual",
    "utility",
    "vacant",
    "vacuum",
    "vague",
    "valid",
    "valley",
    "valve",
    "van",
    "vanish",
    "vapor",
    "various",
    "vast",
    "vault",
    "vehicle",
    "velvet",
    "vendor",
    "venture",
    "venue",
    "verb",
    "verify",
    "version",
    "very",
    "vessel",
    "veteran",
    "viable",
    "vibrant",
    "vicious",
    "victory",
    "video",
    "view",
    "village",
    "vintage",
    "violin",
    "virtual",
    "virus",
    "visa",
    "visit",
    "visual",
    "vital",
    "vivid",
    "vocal",
    "voice",
    "void",
    "volcano",
    "volume",
    "vote",
    "voyage",
    "wage",
    "wagon",
    "wait",
    "walk",
    "wall",
    "walnut",
    "want",
    "warfare",
    "warm",
    "warrior",
    "wash",
    "wasp",
    "waste",
    "water",
    "wave",
    "way",
    "wealth",
    "weapon",
    "wear",
    "weasel",
    "weather",
    "web",
    "wedding",
    "weekend",
    "weird",
    "welcome",
    "west",
    "wet",
    "whale",
    "what",
    "wheat",
    "wheel",
    "when",
    "where",
    "whip",
    "whisper",
    "wide",
    "width",
    "wife",
    "wild",
    "will",
    "win",
    "window",
    "wine",
    "wing",
    "wink",
    "winner",
    "winter",
    "wire",
    "wisdom",
    "wise",
    "wish",
    "witness",
    "wolf",
    "woman",
    "wonder",
    "wood",
    "wool",
    "word",
    "work",
    "world",
    "worry",
    "worth",
    "wrap",
    "wreck",
    "wrestle",
    "wrist",
    "write",
    "wrong",
    "yard",
    "year",
    "yellow",
    "you",
    "young",
    "youth",
    "zebra",
    "zero",
    "zone",
    "zoo",
]

