#   Copyright 2022-2022 Exactpro (Exactpro Systems Limited)
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

import datetime
import json
from pathlib import Path
from typing import Any, Dict, List, Optional, Union

from google.protobuf.json_format import ParseDict
from google.protobuf.timestamp_pb2 import Timestamp
from th2_grpc_common.common_pb2 import ConnectionID, Direction, EventID, ListValue, Message, MessageID, \
    MessageMetadata, Value

from th2_common_utils.event_components import TableComponent, TreeTableComponent


DictMessageType = Union[str, List, Dict]


def _message_to_dict_convert_value(value: Value) -> Optional[DictMessageType]:
    value_kind = value.WhichOneof('kind')

    if value_kind == 'simple_value':
        return value.simple_value  # type: ignore

    elif value_kind == 'list_value':
        return list(map(_message_to_dict_convert_value, value.list_value.values))

    elif value_kind == 'message_value':
        return {
            field: _message_to_dict_convert_value(field_value)
            for field, field_value in value.message_value.fields.items()
        }

    else:
        raise TypeError(f'Expected simple_value, list_value or message_value. {type(value)} object received: {value}')


def message_to_dict(message: Message) -> Dict[str, Optional[DictMessageType]]:
    """Converts th2-message to a dict.
    Fields of th2-message will be converted to a dict. You will lose all metadata.
    Args:
        message: th2 message.
    Returns:
        th2-message fields (message.fields) as a dict. All nested entities will be also converted.
        Conversion rules:
            Value.simple_value - str
            Value.list_value - list
            Value.message_value - dict
    Raises:
        TypeError: Occurs when 'message.fields' contains a field not of the 'Value' type.
    """
    message_metadata = message.metadata

    return {
        'parent_event_id': message.parent_event_id.id,
        'metadata': {
            'session_alias': message_metadata.id.connection_id.session_alias,
            'session_group': message_metadata.id.connection_id.session_group,
            'direction': Direction.Name(message_metadata.id.direction),
            'sequence': message_metadata.id.sequence,
            'subsequence': list(message_metadata.id.subsequence),
            'book_name': message_metadata.id.book_name,
            'timestamp': message_metadata.id.timestamp.ToDatetime() if message_metadata.id.HasField('timestamp')
            else None,
            'message_type': message_metadata.message_type,
            'properties': dict(**message_metadata.properties),
            'protocol': message_metadata.protocol
        },
        'fields': {
            field: _message_to_dict_convert_value(field_value)
            for field, field_value in message.fields.items()
        }
    }


def _dict_to_message_convert_value(entity: Any) -> Value:
    if isinstance(entity, (str, int, float)):
        return Value(simple_value=str(entity))
    elif isinstance(entity, list):
        return Value(list_value=ListValue(values=list(map(_dict_to_message_convert_value, entity))))
    elif isinstance(entity, dict):
        return Value(message_value=Message(fields={k: _dict_to_message_convert_value(v) for k, v in entity.items()}))

    elif isinstance(entity, Value):
        return entity
    elif isinstance(entity, ListValue):
        return Value(list_value=entity)
    elif isinstance(entity, Message):
        return Value(message_value=entity)

    else:
        raise TypeError(f'Cannot convert {type(entity)} object.')


def dict_to_message(fields: dict,
                    parent_event_id: Optional[EventID] = None,
                    message_type: str = '',
                    session_alias: str = '',
                    session_group: str = '',
                    direction: str = 'FIRST',
                    sequence: int = 0,
                    subsequence: Optional[List[int]] = None,
                    book_name: Optional[str] = '',
                    timestamp: Optional[datetime.datetime] = None,
                    properties: Optional[Dict[str, str]] = None,
                    protocol: str = '') -> Message:
    """Converts a dict to th2-message with 'metadata' and 'parent_event_id'.
    Args:
        fields: Message fields as a dict.
        parent_event_id: Parent event id.
        session_alias: Session alias.
        message_type: Message type.
        session_group: Session group.
        direction: Direction.
        sequence: Sequence.
        subsequence: Subsequence.
        book_name: Name of the book.
        timestamp: Timestamp as datetime.datetime object.
        properties: Properties.
        protocol: Protocol.
    Returns:
        th2 message with 'metadata' and 'parent_event_id'. All 'fields' nested entities will be converted.
        Conversion rules:
            str, int, float, Value - Value.simple_value
            list, ListValue - Value.list_value
            dict, Message - Value.message_value.
    Raises:
        TypeError: Occurs when 'message.fields' contains a field of the unsupported type.
    """

    if parent_event_id is None:
        parent_event_id = EventID()

    metadata = MessageMetadata(id=MessageID(connection_id=ConnectionID(session_alias=session_alias,
                                                                       session_group=session_group),
                                            direction=getattr(Direction, direction),
                                            sequence=sequence,
                                            subsequence=subsequence if subsequence is not None else [],
                                            book_name=book_name),
                               message_type=message_type,
                               properties=properties,
                               protocol=protocol)
    if timestamp is not None:
        timestamp_pb = Timestamp()
        timestamp_pb.FromDatetime(timestamp)
        metadata.id.timestamp.CopyFrom(timestamp_pb)

    return Message(parent_event_id=parent_event_id,
                   metadata=metadata,
                   fields={field: _dict_to_message_convert_value(field_value) for field, field_value in fields.items()})


def _message_to_table_convert_value(message_value: Union[str, List, Dict],
                                    columns_names: List[str],
                                    sort: bool) -> Optional[Union[str, TableComponent]]:
    if isinstance(message_value, str):
        return message_value  # type: ignore

    elif isinstance(message_value, list):
        table = TableComponent(columns_names=columns_names, sort=sort)

        for index, list_item in enumerate(message_value):
            table_inner_item = _message_to_table_convert_value(message_value=list_item,
                                                               columns_names=columns_names,
                                                               sort=sort)
            if isinstance(table_inner_item, TableComponent):
                table.add_table(index, table_inner_item)
            else:
                table.add_row(index, table_inner_item)

        return table

    elif isinstance(message_value, dict):
        table = TableComponent(columns_names=columns_names, sort=sort)

        for field_name, field_value in message_value.items():  # type: ignore
            table_inner_item = _message_to_table_convert_value(message_value=field_value,
                                                               columns_names=columns_names,
                                                               sort=sort)
            if isinstance(table_inner_item, TableComponent):
                table.add_table(field_name, table_inner_item)
            else:
                table.add_row(field_name, table_inner_item)

        return table

    else:
        raise TypeError(f'Expected object type of str, int, float, list or dict, got {type(message_value)}')


def message_to_table(message: Union[Dict, Message], sort: bool = False) -> TreeTableComponent:
    """Converts th2-message or dict to a TreeTable.
    Table can have only two columns. Nested tables are allowed. You will lose 'parent_event_id' and 'metadata'
    of the message.
    Args:
        message: th2-message.
        sort: If True, the rows will be sorted by the first field, otherwise, rows will be set in the order
            that you add it.
    Returns:
        Tree table with two columns - one contains the name of the field and the other contains the value of this field.
    Raises:
        TypeError: Occurs when 'message.fields' contains a field not of the 'Value' type.
    """

    if isinstance(message, Message):
        message = message_to_dict(message)['fields']  # type: ignore

    table = TreeTableComponent(columns_names=['Field Value'], sort=sort)

    for field_name in message:  # type: ignore
        table_entity = _message_to_table_convert_value(message_value=message[field_name],  # type: ignore
                                                       columns_names=table.columns_names,
                                                       sort=sort)
        if isinstance(table_entity, TableComponent):
            table.add_table(field_name, table_entity)
        else:
            table.add_row(field_name, table_entity)

    return table


def json_to_message(json_path: Union[str, Path]) -> Message:
    """Read json file and convert its content to th2-message.
    Args:
        json_path: Path to json file.
    Returns:
        th2-message
    """

    with open(json_path, 'r') as read_content:
        json_dict = json.load(read_content)

    return ParseDict(json_dict, Message())
