import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns
import math
import plotly.graph_objects as go
import json
# Import the simfin package and the Python shortcuts for the columns
import simfin as sf
from simfin.names import *


def prep_dataframe(file):
    pd.set_option('display.float_format', lambda x: '%.2f' % x)
    statement = pd.read_csv(file)
    statement.iloc[:, 0] = statement.iloc[:, 0].str.replace(' ', '_')
    statement.iloc[:, 0] = statement.iloc[:, 0].str.replace(',', '')
    statement.iloc[:, 0] = statement.iloc[:, 0].str.replace('&', '')
    statement.iloc[:, 0] = statement.iloc[:, 0].str.lower()
    statement = statement.set_index(statement.columns[0])
    statement=statement.abs()
    return statement

def prep_json(file):
    statement = prep_dataframe(file)
    statement_json = statement.to_json(orient="index")
    parsed = json.loads(statement_json)
    statement_json = json.dumps(parsed)
    return statement_json

#period = annual, quarterly
#country= us
def prepare_finData(country):
    sf.set_api_key('free')
    sf.set_data_dir('~/simfin_data/')
    df_companies = sf.load_companies(market=country)
    df_industries = sf.load_industries()
    
def get_annual_finData_income(country):
    prepare_finData(country)
    df_income = sf.load_income(variant='annual', market=country)
    df_income.columns = df_income.columns.str.replace(' ', '_')
    return df_income

def get_annual_finData_balance(country):
    prepare_finData(country)
    df_balance = sf.load_balance(variant='annual', market=country)
    df_balance.columns = df_balance.columns.str.replace(' ', '_')
    return df_balance

def get_annual_finData_cashflow(country):
    prepare_finData(country)
    df_cashflow = sf.load_cashflow(variant='annual', market=country)
    df_cashflow.columns = df_cashflow.columns.str.replace(' ', '_')
    return df_cashflow

def get_annual_finData_by_symbol(category,symbol,country):
    try:
        if category == "income":
            df = get_annual_finData_income(country)
        if category == "balancesheet":
            df = get_annual_finData_balance(country)
        if category == "cashflow":
            df = get_annual_finData_cashflow(country)
        df = df.loc[symbol].transpose()
        df = df.rename(index={'Fiscal_Year': 'Breakdown'})
        df.columns = df.loc['Breakdown']
        df = df.drop(['SimFinId', 'Currency', 'Breakdown', 'Fiscal_Period', 'Publish_Date', 'Restated_Date']) 
        return df
    except UnboundLocalError:
        print('Not a valid category')
    except KeyError:
        print('Not a valid symbol')

def printStatement(file):
    statement = prep_dataframe(file)
    print(statement)

def lineplot(dataframe,category):
    plotData = dataframe.T
    plotData = plotData.reset_index()
    #plotData.columns[0]= pd.to_datetime(plotData.columns[0]) 
    plotData = plotData.sort_values(plotData.columns[0])
    plotData = plotData[[plotData.columns[0],category]]
    plotData[category] = plotData[category].astype(float)

    plt.figure(figsize=(20,9))
    sns.lineplot(data=plotData, x=plotData.columns[0],y=category)

# def multiLineplot(dataframe,title):
#     statement = dataframe

#     plotData = statement.T
#     plotData = plotData.reset_index()
#     #plotData.iloc[:, 1]= pd.to_datetime(plotData.iloc[:, 1])
#     plotData = plotData.sort_values(plotData.columns[0])

#     columnsList = list(plotData.columns.values)
#     graphRowCount=math.ceil(len(plotData.columns)/3)
#     fig = plt.figure(figsize=(30,45))
#     fig.suptitle(title, fontsize=30)

#     for x in range(1, len(columnsList)):
#         plotNumber = 'ax'+str(x)
#         colname = plotData.columns[x]

#         plotNumber = fig.add_subplot(graphRowCount,3,x)
#         plotNumber.set_title(colname)
#         plotNumber.plot(plotData.iloc[:, 1],
#              plotData[colname])
#         plt.xticks(rotation=45)

#     plt.show()

def calculateMetrics(balanceSheet,incomeStatement):

    frames = [balanceSheet, incomeStatement]
    Ratio = pd.DataFrame()
    dataframeForRatio = pd.concat(frames)
    dataframeForRatio = dataframeForRatio.drop_duplicates()
    dataframeForRatio = dataframeForRatio.T
    dataframeForRatio['average_inventory'] = dataframeForRatio['inventory'].mean()
    dataframeForRatio['average_accounts_receivable'] = dataframeForRatio['net_receivables'].mean()

    #Liquidity Ratios
    Ratio['quick_ratio'] = (dataframeForRatio['cash_and_cash_equivalents']+dataframeForRatio['net_receivables']+dataframeForRatio['short_term_investments'])/dataframeForRatio['total_current_liabilities']
    Ratio['acid-test_ratio'] = dataframeForRatio['total_current_assets']/dataframeForRatio['total_current_liabilities']
    Ratio['cash_ratio'] = (dataframeForRatio['total_current_assets']-dataframeForRatio['inventory'])/dataframeForRatio['total_current_liabilities']

    #Leverage Financial Ratios
    Ratio['debt_ratio'] = dataframeForRatio['total_liabilities']/(dataframeForRatio['total_assets']-dataframeForRatio['total_liabilities'])
    Ratio['interest_coverage_ratio'] = dataframeForRatio['gross_profit']/dataframeForRatio['interest_expense']
    #Ratio['debt_service_coverage_ratio'] = dataframeForRatio['gross_profit']/dataframeForRatio['']

    #Efficiency Ratios
    Ratio['asset_turnover_ratio'] = dataframeForRatio['total_revenue']/dataframeForRatio['total_assets']
    Ratio['inventory_turnover_ratio'] = dataframeForRatio['cost_of_revenue']/dataframeForRatio['average_inventory']
    Ratio['receivables_turnover_ratio'] = dataframeForRatio['total_revenue']/dataframeForRatio['average_accounts_receivable']
    Ratio['days_sales_in_inventory_ratio'] = 365/Ratio['inventory_turnover_ratio']

    #Profitability Ratios
    Ratio['gross_margin_ratio'] = dataframeForRatio['gross_profit']/dataframeForRatio['total_revenue']
    Ratio['operating_margin_ratio'] = dataframeForRatio['net_income']/dataframeForRatio['total_revenue']
    Ratio['return_on_assets_ratio'] = dataframeForRatio['net_income']/dataframeForRatio['total_revenue']
    Ratio['return_on_equity_ratio'] = dataframeForRatio['net_income']/(dataframeForRatio['total_revenue']-dataframeForRatio['total_liabilities'])

    Ratio = Ratio.T
    Ratio = Ratio.round(4)
    print(Ratio)
    
def bulletChart(dataframe,item):
    statement = dataframe
    
    data = statement.T
    data = data.reset_index()
    data = data.sort_values(by='index', ascending=False)
    avg_item = 'avg_'+item
    data[avg_item] = data[item].mean()
    data = data.round(2)
    data = data.iloc[0]
    
    fig = go.Figure(go.Indicator(
        mode = "number+gauge+delta",
        gauge = {'shape': "bullet"},
        value = data[item],
        delta = {'reference': data[avg_item]},
        domain = {'x': [0, 1], 'y': [0, 1]},
        title = {'text': item}))
    fig.update_layout(height = 250)
    
    fig.show()

def horizontalAnalysisLastTwo(dataframe):
    statement = dataframe
    statement = statement.reindex(sorted(statement.columns), axis=1)
    statement_lastPeriods = statement[statement.columns[-2:]]
    statement_lastPeriods_lastColName = statement[statement_lastPeriods.columns[-1:]].columns.values[0]
    statement_lastPeriods_secondLastColName = statement[statement_lastPeriods.columns[-2:-1]].columns.values[0]
    statement_lastPeriods['Amount(Increased/Decreased)'] = statement_lastPeriods[statement_lastPeriods_lastColName]-statement_lastPeriods[statement_lastPeriods_secondLastColName]
    statement_lastPeriods['Percentage(Increased/Decreased)'] = (statement_lastPeriods[statement_lastPeriods_lastColName]/statement_lastPeriods[statement_lastPeriods_secondLastColName])-1
    statement_lastPeriods = statement_lastPeriods.dropna()
    statement_lastPeriods['Percentage(Increased/Decreased)'] = pd.Series(["{0:.2f}%".format(val * 100) for val in statement_lastPeriods['Percentage(Increased/Decreased)']], index = statement_lastPeriods.index)
    statement_lastPeriods.to_csv('horizontalAnalysisLastTwo.csv')
    print("Horizontal Analysis with Last two Periods", statement_lastPeriods, sep='\n')
