"""
kernel-sidecar emits all of its messages using vanilla logging. The setup_logging() function
below will structure kernel-sidecar (and any other loggers) to be streamed out using structlogs
ConsoleRenderer. While this is useful to the kernel-sidecar repo in general, for instance in the
CLI script and in tests, this function primarily serves as a template for how to set up structlog
in your own application and configure it to structure both structlog-emitted messages (your app
logs probably) and vanilla logging (kernel-sidecar logs).
"""
import logging


# Set up structlog "pretty" console rendering. See tests/conftest.py for source template
def setup_logging(log_level: int = logging.INFO):
    # structlog is optional dependency, try/except here and just use plain logging if structlog
    # isn't installed. That will not render the "extra" log info like ZMQ content on send/recv
    # debug logs.
    try:
        import structlog

        structlog.configure(
            processors=[
                structlog.stdlib.PositionalArgumentsFormatter(),
                structlog.processors.StackInfoRenderer(),
                structlog.processors.format_exc_info,
                structlog.stdlib.ProcessorFormatter.wrap_for_formatter,
            ],
            logger_factory=structlog.stdlib.LoggerFactory(),
            wrapper_class=structlog.stdlib.BoundLogger,
            cache_logger_on_first_use=True,
        )

        # shared processors to be applied to both vanilla and structlog messages
        # after each is appropriately pre-processed
        processors = [
            # log level / logger name, effects coloring in ConsoleRenderer(colors=True)
            structlog.stdlib.add_log_level,
            structlog.stdlib.add_logger_name,
            # timestamp format
            structlog.processors.TimeStamper(fmt="iso"),
            # To see all CallsiteParameterAdder options:
            # https://www.structlog.org/en/stable/api.html#structlog.processors.CallsiteParameterAdder
            # more options include module, pathname, process, process_name, thread, thread_name
            structlog.processors.CallsiteParameterAdder(
                {
                    structlog.processors.CallsiteParameter.FILENAME,
                    structlog.processors.CallsiteParameter.FUNC_NAME,
                    structlog.processors.CallsiteParameter.LINENO,
                }
            ),
            # Any structlog.contextvars.bind_contextvars included in middleware/functions
            structlog.contextvars.merge_contextvars,
            # strip _record and _from_structlog keys from event dictionary
            structlog.stdlib.ProcessorFormatter.remove_processors_meta,
            structlog.dev.ConsoleRenderer(colors=True),
            # ^^ In prod with any kind of logging service (datadog, grafana, etc), ConsoleRenderer
            # would probably be replaced with structlog.processors.JSONRenderer() or similar
        ]

        # Configs applied to logs generated by structlog or vanilla logging
        logging.config.dictConfig(
            {
                "version": 1,
                "disable_existing_loggers": False,
                "formatters": {
                    "default": {
                        "()": structlog.stdlib.ProcessorFormatter,
                        "processors": processors,
                        "foreign_pre_chain": [structlog.stdlib.ExtraAdder()],
                    },
                },
                "handlers": {
                    "default": {
                        "class": "logging.StreamHandler",
                        "formatter": "default",
                        "stream": "ext://sys.stdout",
                    },
                },
                "loggers": {
                    # "" for applying handler to "root" (all libraries)
                    # you could set this to "kernel_sidecar" to only see logs from this library
                    "": {
                        "handlers": ["default"],
                        "level": log_level,
                        "propagate": True,
                    },
                },
            }
        )
    except ImportError:
        logger = logging.getLogger()
        logger.warning("Structlog not installed, using vanilla logging")
        logger.setLevel(log_level)
        logger.addHandler(logging.StreamHandler())
