"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheQueryStringBehavior = exports.CacheHeaderBehavior = exports.CacheCookieBehavior = exports.CachePolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Cache Policy configuration.
 *
 * @stability stable
 * @resource AWS::CloudFront::CachePolicy
 * @link https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html
 */
class CachePolicy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.cachePolicyName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_CachePolicyProps(props);
        const cachePolicyName = (_e = props.cachePolicyName) !== null && _e !== void 0 ? _e : `${core_1.Names.uniqueId(this)}-${core_1.Stack.of(this).region}`;
        if (!core_1.Token.isUnresolved(cachePolicyName) && !cachePolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'cachePolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.cachePolicyName}'`);
        }
        const minTtl = ((_f = props.minTtl) !== null && _f !== void 0 ? _f : core_1.Duration.seconds(0)).toSeconds();
        const defaultTtl = Math.max(((_g = props.defaultTtl) !== null && _g !== void 0 ? _g : core_1.Duration.days(1)).toSeconds(), minTtl);
        const maxTtl = Math.max(((_h = props.maxTtl) !== null && _h !== void 0 ? _h : core_1.Duration.days(365)).toSeconds(), defaultTtl);
        const resource = new cloudfront_generated_1.CfnCachePolicy(this, 'Resource', {
            cachePolicyConfig: {
                name: cachePolicyName,
                comment: props.comment,
                minTtl,
                maxTtl,
                defaultTtl,
                parametersInCacheKeyAndForwardedToOrigin: this.renderCacheKey(props),
            },
        });
        this.cachePolicyId = resource.ref;
    }
    /**
     * Imports a Cache Policy from its id.
     *
     * @stability stable
     */
    static fromCachePolicyId(scope, id, cachePolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.cachePolicyId = cachePolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed cache policy. */
    static fromManagedCachePolicy(managedCachePolicyId) {
        return new class {
            constructor() {
                this.cachePolicyId = managedCachePolicyId;
            }
        }();
    }
    renderCacheKey(props) {
        var _e, _f, _g, _h, _j;
        const cookies = (_e = props.cookieBehavior) !== null && _e !== void 0 ? _e : CacheCookieBehavior.none();
        const headers = (_f = props.headerBehavior) !== null && _f !== void 0 ? _f : CacheHeaderBehavior.none();
        const queryStrings = (_g = props.queryStringBehavior) !== null && _g !== void 0 ? _g : CacheQueryStringBehavior.none();
        return {
            cookiesConfig: {
                cookieBehavior: cookies.behavior,
                cookies: cookies.cookies,
            },
            headersConfig: {
                headerBehavior: headers.behavior,
                headers: headers.headers,
            },
            enableAcceptEncodingGzip: (_h = props.enableAcceptEncodingGzip) !== null && _h !== void 0 ? _h : false,
            enableAcceptEncodingBrotli: (_j = props.enableAcceptEncodingBrotli) !== null && _j !== void 0 ? _j : false,
            queryStringsConfig: {
                queryStringBehavior: queryStrings.behavior,
                queryStrings: queryStrings.queryStrings,
            },
        };
    }
}
exports.CachePolicy = CachePolicy;
_a = JSII_RTTI_SYMBOL_1;
CachePolicy[_a] = { fqn: "@aws-cdk/aws-cloudfront.CachePolicy", version: "1.137.0" };
/**
 * This policy is designed for use with an origin that is an AWS Amplify web app.
 *
 * @stability stable
 */
CachePolicy.AMPLIFY = CachePolicy.fromManagedCachePolicy('2e54312d-136d-493c-8eb9-b001f22f67d2');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 *
 * @stability stable
 */
CachePolicy.CACHING_OPTIMIZED = CachePolicy.fromManagedCachePolicy('658327ea-f89d-4fab-a63d-7e88639e58f6');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 * Disables cache compression.
 *
 * @stability stable
 */
CachePolicy.CACHING_OPTIMIZED_FOR_UNCOMPRESSED_OBJECTS = CachePolicy.fromManagedCachePolicy('b2884449-e4de-46a7-ac36-70bc7f1ddd6d');
/**
 * Disables caching.
 *
 * This policy is useful for dynamic content and for requests that are not cacheable.
 *
 * @stability stable
 */
CachePolicy.CACHING_DISABLED = CachePolicy.fromManagedCachePolicy('4135ea2d-6df8-44a3-9df3-4b5a84be39ad');
/**
 * Designed for use with an origin that is an AWS Elemental MediaPackage endpoint.
 *
 * @stability stable
 */
CachePolicy.ELEMENTAL_MEDIA_PACKAGE = CachePolicy.fromManagedCachePolicy('08627262-05a9-4f76-9ded-b50ca2e3a84f');
/**
 * Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheCookieBehavior('none'); }
    /**
     * All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new CacheCookieBehavior('all'); }
    /**
     * Only the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new CacheCookieBehavior('whitelist', cookies);
    }
    /**
     * All cookies except the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to deny must be provided');
        }
        return new CacheCookieBehavior('allExcept', cookies);
    }
}
exports.CacheCookieBehavior = CacheCookieBehavior;
_b = JSII_RTTI_SYMBOL_1;
CacheCookieBehavior[_b] = { fqn: "@aws-cdk/aws-cloudfront.CacheCookieBehavior", version: "1.137.0" };
/**
 * Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheHeaderBehavior('none'); }
    /**
     * Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new CacheHeaderBehavior('whitelist', headers);
    }
}
exports.CacheHeaderBehavior = CacheHeaderBehavior;
_c = JSII_RTTI_SYMBOL_1;
CacheHeaderBehavior[_c] = { fqn: "@aws-cdk/aws-cloudfront.CacheHeaderBehavior", version: "1.137.0" };
/**
 * Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheQueryStringBehavior('none'); }
    /**
     * All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new CacheQueryStringBehavior('all'); }
    /**
     * Only the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new CacheQueryStringBehavior('whitelist', queryStrings);
    }
    /**
     * All query strings except the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to deny must be provided');
        }
        return new CacheQueryStringBehavior('allExcept', queryStrings);
    }
}
exports.CacheQueryStringBehavior = CacheQueryStringBehavior;
_d = JSII_RTTI_SYMBOL_1;
CacheQueryStringBehavior[_d] = { fqn: "@aws-cdk/aws-cloudfront.CacheQueryStringBehavior", version: "1.137.0" };
//# sourceMappingURL=data:application/json;base64,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