# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['jetblack_fixengine',
 'jetblack_fixengine.acceptor',
 'jetblack_fixengine.admin',
 'jetblack_fixengine.initiator',
 'jetblack_fixengine.managers',
 'jetblack_fixengine.persistence',
 'jetblack_fixengine.transports',
 'jetblack_fixengine.utils']

package_data = \
{'': ['*']}

install_requires = \
['aiofiles>=23.1,<24.0',
 'aiosqlite>=0.18.0,<0.19.0',
 'jetblack-fixparser>=2.3,<3.0',
 'pytz>=2022.7,<2023.0',
 'tzlocal>=4.3,<5.0']

setup_kwargs = {
    'name': 'jetblack-fixengine',
    'version': '1.0.0a2',
    'description': 'A pure Python implementation of the FIX protocol',
    'long_description': '# jetblack-fixengine\n\nA pure python asyncio FIX engine implemented as a state machine.\n\n## Status\n\nThis is work in progress.\n\n## Overview\n\nThis project provides a pure Python, asyncio implementation of\na FIX engine, supporting both initiators and acceptors.\n\nThe engine uses the [jetblack-fixparser](https://github.com/rob-blackbourn/jetblack-fixparser)\npackage to present the FIX messages a plain Python objects. For example, a `LOGON` message\ncan be sent as follows:\n\n```python\nawait send_message({\n    \'MsgType\': \'LOGON\',\n    \'MsgSeqNum\': 42,\n    \'SenderCompID\': \'ME\',\n    \'TargetCompID\': \'BANK OF SOMEWHERE\',\n    \'SendingTime\': datetime.now(timezone.utc),\n    \'EncryptMethod\': "NONE",\n    \'HeartBtInt\': 30\n})\n```\n\n### FIX Protocols\n\nThe FIX protocol is a combination of *well known* messages (like `LOGON`)\nand *custom* messages (like an order to buy or sell trades). The protocol\nhas evolved through a number of different versions providing new features.\n\nBecause of this the protocols are provided by config files. Historically\n`XML` was used. While this is supported, `yaml` is preferred and some\nexample protocols are provided in the [etc](etc) folder.\n\nCurrently supported versions are 4.0, 4.1, 4.2, 4.3, 4.4.\n\n### Initiators\n\nAn initiator is a class which inherits from `Initiator`, and implements a\nfew methods, and has access to `send_message`. Here is an example.\n\n```python\nimport asyncio\nimport logging\nimport os.path\nfrom typing import Mapping, Any\n\nfrom jetblack_fixparser import load_yaml_protocol\nfrom jetblack_fixengine import FileStore\nfrom jetblack_fixengine import start_initiator, Initiator\n\nLOGGER = logging.getLogger(__name__)\n\n\nclass MyInitiator(Initiator):\n    """An instance of the initiator"""\n\n    async def on_logon(self, _message: Mapping[str, Any]) -> None:\n        LOGGER.info(\'on_logon\')\n\n    async def on_logout(self, _message: Mapping[str, Any]) -> None:\n        LOGGER.info(\'on_logout\')\n\n    async def on_application_message(self, _message: Mapping[str, Any]) -> None:\n        LOGGER.info(\'on_application_message\')\n\n\nroot = os.path.abspath(os.path.join(os.path.dirname(__file__), \'..\'))\netc = os.path.join(root, \'etc\')\n\nSTORE = FileStore(os.path.join(root, \'store\'))\nHOST = \'127.0.0.1\'\nPORT = 9801\nSENDER_COMP_ID = \'INITIATOR1\'\nTARGET_COMP_ID = \'ACCEPTOR\'\nLOGON_TIMEOUT = 60\nHEARTBEAT_TIMEOUT = 30\nPROTOCOL = load_yaml_protocol(\'etc/FIX44.yaml\')\n\nlogging.basicConfig(level=logging.DEBUG)\n\nasyncio.run(\n    start_initiator(\n        MyInitiator,\n        HOST,\n        PORT,\n        PROTOCOL,\n        SENDER_COMP_ID,\n        TARGET_COMP_ID,\n        STORE,\n        LOGON_TIMEOUT,\n        HEARTBEAT_TIMEOUT,\n        shutdown_timeout=10\n    )\n)\n```\n\n### Acceptor\n\nThe acceptor works in the same way as the initiator. Here is an example:\n\n```python\nimport asyncio\nimport logging\nimport os.path\nfrom typing import Mapping, Any\n\nfrom jetblack_fixparser import load_yaml_protocol\nfrom jetblack_fixengine import FileStore\nfrom jetblack_fixengine import start_acceptor, Acceptor\n\n\nLOGGER = logging.getLogger(__name__)\n\n\nclass MyAcceptor(Acceptor):\n    """An instance of the acceptor"""\n\n    async def on_logon(self, _message: Mapping[str, Any]):\n        LOGGER.info(\'on_logon\')\n\n    async def on_logout(self, _message: Mapping[str, Any]) -> None:\n        LOGGER.info(\'on_logout\')\n\n    async def on_application_message(self, _message: Mapping[str, Any]) -> None:\n        LOGGER.info(\'on_application_message\')\n\n\nroot = os.path.abspath(os.path.join(os.path.dirname(__file__), \'..\'))\netc = os.path.join(root, \'etc\')\n\nSTORE = FileStore(os.path.join(root, \'store\'))\nHOST = \'0.0.0.0\'\nPORT = 9801\nSENDER_COMP_ID = \'ACCEPTOR\'\nTARGET_COMP_ID = \'INITIATOR1\'\nLOGON_TIMEOUT = 60\nHEARTBEAT_TIMEOUT = 30\nPROTOCOL = load_yaml_protocol(\'etc/FIX44.yaml\')\n\nlogging.basicConfig(level=logging.DEBUG)\n\nasyncio.run(\n    start_acceptor(\n        MyAcceptor,\n        HOST,\n        PORT,\n        PROTOCOL,\n        SENDER_COMP_ID,\n        TARGET_COMP_ID,\n        STORE,\n        HEARTBEAT_TIMEOUT,\n        client_shutdown_timeout=10\n    )\n)\n```\n\nNote that throwing the exception `LogonError` from `on_logon` will reject\nthe logon request.\n\n### Stores\n\nThe engines need to store their state. Two stores are currently provided:\na file store (`FileStore`) and sqlite (`SqlStore`).\n\n## Implementation\n\nThe engines are implemented as state machines. This means they can be\ntested without the need for IO.\n',
    'author': 'Rob Blackbourn',
    'author_email': 'rob.blackbourn@googlemail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/rob-blackbourn/jetblack-fixengine',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
