"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UrlShortener = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const redirect_function_1 = require("./redirect-function");
const shortener_function_1 = require("./shortener-function");
/**
 * URL shortener
 */
class UrlShortener extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        // Table to save a counter
        const table = new dynamodb.Table(this, 'Table', {
            partitionKey: {
                name: 'key',
                type: dynamodb.AttributeType.STRING,
            },
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // Bucket to save redirects
        const bucket = new s3.Bucket(this, 'Bucket', {
            lifecycleRules: [{
                    expiration: (_b = props.expiration) !== null && _b !== void 0 ? _b : aws_cdk_lib_1.Duration.days(365),
                }],
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        // CloudFront distribution
        const domainName = props.recordName ? `${props.recordName}.${props.hostedZone.zoneName}` : props.hostedZone.zoneName;
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        const distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(bucket),
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        functionVersion: new redirect_function_1.RedirectFunction(this, 'Redirect'),
                    },
                ],
            },
            certificate,
            domainNames: [domainName],
        });
        // Route53 records
        new route53.ARecord(this, 'ARecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        // Lambda function to increment counter and write redirect in bucket
        const handler = new shortener_function_1.ShortenerFunction(this, 'handler', {
            logRetention: logs.RetentionDays.ONE_MONTH,
            environment: {
                DOMAIN_NAME: domainName,
                BUCKET_NAME: bucket.bucketName,
                TABLE_NAME: table.tableName,
            },
        });
        if (props.corsAllowOrigins) {
            handler.addEnvironment('CORS_ALLOW_ORIGINS', props.corsAllowOrigins.join(' '));
        }
        bucket.grantPut(handler);
        bucket.grantPutAcl(handler);
        table.grant(handler, 'dynamodb:UpdateItem');
        // API
        this.api = new apigateway.RestApi(this, `UrlShortener${props.hostedZone.zoneName}`, {
            endpointTypes: props.apiGatewayEndpoint ? [apigateway.EndpointType.PRIVATE] : undefined,
            policy: props.apiGatewayEndpoint
                ? new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: ['execute-api:Invoke'],
                            principals: [new iam.AnyPrincipal()],
                            resources: [aws_cdk_lib_1.Fn.join('', ['execute-api:/', '*'])],
                            conditions: {
                                StringEquals: { 'aws:SourceVpce': props.apiGatewayEndpoint.vpcEndpointId },
                            },
                        }),
                    ],
                })
                : undefined,
            defaultCorsPreflightOptions: props.corsAllowOrigins
                ? { allowOrigins: props.corsAllowOrigins }
                : undefined,
        });
        this.api.root.addMethod('ANY', new apigateway.LambdaIntegration(handler), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.api.root
            .addResource('{proxy+}')
            .addMethod('ANY', new apigateway.LambdaIntegration(handler), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.apiEndpoint = this.api.url;
    }
}
exports.UrlShortener = UrlShortener;
_a = JSII_RTTI_SYMBOL_1;
UrlShortener[_a] = { fqn: "cloudstructs.UrlShortener", version: "0.4.30" };
//# sourceMappingURL=data:application/json;base64,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