"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jsonPathString = exports.recurseObject = exports.findReferencedPaths = exports.renderObject = exports.JsonPathToken = void 0;
const core_1 = require("@aws-cdk/core");
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            const path = jsonPathString(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleList(_key, x) {
            const path = jsonPathStringList(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleNumber(_key, x) {
            const path = jsonPathNumber(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
function recurseObject(obj, handlers, visited = []) {
    // If the argument received is not actually an object (string, number, boolean, undefined, ...) or null
    // just return it as is as there's nothing to be rendered. This should only happen in the original call to
    // recurseObject as any recursive calls to it are checking for typeof value === 'object' && value !== null
    if (typeof obj !== 'object' || obj === null) {
        return obj;
    }
    // Avoiding infinite recursion
    if (visited.includes(obj)) {
        return {};
    }
    // Marking current object as visited for the current recursion path
    visited.push(obj);
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers, visited));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            ret[key] = recurseObject(value, handlers, visited);
        }
    }
    // Removing from visited after leaving the current recursion path
    // Allowing it to be visited again if it's not causing a recursion (circular reference)
    visited.pop();
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers, visited = []) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers, visited);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
//# sourceMappingURL=data:application/json;base64,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