"""
Code snippet generator for Drun test cases.

Generates executable Shell and Python scripts from test steps.
"""

from __future__ import annotations

import json
import re
from datetime import datetime
from typing import Dict, List, Optional, Any, Set
from pathlib import Path

from drun.models.case import Case


class SnippetGenerator:
    """代码片段生成器 - 为测试步骤生成可执行脚本"""
    
    def __init__(self):
        self._version = self._get_version()
    
    def _get_version(self) -> str:
        """获取 Drun 版本号"""
        try:
            from drun import __version__
            return __version__
        except Exception:
            return "4.2.0"
    
    def generate_shell_script_for_step(
        self,
        case: Case,
        step_idx: int,
        total_steps: int,
        envmap: Optional[Dict[str, Any]] = None
    ) -> str:
        """为单个 Step 生成可执行的 Shell 脚本"""
        from drun.exporters.curl import step_to_curl
        
        step = case.steps[step_idx]
        
        lines = [
            "#!/bin/bash",
            "#",
            f"# Generated by Drun v{self._version}",
            f"# Case: {case.config.name or 'Unknown'}",
            f"# Step: {step_idx + 1}/{total_steps} - {step.name}",
            f"# Generated at: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}",
            "#",
            "",
            "set -e  # Exit on error",
            "",
        ]
        
        # 检查变量依赖
        dependencies = self._detect_variable_dependencies(case, step_idx)
        if dependencies:
            lines.append("# Note: This step depends on variables from previous steps:")
            for var in dependencies:
                lines.append(f'#   export {var}="your-value-here"')
            lines.append("")
        
        lines.append(f'echo "=== Step {step_idx + 1}: {step.name} ==="')
        lines.append("")
        
        # 生成 curl 命令
        curl_cmd = step_to_curl(case, step_idx, multiline=True, shell="sh", envmap=envmap)
        lines.append(curl_cmd)
        
        lines.append("")
        lines.append('echo ""')
        lines.append('echo "✅ Request completed"')
        
        return "\n".join(lines)
    
    def generate_python_script_for_step(
        self,
        case: Case,
        step_idx: int,
        total_steps: int,
        envmap: Optional[Dict[str, Any]] = None
    ) -> str:
        """为单个 Step 生成可执行的 Python 脚本（Postman 格式）"""
        step = case.steps[step_idx]
        req = step.request
        
        lines = [
            "import requests",
            "import json",
            "",
        ]
        
        # URL
        url = self._resolve_url(case, req.path, envmap)
        lines.append(f'url = "{url}"')
        lines.append("")
        
        # Payload (body)
        if req.body:
            body_str = json.dumps(req.body, indent=2, ensure_ascii=False)
            lines.append(f"payload = json.dumps({body_str})")
        elif req.data:
            data_str = json.dumps(req.data, indent=2, ensure_ascii=False)
            lines.append(f"payload = {data_str}")
        else:
            lines.append('payload = ""')
        
        # Headers
        headers = req.headers or {}
        if headers:
            headers_str = json.dumps(headers, indent=2, ensure_ascii=False)
            lines.append(f"headers = {headers_str}")
        else:
            lines.append("headers = {}")
        
        lines.append("")
        
        # Request
        method = (req.method or 'GET').upper()
        if req.body:
            lines.append(f'response = requests.request("{method}", url, headers=headers, data=payload)')
        elif req.data:
            lines.append(f'response = requests.request("{method}", url, headers=headers, data=payload)')
        elif req.params:
            params_str = json.dumps(req.params, ensure_ascii=False)
            lines.append(f'response = requests.request("{method}", url, headers=headers, params={params_str})')
        else:
            lines.append(f'response = requests.request("{method}", url, headers=headers)')
        
        lines.append("")
        lines.append("print(response.text)")
        
        return "\n".join(lines)
    
    def _generate_python_request(
        self,
        case: Case,
        step_idx: int,
        envmap: Optional[Dict[str, Any]] = None
    ) -> str:
        """生成 Python 请求代码"""
        step = case.steps[step_idx]
        req = step.request
        
        # 解析 URL
        url = self._resolve_url(case, req.path, envmap)
        method = (req.method or 'GET').lower()
        
        lines = [f"response = requests.{method}("]
        lines.append(f"    url='{url}',")
        
        # Headers
        if req.headers:
            headers_str = json.dumps(req.headers, ensure_ascii=False)
            lines.append(f"    headers={headers_str},")
        
        # Params
        if req.params:
            params_str = json.dumps(req.params, ensure_ascii=False)
            lines.append(f"    params={params_str},")
        
        # Body (JSON)
        if req.body:
            body_str = json.dumps(req.body, ensure_ascii=False, indent=4)
            # 缩进处理
            body_lines = body_str.split('\n')
            if len(body_lines) > 1:
                lines.append(f"    json={body_lines[0]}")
                for line in body_lines[1:]:
                    lines.append(f"    {line},")
            else:
                lines.append(f"    json={body_str},")
        
        # Data (form)
        elif req.data:
            data_str = json.dumps(req.data, ensure_ascii=False)
            lines.append(f"    data={data_str},")
        
        # Auth
        if req.auth:
            auth_type = req.auth.get('type', '').lower()
            if auth_type == 'bearer':
                token = req.auth.get('token', '')
                # 已在 headers 中处理
                pass
            elif auth_type == 'basic':
                username = req.auth.get('username', '')
                password = req.auth.get('password', '')
                lines.append(f"    auth=('{username}', '{password}'),")
        
        # Timeout
        if req.timeout:
            lines.append(f"    timeout={req.timeout},")
        else:
            lines.append("    timeout=30.0,")
        
        lines.append(")")
        
        return "\n".join(lines)
    
    def _resolve_url(
        self,
        case: Case,
        path: str,
        envmap: Optional[Dict[str, Any]] = None
    ) -> str:
        """解析完整 URL（base_url + path + 变量替换）"""
        from drun.templating.engine import TemplateEngine
        
        templater = TemplateEngine()
        
        # 渲染 path
        resolved_path = templater.render_value(
            path,
            case.config.variables or {},
            envmap=envmap
        )
        
        if str(resolved_path).startswith('http://') or str(resolved_path).startswith('https://'):
            return str(resolved_path)
        
        # 拼接 base_url
        base = case.config.base_url or ''
        if base:
            base = templater.render_value(base, case.config.variables or {}, envmap=envmap)
            base_str = str(base).rstrip('/')
            path_str = str(resolved_path).lstrip('/')
            return f"{base_str}/{path_str}"
        
        return str(resolved_path)
    
    def _detect_variable_dependencies(
        self,
        case: Case,
        step_idx: int
    ) -> List[str]:
        """检测 Step 依赖的变量（来自前面 step 的 extract）"""
        dependencies: Set[str] = set()
        
        # 收集前面所有 step 提取的变量
        extracted_vars: Set[str] = set()
        for i in range(step_idx):
            for var_name in case.steps[i].extract.keys():
                extracted_vars.add(var_name)
                extracted_vars.add(var_name.upper())
        
        # 检查当前 step 是否使用了这些变量
        step = case.steps[step_idx]
        req = step.request
        
        # 将请求对象转为字符串进行检查
        req_str = str(req.model_dump())
        
        # 查找 ${ENV(VAR)} 模式
        env_vars = re.findall(r'\$\{ENV\((\w+)\)\}', req_str)
        for var in env_vars:
            var_upper = var.upper()
            if var in extracted_vars or var_upper in extracted_vars:
                dependencies.add(var_upper)
        
        # 查找 ${VAR} 模式
        expr_vars = re.findall(r'\$\{(\w+)\}', req_str)
        for var in expr_vars:
            if var != 'ENV':  # 排除 ENV 函数
                var_upper = var.upper()
                if var in extracted_vars or var_upper in extracted_vars:
                    dependencies.add(var_upper)
        
        # 查找 $VAR 模式
        simple_vars = re.findall(r'\$(\w+)', req_str)
        for var in simple_vars:
            var_upper = var.upper()
            if var in extracted_vars or var_upper in extracted_vars:
                dependencies.add(var_upper)
        
        return sorted(dependencies)
