"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown && props.cooldown.toSeconds(),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            }
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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