# `deepfeatx`: Deep Learning Feature Extractor of Images using Transfer Learning Models
> Helper for automatic extraction of features from images (and soon text as well) from transfer learning models like ResNet, VGG16 and EfficientNet.


## Install

```python
#hide_output
!pip install deepfeatx
```

## Why this project has been created
- Fill the gap between ML and DL thus allowing estimators beyond only neural networks for computer vision and NLP problems
- Neural network models are too painful to setup and train - data generators, optimizers, learning rates, loss functions, training loops, batch size, etc. 
- State of the art results are possible thanks to pretrained models that allows feature extraction
- With this library we can handle those problems as they were traditional machine learning problems
- Possibility of using low-code APIs like `scikit-learn` for computer vision and NLP problems

## Usage
### Extracting features from an image

```python
from deepfeatx.image import ImageFeatureExtractor
fe = ImageFeatureExtractor()
```

    2021-10-06 11:27:12.595100: I tensorflow/core/common_runtime/pluggable_device/pluggable_device_factory.cc:305] Could not identify NUMA node of platform GPU ID 0, defaulting to 0. Your kernel may not have been built with NUMA support.
    2021-10-06 11:27:12.595191: I tensorflow/core/common_runtime/pluggable_device/pluggable_device_factory.cc:271] Created TensorFlow device (/job:localhost/replica:0/task:0/device:GPU:0 with 0 MB memory) -> physical PluggableDevice (device: 0, name: METAL, pci bus id: <undefined>)


    Metal device set to: Apple M1


```python
im_url='https://raw.githubusercontent.com/WittmannF/deepfeatx/master/sample_data/cats_vs_dogs/valid/dog/dog.124.jpg'
fe.read_img_url(im_url)
```




![png](docs/images/output_5_0.png)



```python
fe.url_to_vector(im_url)
```

    2021-10-06 11:27:13.679687: I tensorflow/compiler/mlir/mlir_graph_optimization_pass.cc:185] None of the MLIR Optimization Passes are enabled (registered 2)
    2021-10-06 11:27:13.679874: W tensorflow/core/platform/profile_utils/cpu_utils.cc:128] Failed to get CPU frequency: 0 Hz
    2021-10-06 11:27:13.846942: I tensorflow/core/grappler/optimizers/custom_graph_optimizer_registry.cc:112] Plugin optimizer for device_type GPU is enabled.





    array([[0.282272  , 1.0504342 , 0.11333481, ..., 0.18499802, 0.02220213,
            0.06158632]], dtype=float32)



### Extracting Features from a Folder with Images

```python
!git clone https://github.com/WittmannF/image-scraper.git
```

    fatal: destination path 'image-scraper' already exists and is not an empty directory.


```python
df=fe.extract_features_from_directory('image-scraper/images/pug',
                                   classes_as_folders=False,
                                   export_vectors_as_df=True)

df.head()
```

    Found 4 validated image filenames.
    1/1 [==============================] - 0s 412ms/step


    2021-10-06 11:27:16.893822: I tensorflow/core/grappler/optimizers/custom_graph_optimizer_registry.cc:112] Plugin optimizer for device_type GPU is enabled.





<div>
<style scoped>
    .dataframe tbody tr th:only-of-type {
        vertical-align: middle;
    }

    .dataframe tbody tr th {
        vertical-align: top;
    }

    .dataframe thead th {
        text-align: right;
    }
</style>
<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th>filepaths</th>
      <th>0</th>
      <th>1</th>
      <th>2</th>
      <th>3</th>
      <th>4</th>
      <th>5</th>
      <th>6</th>
      <th>7</th>
      <th>8</th>
      <th>...</th>
      <th>2038</th>
      <th>2039</th>
      <th>2040</th>
      <th>2041</th>
      <th>2042</th>
      <th>2043</th>
      <th>2044</th>
      <th>2045</th>
      <th>2046</th>
      <th>2047</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>0</th>
      <td>image-scraper/images/pug/efd08a2dc5.jpg</td>
      <td>0.030705</td>
      <td>0.042393</td>
      <td>0.422986</td>
      <td>1.316509</td>
      <td>0.020907</td>
      <td>0.000000</td>
      <td>0.081956</td>
      <td>0.404423</td>
      <td>0.489835</td>
      <td>...</td>
      <td>0.013765</td>
      <td>0.642072</td>
      <td>1.818821</td>
      <td>0.299441</td>
      <td>0.000000</td>
      <td>0.419997</td>
      <td>0.200106</td>
      <td>0.179524</td>
      <td>0.026852</td>
      <td>0.079208</td>
    </tr>
    <tr>
      <th>1</th>
      <td>image-scraper/images/pug/24d0f1eee3.jpg</td>
      <td>0.068498</td>
      <td>0.319734</td>
      <td>0.081250</td>
      <td>1.248270</td>
      <td>0.035602</td>
      <td>0.003398</td>
      <td>0.000000</td>
      <td>0.131528</td>
      <td>0.099514</td>
      <td>...</td>
      <td>0.258502</td>
      <td>1.042543</td>
      <td>0.691716</td>
      <td>0.264937</td>
      <td>0.112621</td>
      <td>0.927995</td>
      <td>0.050389</td>
      <td>0.000000</td>
      <td>0.087217</td>
      <td>0.066992</td>
    </tr>
    <tr>
      <th>2</th>
      <td>image-scraper/images/pug/6fb189ce56.jpg</td>
      <td>0.373005</td>
      <td>0.102008</td>
      <td>0.097662</td>
      <td>0.362927</td>
      <td>0.549803</td>
      <td>0.118015</td>
      <td>0.000000</td>
      <td>0.104320</td>
      <td>0.102526</td>
      <td>...</td>
      <td>0.210635</td>
      <td>0.213147</td>
      <td>0.013510</td>
      <td>0.574433</td>
      <td>0.017234</td>
      <td>0.628009</td>
      <td>0.000000</td>
      <td>0.184550</td>
      <td>0.000000</td>
      <td>0.248099</td>
    </tr>
    <tr>
      <th>3</th>
      <td>image-scraper/images/pug/ee815ebc87.jpg</td>
      <td>0.263904</td>
      <td>0.430294</td>
      <td>0.391808</td>
      <td>0.033076</td>
      <td>0.200174</td>
      <td>0.019310</td>
      <td>0.002792</td>
      <td>0.129120</td>
      <td>0.050257</td>
      <td>...</td>
      <td>0.048244</td>
      <td>0.147806</td>
      <td>1.430154</td>
      <td>0.266686</td>
      <td>0.005126</td>
      <td>0.158225</td>
      <td>0.097526</td>
      <td>0.005045</td>
      <td>0.060016</td>
      <td>1.109626</td>
    </tr>
  </tbody>
</table>
<p>4 rows × 2049 columns</p>
</div>



### Extracting Features from a directory having one sub-folder per class

If the directory structure is the following:
```
main_directory/
...class_a/
......a_image_1.jpg
......a_image_2.jpg
...class_b/
......b_image_1.jpg
......b_image_2.jpg
```
We can enter `main_directory` as input by changing `classes_as_folders` as True:

```python
df=fe.extract_features_from_directory('image-scraper/images/',
                                      classes_as_folders=True,
                                      export_vectors_as_df=True,
                                      export_class_names=True)

df.head()
```

    Found 504 images belonging to 6 classes.


    2021-10-06 11:27:22.669056: I tensorflow/core/grappler/optimizers/custom_graph_optimizer_registry.cc:112] Plugin optimizer for device_type GPU is enabled.


    16/16 [==============================] - 6s 358ms/step





<div>
<style scoped>
    .dataframe tbody tr th:only-of-type {
        vertical-align: middle;
    }

    .dataframe tbody tr th {
        vertical-align: top;
    }

    .dataframe thead th {
        text-align: right;
    }
</style>
<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th>filepaths</th>
      <th>classes</th>
      <th>0</th>
      <th>1</th>
      <th>2</th>
      <th>3</th>
      <th>4</th>
      <th>5</th>
      <th>6</th>
      <th>7</th>
      <th>...</th>
      <th>2038</th>
      <th>2039</th>
      <th>2040</th>
      <th>2041</th>
      <th>2042</th>
      <th>2043</th>
      <th>2044</th>
      <th>2045</th>
      <th>2046</th>
      <th>2047</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>0</th>
      <td>image-scraper/images/chihuahua/00dcf98689.jpg</td>
      <td>chihuahua</td>
      <td>0.640897</td>
      <td>0.887126</td>
      <td>0.017012</td>
      <td>0.723459</td>
      <td>0.164907</td>
      <td>0.010150</td>
      <td>0.042344</td>
      <td>0.987457</td>
      <td>...</td>
      <td>0.289271</td>
      <td>0.182086</td>
      <td>0.638064</td>
      <td>0.092432</td>
      <td>0.212789</td>
      <td>0.077480</td>
      <td>0.255031</td>
      <td>0.006371</td>
      <td>0.489620</td>
      <td>0.028672</td>
    </tr>
    <tr>
      <th>1</th>
      <td>image-scraper/images/chihuahua/01ee02c2fb.jpg</td>
      <td>chihuahua</td>
      <td>0.357992</td>
      <td>0.128552</td>
      <td>0.227736</td>
      <td>0.652588</td>
      <td>0.014283</td>
      <td>0.092680</td>
      <td>0.049545</td>
      <td>0.319637</td>
      <td>...</td>
      <td>0.061090</td>
      <td>0.526585</td>
      <td>2.363337</td>
      <td>0.160859</td>
      <td>0.000000</td>
      <td>0.008739</td>
      <td>0.401081</td>
      <td>1.377398</td>
      <td>0.383465</td>
      <td>0.434211</td>
    </tr>
    <tr>
      <th>2</th>
      <td>image-scraper/images/chihuahua/040df01fb4.jpg</td>
      <td>chihuahua</td>
      <td>0.163308</td>
      <td>0.383921</td>
      <td>0.029490</td>
      <td>0.985443</td>
      <td>0.866045</td>
      <td>0.098337</td>
      <td>0.000000</td>
      <td>0.634062</td>
      <td>...</td>
      <td>0.188044</td>
      <td>0.000000</td>
      <td>0.056569</td>
      <td>1.115319</td>
      <td>0.000000</td>
      <td>0.005084</td>
      <td>0.072280</td>
      <td>0.555855</td>
      <td>0.333000</td>
      <td>0.413303</td>
    </tr>
    <tr>
      <th>3</th>
      <td>image-scraper/images/chihuahua/04d8487a97.jpg</td>
      <td>chihuahua</td>
      <td>0.206927</td>
      <td>3.128521</td>
      <td>0.147507</td>
      <td>0.104669</td>
      <td>0.554029</td>
      <td>2.415109</td>
      <td>0.009964</td>
      <td>0.171642</td>
      <td>...</td>
      <td>0.000000</td>
      <td>1.297839</td>
      <td>1.165449</td>
      <td>0.562891</td>
      <td>0.000000</td>
      <td>0.395750</td>
      <td>0.250796</td>
      <td>0.295067</td>
      <td>0.534072</td>
      <td>0.051334</td>
    </tr>
    <tr>
      <th>4</th>
      <td>image-scraper/images/chihuahua/0d9fa44dea.jpg</td>
      <td>chihuahua</td>
      <td>0.233232</td>
      <td>0.355028</td>
      <td>0.453336</td>
      <td>0.060354</td>
      <td>0.479405</td>
      <td>0.000000</td>
      <td>0.099390</td>
      <td>0.223719</td>
      <td>...</td>
      <td>0.308505</td>
      <td>0.376597</td>
      <td>1.075250</td>
      <td>0.416980</td>
      <td>0.073678</td>
      <td>0.316829</td>
      <td>0.620357</td>
      <td>0.125714</td>
      <td>0.179848</td>
      <td>0.110405</td>
    </tr>
  </tbody>
</table>
<p>5 rows × 2050 columns</p>
</div>



The usage of `export_class_names=True` will add a new column to the dataframe with the classes names.

## Examples
### Cats vs Dogs using Keras vs `deepfeatx`
First let's compare the code of one of the simplest deep learning libraries (Keras) with `deepfeatx`. As example, let's use a subset of Cats vs Dogs:

```python
from deepfeatx.image import download_dataset
download_dataset('https://github.com/dl7days/datasets/raw/master/cats-dogs-data.zip', 'cats-dogs-data.zip')
```

    Downloading Dataset...


    --2021-10-06 11:26:20--  https://github.com/dl7days/datasets/raw/master/cats-dogs-data.zip
    Resolving github.com (github.com)... 20.201.28.151
    Connecting to github.com (github.com)|20.201.28.151|:443... connected.
    HTTP request sent, awaiting response... 302 Found
    Location: https://raw.githubusercontent.com/dl7days/datasets/master/cats-dogs-data.zip [following]
    --2021-10-06 11:26:20--  https://raw.githubusercontent.com/dl7days/datasets/master/cats-dogs-data.zip
    Resolving raw.githubusercontent.com (raw.githubusercontent.com)... 185.199.109.133, 185.199.110.133, 185.199.108.133, ...
    Connecting to raw.githubusercontent.com (raw.githubusercontent.com)|185.199.109.133|:443... connected.
    HTTP request sent, awaiting response... 200 OK
    Length: 55203029 (53M) [application/zip]
    Saving to: ‘cats-dogs-data.zip’
    
         0K .......... .......... .......... .......... ..........  0% 1,53M 34s
        50K .......... .......... .......... .......... ..........  0% 3,92M 24s
       100K .......... .......... .......... .......... ..........  0% 2,51M 23s
       150K .......... .......... .......... .......... ..........  0% 4,11M 20s
       200K .......... .......... .......... .......... ..........  0% 10,0M 17s
       250K .......... .......... .......... .......... ..........  0% 3,24M 17s
       300K .......... .......... .......... .......... ..........  0% 7,93M 16s
       350K .......... .......... .......... .......... ..........  0% 8,54M 14s
       400K .......... .......... .......... .......... ..........  0% 54,7M 13s
       450K .......... .......... .......... .......... ..........  0% 9,20M 12s
       500K .......... .......... .......... .......... ..........  1% 45,5M 11s
       550K .......... .......... .......... .......... ..........  1% 6,86M 11s
       600K .......... .......... .......... .......... ..........  1% 11,1M 10s
       650K .......... .......... .......... .......... ..........  1% 8,95M 10s
       700K .......... .......... .......... .......... ..........  1% 55,4M 9s
       750K .......... .......... .......... .......... ..........  1% 6,09M 9s
       800K .......... .......... .......... .......... ..........  1%  116M 9s
       850K .......... .......... .......... .......... ..........  1%  148M 8s
       900K .......... .......... .......... .......... ..........  1% 30,5M 8s
       950K .......... .......... .......... .......... ..........  1% 88,6M 8s
      1000K .......... .......... .......... .......... ..........  1% 31,5M 7s
      1050K .......... .......... .......... .......... ..........  2% 20,6M 7s
      1100K .......... .......... .......... .......... ..........  2% 57,6M 7s
      1150K .......... .......... .......... .......... ..........  2% 39,6M 7s
      1200K .......... .......... .......... .......... ..........  2% 50,0M 6s
      1250K .......... .......... .......... .......... ..........  2% 4,18M 7s
      1300K .......... .......... .......... .......... ..........  2% 3,63M 7s
      1350K .......... .......... .......... .......... ..........  2% 20,8M 7s
      1400K .......... .......... .......... .......... ..........  2% 97,7M 6s
      1450K .......... .......... .......... .......... ..........  2% 16,5M 6s
      1500K .......... .......... .......... .......... ..........  2%  108M 6s
      1550K .......... .......... .......... .......... ..........  2% 19,8M 6s
      1600K .......... .......... .......... .......... ..........  3% 21,6M 6s
      1650K .......... .......... .......... .......... ..........  3% 38,8M 6s
      1700K .......... .......... .......... .......... ..........  3% 21,7M 6s
      1750K .......... .......... .......... .......... ..........  3% 48,8M 6s
      1800K .......... .......... .......... .......... ..........  3% 21,4M 5s
      1850K .......... .......... .......... .......... ..........  3%  108M 5s
      1900K .......... .......... .......... .......... ..........  3% 22,6M 5s
      1950K .......... .......... .......... .......... ..........  3% 76,9M 5s
      2000K .......... .......... .......... .......... ..........  3% 18,2M 5s
      2050K .......... .......... .......... .......... ..........  3% 25,6M 5s
      2100K .......... .......... .......... .......... ..........  3% 41,5M 5s
      2150K .......... .......... .......... .......... ..........  4% 66,6M 5s
      2200K .......... .......... .......... .......... ..........  4% 11,5M 5s
      2250K .......... .......... .......... .......... ..........  4%  150M 5s
      2300K .......... .......... .......... .......... ..........  4% 56,8M 5s
      2350K .......... .......... .......... .......... ..........  4% 40,7M 5s
      2400K .......... .......... .......... .......... ..........  4% 23,7M 4s
      2450K .......... .......... .......... .......... ..........  4%  110M 4s
      2500K .......... .......... .......... .......... ..........  4% 16,6M 4s
      2550K .......... .......... .......... .......... ..........  4% 24,0M 4s
      2600K .......... .......... .......... .......... ..........  4% 48,4M 4s
      2650K .......... .......... .......... .......... ..........  5% 16,5M 4s
      2700K .......... .......... .......... .......... ..........  5%  119M 4s
      2750K .......... .......... .......... .......... ..........  5% 19,8M 4s
      2800K .......... .......... .......... .......... ..........  5% 45,4M 4s
      2850K .......... .......... .......... .......... ..........  5% 18,1M 4s
      2900K .......... .......... .......... .......... ..........  5% 68,1M 4s
      2950K .......... .......... .......... .......... ..........  5% 44,6M 4s
      3000K .......... .......... .......... .......... ..........  5% 43,5M 4s
      3050K .......... .......... .......... .......... ..........  5% 20,6M 4s
      3100K .......... .......... .......... .......... ..........  5% 75,4M 4s
      3150K .......... .......... .......... .......... ..........  5% 20,3M 4s
      3200K .......... .......... .......... .......... ..........  6% 66,6M 4s
      3250K .......... .......... .......... .......... ..........  6% 23,5M 4s
      3300K .......... .......... .......... .......... ..........  6% 51,1M 4s
      3350K .......... .......... .......... .......... ..........  6% 20,2M 4s
      3400K .......... .......... .......... .......... ..........  6% 18,2M 4s
      3450K .......... .......... .......... .......... ..........  6% 85,4M 4s
      3500K .......... .......... .......... .......... ..........  6% 23,9M 4s
      3550K .......... .......... .......... .......... ..........  6% 26,5M 3s
      3600K .......... .......... .......... .......... ..........  6% 18,7M 3s
      3650K .......... .......... .......... .......... ..........  6% 66,1M 3s
      3700K .......... .......... .......... .......... ..........  6% 23,7M 3s
      3750K .......... .......... .......... .......... ..........  7% 24,3M 3s
      3800K .......... .......... .......... .......... ..........  7% 44,8M 3s
      3850K .......... .......... .......... .......... ..........  7% 36,2M 3s
      3900K .......... .......... .......... .......... ..........  7% 41,0M 3s
      3950K .......... .......... .......... .......... ..........  7% 57,9M 3s
      4000K .......... .......... .......... .......... ..........  7% 15,1M 3s
      4050K .......... .......... .......... .......... ..........  7%  153M 3s
      4100K .......... .......... .......... .......... ..........  7% 22,2M 3s
      4150K .......... .......... .......... .......... ..........  7% 78,6M 3s
      4200K .......... .......... .......... .......... ..........  7% 21,5M 3s
      4250K .......... .......... .......... .......... ..........  7% 88,6M 3s
      4300K .......... .......... .......... .......... ..........  8% 17,6M 3s
      4350K .......... .......... .......... .......... ..........  8% 77,3M 3s
      4400K .......... .......... .......... .......... ..........  8% 16,8M 3s
      4450K .......... .......... .......... .......... ..........  8% 19,9M 3s
      4500K .......... .......... .......... .......... ..........  8% 48,7M 3s
      4550K .......... .......... .......... .......... ..........  8%  150M 3s
      4600K .......... .......... .......... .......... ..........  8% 22,7M 3s
      4650K .......... .......... .......... .......... ..........  8% 49,5M 3s
      4700K .......... .......... .......... .......... ..........  8% 19,3M 3s
      4750K .......... .......... .......... .......... ..........  8%  171M 3s
      4800K .......... .......... .......... .......... ..........  8% 20,5M 3s
      4850K .......... .......... .......... .......... ..........  9% 19,9M 3s
      4900K .......... .......... .......... .......... ..........  9% 52,4M 3s
      4950K .......... .......... .......... .......... ..........  9% 15,2M 3s
      5000K .......... .......... .......... .......... ..........  9% 58,8M 3s
      5050K .......... .......... .......... .......... ..........  9% 16,6M 3s
      5100K .......... .......... .......... .......... ..........  9%  110M 3s
      5150K .......... .......... .......... .......... ..........  9% 43,6M 3s
      5200K .......... .......... .......... .......... ..........  9% 54,6M 3s
      5250K .......... .......... .......... .......... ..........  9% 22,4M 3s
      5300K .......... .......... .......... .......... ..........  9% 49,3M 3s
      5350K .......... .......... .......... .......... .......... 10% 13,7M 3s
      5400K .......... .......... .......... .......... .......... 10%  114M 3s
      5450K .......... .......... .......... .......... .......... 10% 22,9M 3s
      5500K .......... .......... .......... .......... .......... 10% 85,2M 3s
      5550K .......... .......... .......... .......... .......... 10% 16,4M 3s
      5600K .......... .......... .......... .......... .......... 10%  118M 3s
      5650K .......... .......... .......... .......... .......... 10% 34,8M 3s
      5700K .......... .......... .......... .......... .......... 10% 47,6M 3s
      5750K .......... .......... .......... .......... .......... 10% 15,5M 3s
      5800K .......... .......... .......... .......... .......... 10% 19,1M 3s
      5850K .......... .......... .......... .......... .......... 10% 58,4M 3s
      5900K .......... .......... .......... .......... .......... 11% 21,1M 3s
      5950K .......... .......... .......... .......... .......... 11% 92,5M 3s
      6000K .......... .......... .......... .......... .......... 11% 26,6M 3s
      6050K .......... .......... .......... .......... .......... 11% 58,8M 3s
      6100K .......... .......... .......... .......... .......... 11% 20,3M 3s
      6150K .......... .......... .......... .......... .......... 11% 49,1M 3s
      6200K .......... .......... .......... .......... .......... 11% 18,3M 3s
      6250K .......... .......... .......... .......... .......... 11% 53,2M 3s
      6300K .......... .......... .......... .......... .......... 11% 18,2M 3s
      6350K .......... .......... .......... .......... .......... 11% 79,5M 3s
      6400K .......... .......... .......... .......... .......... 11% 21,9M 3s
      6450K .......... .......... .......... .......... .......... 12% 29,8M 3s
      6500K .......... .......... .......... .......... .......... 12% 31,8M 2s
      6550K .......... .......... .......... .......... .......... 12% 54,2M 2s
      6600K .......... .......... .......... .......... .......... 12% 27,2M 2s
      6650K .......... .......... .......... .......... .......... 12% 70,8M 2s
      6700K .......... .......... .......... .......... .......... 12% 15,6M 2s
      6750K .......... .......... .......... .......... .......... 12% 18,9M 2s
      6800K .......... .......... .......... .......... .......... 12% 77,5M 2s
      6850K .......... .......... .......... .......... .......... 12% 22,4M 2s
      6900K .......... .......... .......... .......... .......... 12% 28,4M 2s
      6950K .......... .......... .......... .......... .......... 12% 23,1M 2s
      7000K .......... .......... .......... .......... .......... 13% 77,4M 2s
      7050K .......... .......... .......... .......... .......... 13% 25,6M 2s
      7100K .......... .......... .......... .......... .......... 13%  127M 2s
      7150K .......... .......... .......... .......... .......... 13% 17,2M 2s
      7200K .......... .......... .......... .......... .......... 13% 82,8M 2s
      7250K .......... .......... .......... .......... .......... 13% 15,9M 2s
      7300K .......... .......... .......... .......... .......... 13%  134M 2s
      7350K .......... .......... .......... .......... .......... 13% 29,7M 2s
      7400K .......... .......... .......... .......... .......... 13% 78,1M 2s
      7450K .......... .......... .......... .......... .......... 13% 16,4M 2s
      7500K .......... .......... .......... .......... .......... 14% 78,5M 2s
      7550K .......... .......... .......... .......... .......... 14% 21,0M 2s
      7600K .......... .......... .......... .......... .......... 14% 32,6M 2s
      7650K .......... .......... .......... .......... .......... 14% 27,7M 2s
      7700K .......... .......... .......... .......... .......... 14% 18,3M 2s
      7750K .......... .......... .......... .......... .......... 14% 41,7M 2s
      7800K .......... .......... .......... .......... .......... 14% 19,2M 2s
      7850K .......... .......... .......... .......... .......... 14%  114M 2s
      7900K .......... .......... .......... .......... .......... 14% 18,5M 2s
      7950K .......... .......... .......... .......... .......... 14% 54,0M 2s
      8000K .......... .......... .......... .......... .......... 14% 25,6M 2s
      8050K .......... .......... .......... .......... .......... 15% 87,3M 2s
      8100K .......... .......... .......... .......... .......... 15% 21,5M 2s
      8150K .......... .......... .......... .......... .......... 15% 26,4M 2s
      8200K .......... .......... .......... .......... .......... 15% 20,6M 2s
      8250K .......... .......... .......... .......... .......... 15% 99,2M 2s
      8300K .......... .......... .......... .......... .......... 15% 20,2M 2s
      8350K .......... .......... .......... .......... .......... 15%  151M 2s
      8400K .......... .......... .......... .......... .......... 15% 17,1M 2s
      8450K .......... .......... .......... .......... .......... 15% 59,6M 2s
      8500K .......... .......... .......... .......... .......... 15% 14,7M 2s
      8550K .......... .......... .......... .......... .......... 15% 87,8M 2s
      8600K .......... .......... .......... .......... .......... 16% 57,2M 2s
      8650K .......... .......... .......... .......... .......... 16% 70,7M 2s
      8700K .......... .......... .......... .......... .......... 16% 14,3M 2s
      8750K .......... .......... .......... .......... .......... 16%  169M 2s
      8800K .......... .......... .......... .......... .......... 16% 20,2M 2s
      8850K .......... .......... .......... .......... .......... 16% 19,2M 2s
      8900K .......... .......... .......... .......... .......... 16% 57,5M 2s
      8950K .......... .......... .......... .......... .......... 16% 54,4M 2s
      9000K .......... .......... .......... .......... .......... 16% 26,1M 2s
      9050K .......... .......... .......... .......... .......... 16% 22,6M 2s
      9100K .......... .......... .......... .......... .......... 16% 78,6M 2s
      9150K .......... .......... .......... .......... .......... 17% 14,3M 2s
      9200K .......... .......... .......... .......... .......... 17% 79,4M 2s
      9250K .......... .......... .......... .......... .......... 17% 16,2M 2s
      9300K .......... .......... .......... .......... .......... 17% 68,1M 2s
      9350K .......... .......... .......... .......... .......... 17% 27,8M 2s
      9400K .......... .......... .......... .......... .......... 17% 43,3M 2s
      9450K .......... .......... .......... .......... .......... 17% 29,3M 2s
      9500K .......... .......... .......... .......... .......... 17% 45,5M 2s
      9550K .......... .......... .......... .......... .......... 17% 21,6M 2s
      9600K .......... .......... .......... .......... .......... 17% 13,5M 2s
      9650K .......... .......... .......... .......... .......... 17%  152M 2s
      9700K .......... .......... .......... .......... .......... 18% 33,2M 2s
      9750K .......... .......... .......... .......... .......... 18% 94,6M 2s
      9800K .......... .......... .......... .......... .......... 18% 19,4M 2s
      9850K .......... .......... .......... .......... .......... 18% 67,9M 2s
      9900K .......... .......... .......... .......... .......... 18% 15,5M 2s
      9950K .......... .......... .......... .......... .......... 18% 52,4M 2s
     10000K .......... .......... .......... .......... .......... 18% 12,6M 2s
     10050K .......... .......... .......... .......... .......... 18%  168M 2s
     10100K .......... .......... .......... .......... .......... 18% 21,4M 2s
     10150K .......... .......... .......... .......... .......... 18%  137M 2s
     10200K .......... .......... .......... .......... .......... 19% 20,5M 2s
     10250K .......... .......... .......... .......... .......... 19% 5,43M 2s
     10300K .......... .......... .......... .......... .......... 19%  105M 2s
     10350K .......... .......... .......... .......... .......... 19%  126M 2s
     10400K .......... .......... .......... .......... .......... 19%  157M 2s
     10450K .......... .......... .......... .......... .......... 19%  157M 2s
     10500K .......... .......... .......... .......... .......... 19% 85,2M 2s
     10550K .......... .......... .......... .......... .......... 19% 87,0M 2s
     10600K .......... .......... .......... .......... .......... 19%  168M 2s
     10650K .......... .......... .......... .......... .......... 19%  156M 2s
     10700K .......... .......... .......... .......... .......... 19% 24,3M 2s
     10750K .......... .......... .......... .......... .......... 20% 80,7M 2s
     10800K .......... .......... .......... .......... .......... 20% 13,7M 2s
     10850K .......... .......... .......... .......... .......... 20% 17,5M 2s
     10900K .......... .......... .......... .......... .......... 20%  183M 2s
     10950K .......... .......... .......... .......... .......... 20% 19,8M 2s
     11000K .......... .......... .......... .......... .......... 20% 39,6M 2s
     11050K .......... .......... .......... .......... .......... 20% 37,5M 2s
     11100K .......... .......... .......... .......... .......... 20% 19,6M 2s
     11150K .......... .......... .......... .......... .......... 20%  152M 2s
     11200K .......... .......... .......... .......... .......... 20% 22,8M 2s
     11250K .......... .......... .......... .......... .......... 20% 44,3M 2s
     11300K .......... .......... .......... .......... .......... 21% 16,1M 2s
     11350K .......... .......... .......... .......... .......... 21% 55,4M 2s
     11400K .......... .......... .......... .......... .......... 21% 20,7M 2s
     11450K .......... .......... .......... .......... .......... 21%  103M 2s
     11500K .......... .......... .......... .......... .......... 21% 23,3M 2s
     11550K .......... .......... .......... .......... .......... 21% 95,4M 2s
     11600K .......... .......... .......... .......... .......... 21% 11,1M 2s
     11650K .......... .......... .......... .......... .......... 21%  175M 2s
     11700K .......... .......... .......... .......... .......... 21% 28,4M 2s
     11750K .......... .......... .......... .......... .......... 21%  103M 2s
     11800K .......... .......... .......... .......... .......... 21% 23,8M 2s
     11850K .......... .......... .......... .......... .......... 22% 68,7M 2s
     11900K .......... .......... .......... .......... .......... 22% 20,9M 2s
     11950K .......... .......... .......... .......... .......... 22% 51,0M 2s
     12000K .......... .......... .......... .......... .......... 22% 5,65M 2s
     12050K .......... .......... .......... .......... .......... 22%  117M 2s
     12100K .......... .......... .......... .......... .......... 22%  141M 2s
     12150K .......... .......... .......... .......... .......... 22%  139M 2s
     12200K .......... .......... .......... .......... .......... 22%  164M 2s
     12250K .......... .......... .......... .......... .......... 22% 54,0M 2s
     12300K .......... .......... .......... .......... .......... 22% 75,5M 2s
     12350K .......... .......... .......... .......... .......... 23% 22,6M 2s
     12400K .......... .......... .......... .......... .......... 23% 8,03M 2s
     12450K .......... .......... .......... .......... .......... 23% 80,7M 2s
     12500K .......... .......... .......... .......... .......... 23%  147M 2s
     12550K .......... .......... .......... .......... .......... 23%  121M 2s
     12600K .......... .......... .......... .......... .......... 23%  143M 2s
     12650K .......... .......... .......... .......... .......... 23% 33,3M 2s
     12700K .......... .......... .......... .......... .......... 23% 17,4M 2s
     12750K .......... .......... .......... .......... .......... 23% 62,4M 2s
     12800K .......... .......... .......... .......... .......... 23% 20,1M 2s
     12850K .......... .......... .......... .......... .......... 23% 82,5M 2s
     12900K .......... .......... .......... .......... .......... 24% 23,9M 2s
     12950K .......... .......... .......... .......... .......... 24% 40,5M 2s
     13000K .......... .......... .......... .......... .......... 24% 16,8M 2s
     13050K .......... .......... .......... .......... .......... 24%  139M 2s
     13100K .......... .......... .......... .......... .......... 24% 22,1M 2s
     13150K .......... .......... .......... .......... .......... 24% 87,8M 2s
     13200K .......... .......... .......... .......... .......... 24% 13,0M 2s
     13250K .......... .......... .......... .......... .......... 24% 95,2M 2s
     13300K .......... .......... .......... .......... .......... 24% 20,0M 2s
     13350K .......... .......... .......... .......... .......... 24%  124M 2s
     13400K .......... .......... .......... .......... .......... 24% 18,1M 2s
     13450K .......... .......... .......... .......... .......... 25%  148M 2s
     13500K .......... .......... .......... .......... .......... 25% 19,0M 2s
     13550K .......... .......... .......... .......... .......... 25% 15,6M 2s
     13600K .......... .......... .......... .......... .......... 25%  131M 2s
     13650K .......... .......... .......... .......... .......... 25%  180M 2s
     13700K .......... .......... .......... .......... .......... 25% 19,5M 2s
     13750K .......... .......... .......... .......... .......... 25% 18,3M 2s
     13800K .......... .......... .......... .......... .......... 25% 70,3M 2s
     13850K .......... .......... .......... .......... .......... 25% 16,2M 2s
     13900K .......... .......... .......... .......... .......... 25% 18,9M 2s
     13950K .......... .......... .......... .......... .......... 25%  141M 2s
     14000K .......... .......... .......... .......... .......... 26%  134M 2s
     14050K .......... .......... .......... .......... .......... 26% 19,0M 2s
     14100K .......... .......... .......... .......... .......... 26% 90,8M 2s
     14150K .......... .......... .......... .......... .......... 26% 18,6M 2s
     14200K .......... .......... .......... .......... .......... 26% 83,5M 2s
     14250K .......... .......... .......... .......... .......... 26% 22,4M 2s
     14300K .......... .......... .......... .......... .......... 26% 48,6M 2s
     14350K .......... .......... .......... .......... .......... 26% 23,4M 2s
     14400K .......... .......... .......... .......... .......... 26% 68,8M 2s
     14450K .......... .......... .......... .......... .......... 26% 15,3M 2s
     14500K .......... .......... .......... .......... .......... 26%  167M 2s
     14550K .......... .......... .......... .......... .......... 27% 15,9M 2s
     14600K .......... .......... .......... .......... .......... 27%  168M 2s
     14650K .......... .......... .......... .......... .......... 27% 18,0M 2s
     14700K .......... .......... .......... .......... .......... 27%  160M 2s
     14750K .......... .......... .......... .......... .......... 27% 15,3M 2s
     14800K .......... .......... .......... .......... .......... 27% 66,4M 2s
     14850K .......... .......... .......... .......... .......... 27% 24,8M 2s
     14900K .......... .......... .......... .......... .......... 27% 24,3M 2s
     14950K .......... .......... .......... .......... .......... 27%  115M 2s
     15000K .......... .......... .......... .......... .......... 27% 19,7M 2s
     15050K .......... .......... .......... .......... .......... 28% 72,0M 2s
     15100K .......... .......... .......... .......... .......... 28% 13,8M 2s
     15150K .......... .......... .......... .......... .......... 28%  110M 2s
     15200K .......... .......... .......... .......... .......... 28% 38,9M 2s
     15250K .......... .......... .......... .......... .......... 28% 54,0M 2s
     15300K .......... .......... .......... .......... .......... 28% 21,5M 2s
     15350K .......... .......... .......... .......... .......... 28% 63,9M 2s
     15400K .......... .......... .......... .......... .......... 28% 19,8M 2s
     15450K .......... .......... .......... .......... .......... 28% 77,4M 2s
     15500K .......... .......... .......... .......... .......... 28% 15,9M 2s
     15550K .......... .......... .......... .......... .......... 28%  107M 2s
     15600K .......... .......... .......... .......... .......... 29% 11,6M 2s
     15650K .......... .......... .......... .......... .......... 29% 75,1M 2s
     15700K .......... .......... .......... .......... .......... 29%  163M 2s
     15750K .......... .......... .......... .......... .......... 29% 16,1M 2s
     15800K .......... .......... .......... .......... .......... 29%  101M 2s
     15850K .......... .......... .......... .......... .......... 29% 16,7M 2s
     15900K .......... .......... .......... .......... .......... 29% 84,8M 2s
     15950K .......... .......... .......... .......... .......... 29% 17,9M 2s
     16000K .......... .......... .......... .......... .......... 29% 84,5M 2s
     16050K .......... .......... .......... .......... .......... 29% 23,1M 2s
     16100K .......... .......... .......... .......... .......... 29% 79,0M 2s
     16150K .......... .......... .......... .......... .......... 30% 19,3M 2s
     16200K .......... .......... .......... .......... .......... 30% 60,4M 2s
     16250K .......... .......... .......... .......... .......... 30% 18,9M 2s
     16300K .......... .......... .......... .......... .......... 30% 99,9M 2s
     16350K .......... .......... .......... .......... .......... 30% 11,8M 2s
     16400K .......... .......... .......... .......... .......... 30% 85,4M 2s
     16450K .......... .......... .......... .......... .......... 30% 63,2M 2s
     16500K .......... .......... .......... .......... .......... 30% 19,3M 2s
     16550K .......... .......... .......... .......... .......... 30% 81,8M 2s
     16600K .......... .......... .......... .......... .......... 30% 12,3M 2s
     16650K .......... .......... .......... .......... .......... 30% 95,4M 2s
     16700K .......... .......... .......... .......... .......... 31% 24,4M 2s
     16750K .......... .......... .......... .......... .......... 31% 78,4M 2s
     16800K .......... .......... .......... .......... .......... 31% 20,7M 1s
     16850K .......... .......... .......... .......... .......... 31% 47,2M 1s
     16900K .......... .......... .......... .......... .......... 31%  172M 1s
     16950K .......... .......... .......... .......... .......... 31% 19,1M 1s
     17000K .......... .......... .......... .......... .......... 31% 21,4M 1s
     17050K .......... .......... .......... .......... .......... 31% 44,0M 1s
     17100K .......... .......... .......... .......... .......... 31% 25,4M 1s
     17150K .......... .......... .......... .......... .......... 31% 74,9M 1s
     17200K .......... .......... .......... .......... .......... 31% 25,1M 1s
     17250K .......... .......... .......... .......... .......... 32% 86,4M 1s
     17300K .......... .......... .......... .......... .......... 32% 18,2M 1s
     17350K .......... .......... .......... .......... .......... 32% 57,9M 1s
     17400K .......... .......... .......... .......... .......... 32% 21,1M 1s
     17450K .......... .......... .......... .......... .......... 32%  101M 1s
     17500K .......... .......... .......... .......... .......... 32% 16,0M 1s
     17550K .......... .......... .......... .......... .......... 32% 22,5M 1s
     17600K .......... .......... .......... .......... .......... 32% 54,6M 1s
     17650K .......... .......... .......... .......... .......... 32% 18,2M 1s
     17700K .......... .......... .......... .......... .......... 32% 95,4M 1s
     17750K .......... .......... .......... .......... .......... 33% 22,4M 1s
     17800K .......... .......... .......... .......... .......... 33% 60,8M 1s
     17850K .......... .......... .......... .......... .......... 33% 21,5M 1s
     17900K .......... .......... .......... .......... .......... 33% 82,1M 1s
     17950K .......... .......... .......... .......... .......... 33% 13,0M 1s
     18000K .......... .......... .......... .......... .......... 33%  109M 1s
     18050K .......... .......... .......... .......... .......... 33% 20,6M 1s
     18100K .......... .......... .......... .......... .......... 33% 73,2M 1s
     18150K .......... .......... .......... .......... .......... 33% 24,2M 1s
     18200K .......... .......... .......... .......... .......... 33%  120M 1s
     18250K .......... .......... .......... .......... .......... 33% 16,9M 1s
     18300K .......... .......... .......... .......... .......... 34% 15,3M 1s
     18350K .......... .......... .......... .......... .......... 34%  116M 1s
     18400K .......... .......... .......... .......... .......... 34% 18,3M 1s
     18450K .......... .......... .......... .......... .......... 34% 51,3M 1s
     18500K .......... .......... .......... .......... .......... 34% 23,1M 1s
     18550K .......... .......... .......... .......... .......... 34%  125M 1s
     18600K .......... .......... .......... .......... .......... 34% 17,2M 1s
     18650K .......... .......... .......... .......... .......... 34% 69,0M 1s
     18700K .......... .......... .......... .......... .......... 34% 20,0M 1s
     18750K .......... .......... .......... .......... .......... 34% 72,9M 1s
     18800K .......... .......... .......... .......... .......... 34% 22,3M 1s
     18850K .......... .......... .......... .......... .......... 35% 50,0M 1s
     18900K .......... .......... .......... .......... .......... 35% 19,9M 1s
     18950K .......... .......... .......... .......... .......... 35%  109M 1s
     19000K .......... .......... .......... .......... .......... 35% 21,1M 1s
     19050K .......... .......... .......... .......... .......... 35% 45,4M 1s
     19100K .......... .......... .......... .......... .......... 35% 19,2M 1s
     19150K .......... .......... .......... .......... .......... 35% 66,7M 1s
     19200K .......... .......... .......... .......... .......... 35% 26,6M 1s
     19250K .......... .......... .......... .......... .......... 35% 72,6M 1s
     19300K .......... .......... .......... .......... .......... 35% 20,8M 1s
     19350K .......... .......... .......... .......... .......... 35% 49,4M 1s
     19400K .......... .......... .......... .......... .......... 36% 17,1M 1s
     19450K .......... .......... .......... .......... .......... 36% 37,7M 1s
     19500K .......... .......... .......... .......... .......... 36% 13,4M 1s
     19550K .......... .......... .......... .......... .......... 36%  151M 1s
     19600K .......... .......... .......... .......... .......... 36% 43,3M 1s
     19650K .......... .......... .......... .......... .......... 36%  143M 1s
     19700K .......... .......... .......... .......... .......... 36% 17,6M 1s
     19750K .......... .......... .......... .......... .......... 36% 55,4M 1s
     19800K .......... .......... .......... .......... .......... 36% 20,7M 1s
     19850K .......... .......... .......... .......... .......... 36% 58,8M 1s
     19900K .......... .......... .......... .......... .......... 37% 19,4M 1s
     19950K .......... .......... .......... .......... .......... 37% 86,9M 1s
     20000K .......... .......... .......... .......... .......... 37% 28,1M 1s
     20050K .......... .......... .......... .......... .......... 37% 13,7M 1s
     20100K .......... .......... .......... .......... .......... 37% 56,4M 1s
     20150K .......... .......... .......... .......... .......... 37% 17,3M 1s
     20200K .......... .......... .......... .......... .......... 37%  130M 1s
     20250K .......... .......... .......... .......... .......... 37% 32,8M 1s
     20300K .......... .......... .......... .......... .......... 37% 51,7M 1s
     20350K .......... .......... .......... .......... .......... 37% 21,0M 1s
     20400K .......... .......... .......... .......... .......... 37% 85,1M 1s
     20450K .......... .......... .......... .......... .......... 38% 19,2M 1s
     20500K .......... .......... .......... .......... .......... 38% 97,3M 1s
     20550K .......... .......... .......... .......... .......... 38% 15,5M 1s
     20600K .......... .......... .......... .......... .......... 38% 59,0M 1s
     20650K .......... .......... .......... .......... .......... 38% 26,7M 1s
     20700K .......... .......... .......... .......... .......... 38% 52,8M 1s
     20750K .......... .......... .......... .......... .......... 38% 17,9M 1s
     20800K .......... .......... .......... .......... .......... 38% 15,5M 1s
     20850K .......... .......... .......... .......... .......... 38%  108M 1s
     20900K .......... .......... .......... .......... .......... 38% 16,3M 1s
     20950K .......... .......... .......... .......... .......... 38% 67,5M 1s
     21000K .......... .......... .......... .......... .......... 39% 50,0M 1s
     21050K .......... .......... .......... .......... .......... 39% 49,2M 1s
     21100K .......... .......... .......... .......... .......... 39% 12,9M 1s
     21150K .......... .......... .......... .......... .......... 39% 64,2M 1s
     21200K .......... .......... .......... .......... .......... 39% 50,2M 1s
     21250K .......... .......... .......... .......... .......... 39% 85,5M 1s
     21300K .......... .......... .......... .......... .......... 39% 18,0M 1s
     21350K .......... .......... .......... .......... .......... 39% 27,5M 1s
     21400K .......... .......... .......... .......... .......... 39% 29,2M 1s
     21450K .......... .......... .......... .......... .......... 39%  144M 1s
     21500K .......... .......... .......... .......... .......... 39% 16,4M 1s
     21550K .......... .......... .......... .......... .......... 40% 21,8M 1s
     21600K .......... .......... .......... .......... .......... 40% 68,2M 1s
     21650K .......... .......... .......... .......... .......... 40% 86,3M 1s
     21700K .......... .......... .......... .......... .......... 40% 22,9M 1s
     21750K .......... .......... .......... .......... .......... 40% 47,5M 1s
     21800K .......... .......... .......... .......... .......... 40% 18,7M 1s
     21850K .......... .......... .......... .......... .......... 40% 64,8M 1s
     21900K .......... .......... .......... .......... .......... 40% 21,8M 1s
     21950K .......... .......... .......... .......... .......... 40% 16,8M 1s
     22000K .......... .......... .......... .......... .......... 40% 85,2M 1s
     22050K .......... .......... .......... .......... .......... 40% 20,3M 1s
     22100K .......... .......... .......... .......... .......... 41%  126M 1s
     22150K .......... .......... .......... .......... .......... 41% 17,0M 1s
     22200K .......... .......... .......... .......... .......... 41% 95,4M 1s
     22250K .......... .......... .......... .......... .......... 41% 25,8M 1s
     22300K .......... .......... .......... .......... .......... 41%  100M 1s
     22350K .......... .......... .......... .......... .......... 41% 15,5M 1s
     22400K .......... .......... .......... .......... .......... 41%  146M 1s
     22450K .......... .......... .......... .......... .......... 41% 28,3M 1s
     22500K .......... .......... .......... .......... .......... 41% 45,5M 1s
     22550K .......... .......... .......... .......... .......... 41% 21,5M 1s
     22600K .......... .......... .......... .......... .......... 42% 74,3M 1s
     22650K .......... .......... .......... .......... .......... 42% 10,0M 1s
     22700K .......... .......... .......... .......... .......... 42% 62,5M 1s
     22750K .......... .......... .......... .......... .......... 42% 68,2M 1s
     22800K .......... .......... .......... .......... .......... 42% 18,1M 1s
     22850K .......... .......... .......... .......... .......... 42% 93,0M 1s
     22900K .......... .......... .......... .......... .......... 42% 25,5M 1s
     22950K .......... .......... .......... .......... .......... 42% 93,2M 1s
     23000K .......... .......... .......... .......... .......... 42% 21,4M 1s
     23050K .......... .......... .......... .......... .......... 42% 61,7M 1s
     23100K .......... .......... .......... .......... .......... 42% 12,4M 1s
     23150K .......... .......... .......... .......... .......... 43%  101M 1s
     23200K .......... .......... .......... .......... .......... 43% 43,2M 1s
     23250K .......... .......... .......... .......... .......... 43% 49,3M 1s
     23300K .......... .......... .......... .......... .......... 43% 17,3M 1s
     23350K .......... .......... .......... .......... .......... 43%  145M 1s
     23400K .......... .......... .......... .......... .......... 43% 24,3M 1s
     23450K .......... .......... .......... .......... .......... 43% 50,2M 1s
     23500K .......... .......... .......... .......... .......... 43% 15,9M 1s
     23550K .......... .......... .......... .......... .......... 43% 19,4M 1s
     23600K .......... .......... .......... .......... .......... 43% 57,4M 1s
     23650K .......... .......... .......... .......... .......... 43% 17,7M 1s
     23700K .......... .......... .......... .......... .......... 44%  148M 1s
     23750K .......... .......... .......... .......... .......... 44% 19,6M 1s
     23800K .......... .......... .......... .......... .......... 44% 59,4M 1s
     23850K .......... .......... .......... .......... .......... 44% 15,7M 1s
     23900K .......... .......... .......... .......... .......... 44%  146M 1s
     23950K .......... .......... .......... .......... .......... 44% 29,2M 1s
     24000K .......... .......... .......... .......... .......... 44% 50,4M 1s
     24050K .......... .......... .......... .......... .......... 44% 21,3M 1s
     24100K .......... .......... .......... .......... .......... 44% 88,1M 1s
     24150K .......... .......... .......... .......... .......... 44% 18,4M 1s
     24200K .......... .......... .......... .......... .......... 44% 97,3M 1s
     24250K .......... .......... .......... .......... .......... 45% 11,2M 1s
     24300K .......... .......... .......... .......... .......... 45% 28,8M 1s
     24350K .......... .......... .......... .......... .......... 45% 85,2M 1s
     24400K .......... .......... .......... .......... .......... 45%  162M 1s
     24450K .......... .......... .......... .......... .......... 45% 22,5M 1s
     24500K .......... .......... .......... .......... .......... 45% 79,9M 1s
     24550K .......... .......... .......... .......... .......... 45% 12,2M 1s
     24600K .......... .......... .......... .......... .......... 45%  123M 1s
     24650K .......... .......... .......... .......... .......... 45% 25,0M 1s
     24700K .......... .......... .......... .......... .......... 45%  146M 1s
     24750K .......... .......... .......... .......... .......... 46% 19,8M 1s
     24800K .......... .......... .......... .......... .......... 46% 40,2M 1s
     24850K .......... .......... .......... .......... .......... 46% 19,5M 1s
     24900K .......... .......... .......... .......... .......... 46%  124M 1s
     24950K .......... .......... .......... .......... .......... 46% 21,9M 1s
     25000K .......... .......... .......... .......... .......... 46%  131M 1s
     25050K .......... .......... .......... .......... .......... 46% 23,3M 1s
     25100K .......... .......... .......... .......... .......... 46% 14,0M 1s
     25150K .......... .......... .......... .......... .......... 46%  135M 1s
     25200K .......... .......... .......... .......... .......... 46% 20,1M 1s
     25250K .......... .......... .......... .......... .......... 46% 62,2M 1s
     25300K .......... .......... .......... .......... .......... 47% 22,6M 1s
     25350K .......... .......... .......... .......... .......... 47% 77,6M 1s
     25400K .......... .......... .......... .......... .......... 47% 18,4M 1s
     25450K .......... .......... .......... .......... .......... 47% 9,80M 1s
     25500K .......... .......... .......... .......... .......... 47%  121M 1s
     25550K .......... .......... .......... .......... .......... 47%  109M 1s
     25600K .......... .......... .......... .......... .......... 47% 71,4M 1s
     25650K .......... .......... .......... .......... .......... 47% 47,5M 1s
     25700K .......... .......... .......... .......... .......... 47% 18,4M 1s
     25750K .......... .......... .......... .......... .......... 47% 46,7M 1s
     25800K .......... .......... .......... .......... .......... 47% 16,3M 1s
     25850K .......... .......... .......... .......... .......... 48%  110M 1s
     25900K .......... .......... .......... .......... .......... 48% 19,3M 1s
     25950K .......... .......... .......... .......... .......... 48%  113M 1s
     26000K .......... .......... .......... .......... .......... 48% 25,9M 1s
     26050K .......... .......... .......... .......... .......... 48% 64,3M 1s
     26100K .......... .......... .......... .......... .......... 48% 16,2M 1s
     26150K .......... .......... .......... .......... .......... 48% 65,1M 1s
     26200K .......... .......... .......... .......... .......... 48% 14,7M 1s
     26250K .......... .......... .......... .......... .......... 48%  129M 1s
     26300K .......... .......... .......... .......... .......... 48% 26,2M 1s
     26350K .......... .......... .......... .......... .......... 48% 55,3M 1s
     26400K .......... .......... .......... .......... .......... 49% 19,5M 1s
     26450K .......... .......... .......... .......... .......... 49% 66,1M 1s
     26500K .......... .......... .......... .......... .......... 49% 17,2M 1s
     26550K .......... .......... .......... .......... .......... 49%  112M 1s
     26600K .......... .......... .......... .......... .......... 49% 16,4M 1s
     26650K .......... .......... .......... .......... .......... 49%  235M 1s
     26700K .......... .......... .......... .......... .......... 49% 13,6M 1s
     26750K .......... .......... .......... .......... .......... 49%  160M 1s
     26800K .......... .......... .......... .......... .......... 49% 22,7M 1s
     26850K .......... .......... .......... .......... .......... 49%  110M 1s
     26900K .......... .......... .......... .......... .......... 49% 22,7M 1s
     26950K .......... .......... .......... .......... .......... 50%  108M 1s
     27000K .......... .......... .......... .......... .......... 50% 19,6M 1s
     27050K .......... .......... .......... .......... .......... 50%  128M 1s
     27100K .......... .......... .......... .......... .......... 50% 17,6M 1s
     27150K .......... .......... .......... .......... .......... 50% 44,7M 1s
     27200K .......... .......... .......... .......... .......... 50% 17,6M 1s
     27250K .......... .......... .......... .......... .......... 50% 30,6M 1s
     27300K .......... .......... .......... .......... .......... 50% 20,9M 1s
     27350K .......... .......... .......... .......... .......... 50% 21,1M 1s
     27400K .......... .......... .......... .......... .......... 50%  129M 1s
     27450K .......... .......... .......... .......... .......... 51%  143M 1s
     27500K .......... .......... .......... .......... .......... 51% 23,0M 1s
     27550K .......... .......... .......... .......... .......... 51% 80,3M 1s
     27600K .......... .......... .......... .......... .......... 51% 15,6M 1s
     27650K .......... .......... .......... .......... .......... 51% 91,6M 1s
     27700K .......... .......... .......... .......... .......... 51% 17,1M 1s
     27750K .......... .......... .......... .......... .......... 51%  105M 1s
     27800K .......... .......... .......... .......... .......... 51% 16,0M 1s
     27850K .......... .......... .......... .......... .......... 51%  265M 1s
     27900K .......... .......... .......... .......... .......... 51% 19,7M 1s
     27950K .......... .......... .......... .......... .......... 51% 92,1M 1s
     28000K .......... .......... .......... .......... .......... 52% 16,3M 1s
     28050K .......... .......... .......... .......... .......... 52% 18,3M 1s
     28100K .......... .......... .......... .......... .......... 52%  167M 1s
     28150K .......... .......... .......... .......... .......... 52% 20,8M 1s
     28200K .......... .......... .......... .......... .......... 52% 66,3M 1s
     28250K .......... .......... .......... .......... .......... 52% 20,8M 1s
     28300K .......... .......... .......... .......... .......... 52%  103M 1s
     28350K .......... .......... .......... .......... .......... 52% 16,3M 1s
     28400K .......... .......... .......... .......... .......... 52% 54,6M 1s
     28450K .......... .......... .......... .......... .......... 52% 36,1M 1s
     28500K .......... .......... .......... .......... .......... 52% 44,8M 1s
     28550K .......... .......... .......... .......... .......... 53% 14,7M 1s
     28600K .......... .......... .......... .......... .......... 53%  114M 1s
     28650K .......... .......... .......... .......... .......... 53% 15,7M 1s
     28700K .......... .......... .......... .......... .......... 53%  102M 1s
     28750K .......... .......... .......... .......... .......... 53% 27,3M 1s
     28800K .......... .......... .......... .......... .......... 53% 56,7M 1s
     28850K .......... .......... .......... .......... .......... 53% 15,2M 1s
     28900K .......... .......... .......... .......... .......... 53% 81,1M 1s
     28950K .......... .......... .......... .......... .......... 53% 19,9M 1s
     29000K .......... .......... .......... .......... .......... 53%  182M 1s
     29050K .......... .......... .......... .......... .......... 53% 29,0M 1s
     29100K .......... .......... .......... .......... .......... 54% 17,0M 1s
     29150K .......... .......... .......... .......... .......... 54% 90,9M 1s
     29200K .......... .......... .......... .......... .......... 54% 21,1M 1s
     29250K .......... .......... .......... .......... .......... 54% 52,8M 1s
     29300K .......... .......... .......... .......... .......... 54% 17,2M 1s
     29350K .......... .......... .......... .......... .......... 54% 72,2M 1s
     29400K .......... .......... .......... .......... .......... 54% 29,3M 1s
     29450K .......... .......... .......... .......... .......... 54% 83,2M 1s
     29500K .......... .......... .......... .......... .......... 54% 23,9M 1s
     29550K .......... .......... .......... .......... .......... 54% 53,5M 1s
     29600K .......... .......... .......... .......... .......... 54% 14,4M 1s
     29650K .......... .......... .......... .......... .......... 55%  242M 1s
     29700K .......... .......... .......... .......... .......... 55% 22,0M 1s
     29750K .......... .......... .......... .......... .......... 55%  106M 1s
     29800K .......... .......... .......... .......... .......... 55% 16,3M 1s
     29850K .......... .......... .......... .......... .......... 55% 68,3M 1s
     29900K .......... .......... .......... .......... .......... 55% 16,2M 1s
     29950K .......... .......... .......... .......... .......... 55% 17,0M 1s
     30000K .......... .......... .......... .......... .......... 55%  111M 1s
     30050K .......... .......... .......... .......... .......... 55% 19,3M 1s
     30100K .......... .......... .......... .......... .......... 55% 57,6M 1s
     30150K .......... .......... .......... .......... .......... 56% 24,2M 1s
     30200K .......... .......... .......... .......... .......... 56%  254M 1s
     30250K .......... .......... .......... .......... .......... 56% 22,7M 1s
     30300K .......... .......... .......... .......... .......... 56% 55,9M 1s
     30350K .......... .......... .......... .......... .......... 56% 15,1M 1s
     30400K .......... .......... .......... .......... .......... 56% 93,2M 1s
     30450K .......... .......... .......... .......... .......... 56% 13,6M 1s
     30500K .......... .......... .......... .......... .......... 56% 59,6M 1s
     30550K .......... .......... .......... .......... .......... 56% 32,4M 1s
     30600K .......... .......... .......... .......... .......... 56% 82,6M 1s
     30650K .......... .......... .......... .......... .......... 56% 17,1M 1s
     30700K .......... .......... .......... .......... .......... 57% 74,8M 1s
     30750K .......... .......... .......... .......... .......... 57% 28,9M 1s
     30800K .......... .......... .......... .......... .......... 57%  108M 1s
     30850K .......... .......... .......... .......... .......... 57% 16,3M 1s
     30900K .......... .......... .......... .......... .......... 57%  127M 1s
     30950K .......... .......... .......... .......... .......... 57% 21,0M 1s
     31000K .......... .......... .......... .......... .......... 57% 13,1M 1s
     31050K .......... .......... .......... .......... .......... 57%  155M 1s
     31100K .......... .......... .......... .......... .......... 57% 26,2M 1s
     31150K .......... .......... .......... .......... .......... 57%  131M 1s
     31200K .......... .......... .......... .......... .......... 57% 16,1M 1s
     31250K .......... .......... .......... .......... .......... 58% 69,2M 1s
     31300K .......... .......... .......... .......... .......... 58% 20,9M 1s
     31350K .......... .......... .......... .......... .......... 58% 29,0M 1s
     31400K .......... .......... .......... .......... .......... 58% 27,1M 1s
     31450K .......... .......... .......... .......... .......... 58%  185M 1s
     31500K .......... .......... .......... .......... .......... 58% 14,9M 1s
     31550K .......... .......... .......... .......... .......... 58%  170M 1s
     31600K .......... .......... .......... .......... .......... 58% 23,9M 1s
     31650K .......... .......... .......... .......... .......... 58% 94,6M 1s
     31700K .......... .......... .......... .......... .......... 58% 17,9M 1s
     31750K .......... .......... .......... .......... .......... 58% 98,8M 1s
     31800K .......... .......... .......... .......... .......... 59% 18,2M 1s
     31850K .......... .......... .......... .......... .......... 59% 23,1M 1s
     31900K .......... .......... .......... .......... .......... 59% 52,7M 1s
     31950K .......... .......... .......... .......... .......... 59% 14,8M 1s
     32000K .......... .......... .......... .......... .......... 59%  158M 1s
     32050K .......... .......... .......... .......... .......... 59% 25,1M 1s
     32100K .......... .......... .......... .......... .......... 59% 70,6M 1s
     32150K .......... .......... .......... .......... .......... 59% 23,3M 1s
     32200K .......... .......... .......... .......... .......... 59% 49,8M 1s
     32250K .......... .......... .......... .......... .......... 59% 19,2M 1s
     32300K .......... .......... .......... .......... .......... 60% 75,4M 1s
     32350K .......... .......... .......... .......... .......... 60% 14,5M 1s
     32400K .......... .......... .......... .......... .......... 60%  185M 1s
     32450K .......... .......... .......... .......... .......... 60% 24,8M 1s
     32500K .......... .......... .......... .......... .......... 60%  150M 1s
     32550K .......... .......... .......... .......... .......... 60% 15,6M 1s
     32600K .......... .......... .......... .......... .......... 60%  168M 1s
     32650K .......... .......... .......... .......... .......... 60% 17,0M 1s
     32700K .......... .......... .......... .......... .......... 60% 90,1M 1s
     32750K .......... .......... .......... .......... .......... 60% 7,01M 1s
     32800K .......... .......... .......... .......... .......... 60% 30,7M 1s
     32850K .......... .......... .......... .......... .......... 61%  208M 1s
     32900K .......... .......... .......... .......... .......... 61%  135M 1s
     32950K .......... .......... .......... .......... .......... 61%  191M 1s
     33000K .......... .......... .......... .......... .......... 61% 86,7M 1s
     33050K .......... .......... .......... .......... .......... 61% 16,6M 1s
     33100K .......... .......... .......... .......... .......... 61%  135M 1s
     33150K .......... .......... .......... .......... .......... 61% 16,7M 1s
     33200K .......... .......... .......... .......... .......... 61% 19,8M 1s
     33250K .......... .......... .......... .......... .......... 61% 92,8M 1s
     33300K .......... .......... .......... .......... .......... 61% 19,6M 1s
     33350K .......... .......... .......... .......... .......... 61% 76,5M 1s
     33400K .......... .......... .......... .......... .......... 62% 31,8M 1s
     33450K .......... .......... .......... .......... .......... 62% 53,1M 1s
     33500K .......... .......... .......... .......... .......... 62% 18,5M 1s
     33550K .......... .......... .......... .......... .......... 62% 67,1M 1s
     33600K .......... .......... .......... .......... .......... 62% 18,4M 1s
     33650K .......... .......... .......... .......... .......... 62% 73,0M 1s
     33700K .......... .......... .......... .......... .......... 62% 25,7M 1s
     33750K .......... .......... .......... .......... .......... 62% 53,6M 1s
     33800K .......... .......... .......... .......... .......... 62% 16,8M 1s
     33850K .......... .......... .......... .......... .......... 62% 20,7M 1s
     33900K .......... .......... .......... .......... .......... 62%  132M 1s
     33950K .......... .......... .......... .......... .......... 63% 13,8M 1s
     34000K .......... .......... .......... .......... .......... 63%  152M 1s
     34050K .......... .......... .......... .......... .......... 63% 29,3M 1s
     34100K .......... .......... .......... .......... .......... 63% 64,8M 1s
     34150K .......... .......... .......... .......... .......... 63% 21,5M 1s
     34200K .......... .......... .......... .......... .......... 63% 53,9M 1s
     34250K .......... .......... .......... .......... .......... 63% 24,4M 1s
     34300K .......... .......... .......... .......... .......... 63% 43,2M 1s
     34350K .......... .......... .......... .......... .......... 63% 24,5M 1s
     34400K .......... .......... .......... .......... .......... 63%  105M 1s
     34450K .......... .......... .......... .......... .......... 63% 14,4M 1s
     34500K .......... .......... .......... .......... .......... 64% 20,0M 1s
     34550K .......... .......... .......... .......... .......... 64%  115M 1s
     34600K .......... .......... .......... .......... .......... 64% 22,3M 1s
     34650K .......... .......... .......... .......... .......... 64%  103M 1s
     34700K .......... .......... .......... .......... .......... 64% 17,4M 1s
     34750K .......... .......... .......... .......... .......... 64% 59,4M 1s
     34800K .......... .......... .......... .......... .......... 64% 20,2M 1s
     34850K .......... .......... .......... .......... .......... 64% 80,3M 1s
     34900K .......... .......... .......... .......... .......... 64% 19,6M 1s
     34950K .......... .......... .......... .......... .......... 64% 66,3M 1s
     35000K .......... .......... .......... .......... .......... 65% 15,3M 1s
     35050K .......... .......... .......... .......... .......... 65%  178M 1s
     35100K .......... .......... .......... .......... .......... 65% 25,0M 1s
     35150K .......... .......... .......... .......... .......... 65%  129M 1s
     35200K .......... .......... .......... .......... .......... 65% 12,6M 1s
     35250K .......... .......... .......... .......... .......... 65% 25,8M 1s
     35300K .......... .......... .......... .......... .......... 65% 75,2M 1s
     35350K .......... .......... .......... .......... .......... 65%  171M 1s
     35400K .......... .......... .......... .......... .......... 65% 13,8M 1s
     35450K .......... .......... .......... .......... .......... 65% 24,9M 1s
     35500K .......... .......... .......... .......... .......... 65% 96,5M 1s
     35550K .......... .......... .......... .......... .......... 66% 20,0M 1s
     35600K .......... .......... .......... .......... .......... 66% 88,9M 1s
     35650K .......... .......... .......... .......... .......... 66% 16,8M 1s
     35700K .......... .......... .......... .......... .......... 66% 22,7M 1s
     35750K .......... .......... .......... .......... .......... 66%  243M 1s
     35800K .......... .......... .......... .......... .......... 66% 37,2M 1s
     35850K .......... .......... .......... .......... .......... 66% 24,2M 1s
     35900K .......... .......... .......... .......... .......... 66% 51,5M 1s
     35950K .......... .......... .......... .......... .......... 66% 16,7M 1s
     36000K .......... .......... .......... .......... .......... 66%  183M 1s
     36050K .......... .......... .......... .......... .......... 66% 21,4M 1s
     36100K .......... .......... .......... .......... .......... 67% 56,3M 1s
     36150K .......... .......... .......... .......... .......... 67% 14,1M 1s
     36200K .......... .......... .......... .......... .......... 67%  107M 1s
     36250K .......... .......... .......... .......... .......... 67% 30,7M 1s
     36300K .......... .......... .......... .......... .......... 67% 51,1M 1s
     36350K .......... .......... .......... .......... .......... 67% 38,3M 1s
     36400K .......... .......... .......... .......... .......... 67% 14,8M 1s
     36450K .......... .......... .......... .......... .......... 67% 75,6M 1s
     36500K .......... .......... .......... .......... .......... 67% 13,6M 1s
     36550K .......... .......... .......... .......... .......... 67%  156M 1s
     36600K .......... .......... .......... .......... .......... 67% 17,8M 1s
     36650K .......... .......... .......... .......... .......... 68%  148M 1s
     36700K .......... .......... .......... .......... .......... 68% 23,0M 1s
     36750K .......... .......... .......... .......... .......... 68% 91,4M 1s
     36800K .......... .......... .......... .......... .......... 68% 9,21M 1s
     36850K .......... .......... .......... .......... .......... 68% 48,7M 1s
     36900K .......... .......... .......... .......... .......... 68% 93,5M 1s
     36950K .......... .......... .......... .......... .......... 68%  111M 1s
     37000K .......... .......... .......... .......... .......... 68% 39,8M 1s
     37050K .......... .......... .......... .......... .......... 68% 51,8M 1s
     37100K .......... .......... .......... .......... .......... 68% 17,0M 1s
     37150K .......... .......... .......... .......... .......... 69% 26,7M 1s
     37200K .......... .......... .......... .......... .......... 69%  135M 1s
     37250K .......... .......... .......... .......... .......... 69% 21,6M 1s
     37300K .......... .......... .......... .......... .......... 69% 38,3M 1s
     37350K .......... .......... .......... .......... .......... 69% 15,5M 1s
     37400K .......... .......... .......... .......... .......... 69% 72,6M 1s
     37450K .......... .......... .......... .......... .......... 69%  212M 1s
     37500K .......... .......... .......... .......... .......... 69% 18,3M 1s
     37550K .......... .......... .......... .......... .......... 69% 19,1M 1s
     37600K .......... .......... .......... .......... .......... 69% 55,5M 1s
     37650K .......... .......... .......... .......... .......... 69% 20,1M 1s
     37700K .......... .......... .......... .......... .......... 70% 60,5M 1s
     37750K .......... .......... .......... .......... .......... 70% 22,2M 1s
     37800K .......... .......... .......... .......... .......... 70% 36,2M 1s
     37850K .......... .......... .......... .......... .......... 70% 20,6M 1s
     37900K .......... .......... .......... .......... .......... 70%  125M 1s
     37950K .......... .......... .......... .......... .......... 70% 29,5M 1s
     38000K .......... .......... .......... .......... .......... 70% 80,7M 1s
     38050K .......... .......... .......... .......... .......... 70% 17,1M 1s
     38100K .......... .......... .......... .......... .......... 70% 58,7M 1s
     38150K .......... .......... .......... .......... .......... 70% 27,5M 1s
     38200K .......... .......... .......... .......... .......... 70% 43,1M 1s
     38250K .......... .......... .......... .......... .......... 71% 13,0M 1s
     38300K .......... .......... .......... .......... .......... 71% 25,0M 1s
     38350K .......... .......... .......... .......... .......... 71% 25,7M 1s
     38400K .......... .......... .......... .......... .......... 71% 58,8M 1s
     38450K .......... .......... .......... .......... .......... 71%  136M 1s
     38500K .......... .......... .......... .......... .......... 71% 18,6M 1s
     38550K .......... .......... .......... .......... .......... 71% 52,3M 1s
     38600K .......... .......... .......... .......... .......... 71% 14,9M 1s
     38650K .......... .......... .......... .......... .......... 71%  147M 1s
     38700K .......... .......... .......... .......... .......... 71% 18,8M 1s
     38750K .......... .......... .......... .......... .......... 71% 61,9M 1s
     38800K .......... .......... .......... .......... .......... 72% 21,1M 1s
     38850K .......... .......... .......... .......... .......... 72% 59,4M 1s
     38900K .......... .......... .......... .......... .......... 72% 20,1M 1s
     38950K .......... .......... .......... .......... .......... 72% 61,4M 1s
     39000K .......... .......... .......... .......... .......... 72% 15,6M 1s
     39050K .......... .......... .......... .......... .......... 72%  135M 1s
     39100K .......... .......... .......... .......... .......... 72% 29,2M 1s
     39150K .......... .......... .......... .......... .......... 72% 63,7M 1s
     39200K .......... .......... .......... .......... .......... 72% 16,3M 1s
     39250K .......... .......... .......... .......... .......... 72%  126M 1s
     39300K .......... .......... .......... .......... .......... 72% 24,4M 1s
     39350K .......... .......... .......... .......... .......... 73% 37,5M 1s
     39400K .......... .......... .......... .......... .......... 73% 17,5M 1s
     39450K .......... .......... .......... .......... .......... 73%  198M 1s
     39500K .......... .......... .......... .......... .......... 73% 20,4M 1s
     39550K .......... .......... .......... .......... .......... 73%  148M 1s
     39600K .......... .......... .......... .......... .......... 73% 16,9M 1s
     39650K .......... .......... .......... .......... .......... 73% 76,8M 1s
     39700K .......... .......... .......... .......... .......... 73% 17,8M 1s
     39750K .......... .......... .......... .......... .......... 73%  164M 1s
     39800K .......... .......... .......... .......... .......... 73% 23,1M 1s
     39850K .......... .......... .......... .......... .......... 74% 92,3M 1s
     39900K .......... .......... .......... .......... .......... 74% 12,9M 1s
     39950K .......... .......... .......... .......... .......... 74% 22,9M 0s
     40000K .......... .......... .......... .......... .......... 74%  220M 0s
     40050K .......... .......... .......... .......... .......... 74% 16,1M 0s
     40100K .......... .......... .......... .......... .......... 74% 76,5M 0s
     40150K .......... .......... .......... .......... .......... 74% 27,7M 0s
     40200K .......... .......... .......... .......... .......... 74% 55,9M 0s
     40250K .......... .......... .......... .......... .......... 74% 17,6M 0s
     40300K .......... .......... .......... .......... .......... 74%  225M 0s
     40350K .......... .......... .......... .......... .......... 74% 24,7M 0s
     40400K .......... .......... .......... .......... .......... 75% 61,7M 0s
     40450K .......... .......... .......... .......... .......... 75% 21,9M 0s
     40500K .......... .......... .......... .......... .......... 75% 49,2M 0s
     40550K .......... .......... .......... .......... .......... 75% 20,0M 0s
     40600K .......... .......... .......... .......... .......... 75% 75,8M 0s
     40650K .......... .......... .......... .......... .......... 75% 17,0M 0s
     40700K .......... .......... .......... .......... .......... 75%  187M 0s
     40750K .......... .......... .......... .......... .......... 75% 16,3M 0s
     40800K .......... .......... .......... .......... .......... 75% 25,0M 0s
     40850K .......... .......... .......... .......... .......... 75% 48,8M 0s
     40900K .......... .......... .......... .......... .......... 75% 22,2M 0s
     40950K .......... .......... .......... .......... .......... 76% 56,1M 0s
     41000K .......... .......... .......... .......... .......... 76% 20,0M 0s
     41050K .......... .......... .......... .......... .......... 76% 15,9M 0s
     41100K .......... .......... .......... .......... .......... 76% 81,5M 0s
     41150K .......... .......... .......... .......... .......... 76% 35,8M 0s
     41200K .......... .......... .......... .......... .......... 76% 20,9M 0s
     41250K .......... .......... .......... .......... .......... 76%  167M 0s
     41300K .......... .......... .......... .......... .......... 76% 18,7M 0s
     41350K .......... .......... .......... .......... .......... 76%  155M 0s
     41400K .......... .......... .......... .......... .......... 76% 26,0M 0s
     41450K .......... .......... .......... .......... .......... 76% 81,5M 0s
     41500K .......... .......... .......... .......... .......... 77% 13,4M 0s
     41550K .......... .......... .......... .......... .......... 77% 20,1M 0s
     41600K .......... .......... .......... .......... .......... 77%  114M 0s
     41650K .......... .......... .......... .......... .......... 77% 14,7M 0s
     41700K .......... .......... .......... .......... .......... 77%  146M 0s
     41750K .......... .......... .......... .......... .......... 77% 32,8M 0s
     41800K .......... .......... .......... .......... .......... 77% 98,6M 0s
     41850K .......... .......... .......... .......... .......... 77% 15,9M 0s
     41900K .......... .......... .......... .......... .......... 77%  144M 0s
     41950K .......... .......... .......... .......... .......... 77% 20,5M 0s
     42000K .......... .......... .......... .......... .......... 78% 70,1M 0s
     42050K .......... .......... .......... .......... .......... 78% 15,9M 0s
     42100K .......... .......... .......... .......... .......... 78%  227M 0s
     42150K .......... .......... .......... .......... .......... 78% 28,2M 0s
     42200K .......... .......... .......... .......... .......... 78% 62,4M 0s
     42250K .......... .......... .......... .......... .......... 78% 19,4M 0s
     42300K .......... .......... .......... .......... .......... 78% 93,9M 0s
     42350K .......... .......... .......... .......... .......... 78% 13,6M 0s
     42400K .......... .......... .......... .......... .......... 78%  103M 0s
     42450K .......... .......... .......... .......... .......... 78% 21,0M 0s
     42500K .......... .......... .......... .......... .......... 78% 43,2M 0s
     42550K .......... .......... .......... .......... .......... 79% 17,7M 0s
     42600K .......... .......... .......... .......... .......... 79%  102M 0s
     42650K .......... .......... .......... .......... .......... 79% 22,3M 0s
     42700K .......... .......... .......... .......... .......... 79%  116M 0s
     42750K .......... .......... .......... .......... .......... 79% 18,3M 0s
     42800K .......... .......... .......... .......... .......... 79% 15,8M 0s
     42850K .......... .......... .......... .......... .......... 79%  103M 0s
     42900K .......... .......... .......... .......... .......... 79% 16,4M 0s
     42950K .......... .......... .......... .......... .......... 79% 93,7M 0s
     43000K .......... .......... .......... .......... .......... 79% 15,4M 0s
     43050K .......... .......... .......... .......... .......... 79%  263M 0s
     43100K .......... .......... .......... .......... .......... 80% 26,9M 0s
     43150K .......... .......... .......... .......... .......... 80% 55,4M 0s
     43200K .......... .......... .......... .......... .......... 80% 20,0M 0s
     43250K .......... .......... .......... .......... .......... 80% 98,8M 0s
     43300K .......... .......... .......... .......... .......... 80% 14,8M 0s
     43350K .......... .......... .......... .......... .......... 80%  208M 0s
     43400K .......... .......... .......... .......... .......... 80% 21,0M 0s
     43450K .......... .......... .......... .......... .......... 80% 19,0M 0s
     43500K .......... .......... .......... .......... .......... 80%  103M 0s
     43550K .......... .......... .......... .......... .......... 80% 23,5M 0s
     43600K .......... .......... .......... .......... .......... 80%  108M 0s
     43650K .......... .......... .......... .......... .......... 81% 14,8M 0s
     43700K .......... .......... .......... .......... .......... 81% 93,0M 0s
     43750K .......... .......... .......... .......... .......... 81% 29,8M 0s
     43800K .......... .......... .......... .......... .......... 81% 63,2M 0s
     43850K .......... .......... .......... .......... .......... 81% 17,4M 0s
     43900K .......... .......... .......... .......... .......... 81%  160M 0s
     43950K .......... .......... .......... .......... .......... 81% 23,4M 0s
     44000K .......... .......... .......... .......... .......... 81% 38,0M 0s
     44050K .......... .......... .......... .......... .......... 81% 25,3M 0s
     44100K .......... .......... .......... .......... .......... 81% 13,4M 0s
     44150K .......... .......... .......... .......... .......... 81% 57,2M 0s
     44200K .......... .......... .......... .......... .......... 82% 26,6M 0s
     44250K .......... .......... .......... .......... .......... 82% 26,0M 0s
     44300K .......... .......... .......... .......... .......... 82% 36,7M 0s
     44350K .......... .......... .......... .......... .......... 82% 37,9M 0s
     44400K .......... .......... .......... .......... .......... 82% 28,9M 0s
     44450K .......... .......... .......... .......... .......... 82% 74,3M 0s
     44500K .......... .......... .......... .......... .......... 82% 18,1M 0s
     44550K .......... .......... .......... .......... .......... 82%  151M 0s
     44600K .......... .......... .......... .......... .......... 82% 13,0M 0s
     44650K .......... .......... .......... .......... .......... 82% 47,4M 0s
     44700K .......... .......... .......... .......... .......... 83% 21,9M 0s
     44750K .......... .......... .......... .......... .......... 83%  221M 0s
     44800K .......... .......... .......... .......... .......... 83% 37,9M 0s
     44850K .......... .......... .......... .......... .......... 83% 99,6M 0s
     44900K .......... .......... .......... .......... .......... 83% 20,3M 0s
     44950K .......... .......... .......... .......... .......... 83% 63,9M 0s
     45000K .......... .......... .......... .......... .......... 83% 19,5M 0s
     45050K .......... .......... .......... .......... .......... 83% 41,4M 0s
     45100K .......... .......... .......... .......... .......... 83% 19,6M 0s
     45150K .......... .......... .......... .......... .......... 83% 17,8M 0s
     45200K .......... .......... .......... .......... .......... 83%  198M 0s
     45250K .......... .......... .......... .......... .......... 84% 17,7M 0s
     45300K .......... .......... .......... .......... .......... 84% 88,9M 0s
     45350K .......... .......... .......... .......... .......... 84% 22,8M 0s
     45400K .......... .......... .......... .......... .......... 84% 60,4M 0s
     45450K .......... .......... .......... .......... .......... 84% 13,8M 0s
     45500K .......... .......... .......... .......... .......... 84%  250M 0s
     45550K .......... .......... .......... .......... .......... 84% 20,1M 0s
     45600K .......... .......... .......... .......... .......... 84% 18,7M 0s
     45650K .......... .......... .......... .......... .......... 84%  122M 0s
     45700K .......... .......... .......... .......... .......... 84% 21,5M 0s
     45750K .......... .......... .......... .......... .......... 84% 61,3M 0s
     45800K .......... .......... .......... .......... .......... 85% 22,3M 0s
     45850K .......... .......... .......... .......... .......... 85% 55,5M 0s
     45900K .......... .......... .......... .......... .......... 85% 17,8M 0s
     45950K .......... .......... .......... .......... .......... 85% 30,5M 0s
     46000K .......... .......... .......... .......... .......... 85% 24,0M 0s
     46050K .......... .......... .......... .......... .......... 85% 73,4M 0s
     46100K .......... .......... .......... .......... .......... 85% 21,0M 0s
     46150K .......... .......... .......... .......... .......... 85% 57,5M 0s
     46200K .......... .......... .......... .......... .......... 85% 14,6M 0s
     46250K .......... .......... .......... .......... .......... 85%  154M 0s
     46300K .......... .......... .......... .......... .......... 85% 25,7M 0s
     46350K .......... .......... .......... .......... .......... 86% 78,1M 0s
     46400K .......... .......... .......... .......... .......... 86% 22,1M 0s
     46450K .......... .......... .......... .......... .......... 86% 72,0M 0s
     46500K .......... .......... .......... .......... .......... 86% 18,2M 0s
     46550K .......... .......... .......... .......... .......... 86%  138M 0s
     46600K .......... .......... .......... .......... .......... 86% 26,0M 0s
     46650K .......... .......... .......... .......... .......... 86% 57,0M 0s
     46700K .......... .......... .......... .......... .......... 86% 16,6M 0s
     46750K .......... .......... .......... .......... .......... 86% 64,2M 0s
     46800K .......... .......... .......... .......... .......... 86% 21,5M 0s
     46850K .......... .......... .......... .......... .......... 86% 14,1M 0s
     46900K .......... .......... .......... .......... .......... 87% 82,8M 0s
     46950K .......... .......... .......... .......... .......... 87% 41,9M 0s
     47000K .......... .......... .......... .......... .......... 87% 75,5M 0s
     47050K .......... .......... .......... .......... .......... 87% 16,9M 0s
     47100K .......... .......... .......... .......... .......... 87% 61,7M 0s
     47150K .......... .......... .......... .......... .......... 87% 26,2M 0s
     47200K .......... .......... .......... .......... .......... 87% 39,4M 0s
     47250K .......... .......... .......... .......... .......... 87% 25,9M 0s
     47300K .......... .......... .......... .......... .......... 87% 95,6M 0s
     47350K .......... .......... .......... .......... .......... 87% 19,4M 0s
     47400K .......... .......... .......... .......... .......... 88% 80,3M 0s
     47450K .......... .......... .......... .......... .......... 88% 16,3M 0s
     47500K .......... .......... .......... .......... .......... 88% 59,4M 0s
     47550K .......... .......... .......... .......... .......... 88% 15,0M 0s
     47600K .......... .......... .......... .......... .......... 88%  201M 0s
     47650K .......... .......... .......... .......... .......... 88% 17,0M 0s
     47700K .......... .......... .......... .......... .......... 88%  242M 0s
     47750K .......... .......... .......... .......... .......... 88% 20,6M 0s
     47800K .......... .......... .......... .......... .......... 88% 29,5M 0s
     47850K .......... .......... .......... .......... .......... 88% 22,1M 0s
     47900K .......... .......... .......... .......... .......... 88%  279M 0s
     47950K .......... .......... .......... .......... .......... 89% 24,6M 0s
     48000K .......... .......... .......... .......... .......... 89% 16,6M 0s
     48050K .......... .......... .......... .......... .......... 89%  286M 0s
     48100K .......... .......... .......... .......... .......... 89% 63,4M 0s
     48150K .......... .......... .......... .......... .......... 89% 19,3M 0s
     48200K .......... .......... .......... .......... .......... 89% 21,8M 0s
     48250K .......... .......... .......... .......... .......... 89% 57,7M 0s
     48300K .......... .......... .......... .......... .......... 89% 14,3M 0s
     48350K .......... .......... .......... .......... .......... 89% 98,2M 0s
     48400K .......... .......... .......... .......... .......... 89% 11,7M 0s
     48450K .......... .......... .......... .......... .......... 89%  151M 0s
     48500K .......... .......... .......... .......... .......... 90% 32,9M 0s
     48550K .......... .......... .......... .......... .......... 90%  159M 0s
     48600K .......... .......... .......... .......... .......... 90% 18,9M 0s
     48650K .......... .......... .......... .......... .......... 90%  118M 0s
     48700K .......... .......... .......... .......... .......... 90% 18,7M 0s
     48750K .......... .......... .......... .......... .......... 90% 60,3M 0s
     48800K .......... .......... .......... .......... .......... 90% 19,8M 0s
     48850K .......... .......... .......... .......... .......... 90% 55,0M 0s
     48900K .......... .......... .......... .......... .......... 90% 20,4M 0s
     48950K .......... .......... .......... .......... .......... 90%  128M 0s
     49000K .......... .......... .......... .......... .......... 90% 14,7M 0s
     49050K .......... .......... .......... .......... .......... 91%  107M 0s
     49100K .......... .......... .......... .......... .......... 91% 21,0M 0s
     49150K .......... .......... .......... .......... .......... 91%  103M 0s
     49200K .......... .......... .......... .......... .......... 91% 25,1M 0s
     49250K .......... .......... .......... .......... .......... 91%  130M 0s
     49300K .......... .......... .......... .......... .......... 91% 16,2M 0s
     49350K .......... .......... .......... .......... .......... 91% 99,2M 0s
     49400K .......... .......... .......... .......... .......... 91% 13,1M 0s
     49450K .......... .......... .......... .......... .......... 91% 38,3M 0s
     49500K .......... .......... .......... .......... .......... 91% 66,3M 0s
     49550K .......... .......... .......... .......... .......... 92% 16,6M 0s
     49600K .......... .......... .......... .......... .......... 92%  121M 0s
     49650K .......... .......... .......... .......... .......... 92% 25,9M 0s
     49700K .......... .......... .......... .......... .......... 92% 53,1M 0s
     49750K .......... .......... .......... .......... .......... 92% 20,4M 0s
     49800K .......... .......... .......... .......... .......... 92% 65,1M 0s
     49850K .......... .......... .......... .......... .......... 92% 17,1M 0s
     49900K .......... .......... .......... .......... .......... 92%  229M 0s
     49950K .......... .......... .......... .......... .......... 92% 20,1M 0s
     50000K .......... .......... .......... .......... .......... 92% 79,5M 0s
     50050K .......... .......... .......... .......... .......... 92% 14,0M 0s
     50100K .......... .......... .......... .......... .......... 93% 16,5M 0s
     50150K .......... .......... .......... .......... .......... 93% 62,6M 0s
     50200K .......... .......... .......... .......... .......... 93% 29,8M 0s
     50250K .......... .......... .......... .......... .......... 93% 53,4M 0s
     50300K .......... .......... .......... .......... .......... 93% 19,2M 0s
     50350K .......... .......... .......... .......... .......... 93% 80,8M 0s
     50400K .......... .......... .......... .......... .......... 93% 18,2M 0s
     50450K .......... .......... .......... .......... .......... 93% 37,3M 0s
     50500K .......... .......... .......... .......... .......... 93% 25,5M 0s
     50550K .......... .......... .......... .......... .......... 93%  159M 0s
     50600K .......... .......... .......... .......... .......... 93% 18,6M 0s
     50650K .......... .......... .......... .......... .......... 94%  156M 0s
     50700K .......... .......... .......... .......... .......... 94% 20,1M 0s
     50750K .......... .......... .......... .......... .......... 94% 19,6M 0s
     50800K .......... .......... .......... .......... .......... 94%  180M 0s
     50850K .......... .......... .......... .......... .......... 94%  193M 0s
     50900K .......... .......... .......... .......... .......... 94% 16,5M 0s
     50950K .......... .......... .......... .......... .......... 94% 13,9M 0s
     51000K .......... .......... .......... .......... .......... 94% 87,7M 0s
     51050K .......... .......... .......... .......... .......... 94% 34,2M 0s
     51100K .......... .......... .......... .......... .......... 94% 55,4M 0s
     51150K .......... .......... .......... .......... .......... 94% 17,1M 0s
     51200K .......... .......... .......... .......... .......... 95% 45,1M 0s
     51250K .......... .......... .......... .......... .......... 95% 17,1M 0s
     51300K .......... .......... .......... .......... .......... 95%  119M 0s
     51350K .......... .......... .......... .......... .......... 95% 26,2M 0s
     51400K .......... .......... .......... .......... .......... 95% 80,2M 0s
     51450K .......... .......... .......... .......... .......... 95% 18,6M 0s
     51500K .......... .......... .......... .......... .......... 95% 69,3M 0s
     51550K .......... .......... .......... .......... .......... 95% 19,2M 0s
     51600K .......... .......... .......... .......... .......... 95% 53,2M 0s
     51650K .......... .......... .......... .......... .......... 95% 25,9M 0s
     51700K .......... .......... .......... .......... .......... 95% 24,2M 0s
     51750K .......... .......... .......... .......... .......... 96% 53,8M 0s
     51800K .......... .......... .......... .......... .......... 96% 49,7M 0s
     51850K .......... .......... .......... .......... .......... 96% 23,2M 0s
     51900K .......... .......... .......... .......... .......... 96% 61,0M 0s
     51950K .......... .......... .......... .......... .......... 96% 15,9M 0s
     52000K .......... .......... .......... .......... .......... 96% 19,0M 0s
     52050K .......... .......... .......... .......... .......... 96% 69,9M 0s
     52100K .......... .......... .......... .......... .......... 96%  146M 0s
     52150K .......... .......... .......... .......... .......... 96% 16,1M 0s
     52200K .......... .......... .......... .......... .......... 96% 92,5M 0s
     52250K .......... .......... .......... .......... .......... 97% 22,3M 0s
     52300K .......... .......... .......... .......... .......... 97% 46,5M 0s
     52350K .......... .......... .......... .......... .......... 97% 18,3M 0s
     52400K .......... .......... .......... .......... .......... 97% 25,9M 0s
     52450K .......... .......... .......... .......... .......... 97% 69,9M 0s
     52500K .......... .......... .......... .......... .......... 97% 13,4M 0s
     52550K .......... .......... .......... .......... .......... 97% 86,4M 0s
     52600K .......... .......... .......... .......... .......... 97% 27,6M 0s
     52650K .......... .......... .......... .......... .......... 97% 84,5M 0s
     52700K .......... .......... .......... .......... .......... 97% 19,4M 0s
     52750K .......... .......... .......... .......... .......... 97%  104M 0s
     52800K .......... .......... .......... .......... .......... 98% 22,4M 0s
     52850K .......... .......... .......... .......... .......... 98% 74,7M 0s
     52900K .......... .......... .......... .......... .......... 98% 20,3M 0s
     52950K .......... .......... .......... .......... .......... 98% 66,6M 0s
     53000K .......... .......... .......... .......... .......... 98% 17,5M 0s
     53050K .......... .......... .......... .......... .......... 98%  149M 0s
     53100K .......... .......... .......... .......... .......... 98% 14,0M 0s
     53150K .......... .......... .......... .......... .......... 98% 97,7M 0s
     53200K .......... .......... .......... .......... .......... 98% 15,1M 0s
     53250K .......... .......... .......... .......... .......... 98%  226M 0s
     53300K .......... .......... .......... .......... .......... 98% 24,2M 0s
     53350K .......... .......... .......... .......... .......... 99% 12,8M 0s
     53400K .......... .......... .......... .......... .......... 99%  231M 0s
     53450K .......... .......... .......... .......... .......... 99% 23,0M 0s
     53500K .......... .......... .......... .......... .......... 99%  213M 0s
     53550K .......... .......... .......... .......... .......... 99% 16,5M 0s
     53600K .......

    Unzipping Dataset
    Archive:  cats-dogs-data.zip
    Removing .zip file


    ... .......... .......... .......... .......... 99% 47,1M 0s
     53650K .......... .......... .......... .......... .......... 99% 20,0M 0s
     53700K .......... .......... .......... .......... .......... 99% 73,2M 0s
     53750K .......... .......... .......... .......... .......... 99% 26,8M 0s
     53800K .......... .......... .......... .......... .......... 99% 56,3M 0s
     53850K .......... .......... .......... .......... .......... 99% 6,71M 0s
     53900K .........                                             100%  132M=1,9s
    
    2021-10-06 11:26:22 (27,8 MB/s) - ‘cats-dogs-data.zip’ saved [55203029/55203029]
    
    replace cats-dogs-data/.DS_Store? [y]es, [n]o, [A]ll, [N]one, [r]ename:  NULL
    (EOF or read error, treating as "[N]one" ...)


Here's the keras implementation for a great performance result:

```python
from tensorflow.keras.models import Sequential
from tensorflow.keras.layers import Flatten, Dense, GlobalAveragePooling2D
from tensorflow.keras.applications.resnet50 import ResNet50, preprocess_input
from tensorflow.keras.preprocessing.image import ImageDataGenerator
from tensorflow.keras.optimizers import Adam

TARGET_SHAPE = (224, 224, 3)
TRAIN_PATH = 'cats-dogs-data/train'
VALID_PATH = 'cats-dogs-data/valid'

datagen = ImageDataGenerator(preprocessing_function=preprocess_input)
train_gen = datagen.flow_from_directory(TRAIN_PATH, 
                                        target_size=TARGET_SHAPE[:2], 
                                        class_mode='sparse')
valid_gen = datagen.flow_from_directory(VALID_PATH, 
                                        target_size=TARGET_SHAPE[:2], 
                                        class_mode='sparse',
                                        shuffle=False)

base_model = ResNet50(include_top=False, input_shape=TARGET_SHAPE)

for layer in base_model.layers:
    layer.trainable=False
    
model = Sequential([base_model,
                    GlobalAveragePooling2D(),
                    Dense(1024, activation='relu'),
                    Dense(2, activation='softmax')])
```

    Found 2000 images belonging to 2 classes.
    Found 400 images belonging to 2 classes.


```python
model.compile(optimizer=Adam(learning_rate=1e-4), loss='sparse_categorical_crossentropy', metrics=['accuracy'])
```

```python
model.fit(train_gen, epochs=3, validation_data=valid_gen)
```

    Epoch 1/3


    2021-10-06 11:29:26.088412: I tensorflow/core/grappler/optimizers/custom_graph_optimizer_registry.cc:112] Plugin optimizer for device_type GPU is enabled.
    2021-10-06 11:29:26.256 python[95381:1139332] -[MPSGraph adamUpdateWithLearningRateTensor:beta1Tensor:beta2Tensor:epsilonTensor:beta1PowerTensor:beta2PowerTensor:valuesTensor:momentumTensor:velocityTensor:maximumVelocityTensor:gradientTensor:name:]: unrecognized selector sent to instance 0x2a93793b0


By looking at `val_accuracy` we can confirm the results seems great. Let's also plot some other metrics:

```python
from sklearn.metrics import roc_auc_score, classification_report, confusion_matrix
import seaborn as sns
y_pred = model.predict(valid_gen)
y_test = valid_gen.classes
roc = roc_auc_score(y_test, y_pred[:, 1])
print("ROC AUC Score", roc)
```

    ROC AUC Score 0.9989


```python
cm=confusion_matrix(y_test, y_pred.argmax(axis=1))
sns.heatmap(cm, annot=True, fmt='g')
```




    <matplotlib.axes._subplots.AxesSubplot at 0x7fcc41c57090>




![png](docs/images/output_21_1.png)


Although we got an almost perfect clssifier, there are multiple details that someone who is coming from sklearn has to be careful when using Keras, for example:
- Correctly setup the Data Generator
- Fine tune the learning rate
- Adjust the batch size

Now let's replicate the same results using `deepfeatx`:

```python
from deepfeatx.image import ImageFeatureExtractor
from sklearn.linear_model import LogisticRegression

TRAIN_PATH = 'cats-dogs-data/train'
VALID_PATH = 'cats-dogs-data/valid'

fe = ImageFeatureExtractor()

train=fe.extract_features_from_directory(TRAIN_PATH, 
                                         classes_as_folders=True,
                                         export_class_names=True)
test=fe.extract_features_from_directory(VALID_PATH, 
                                         classes_as_folders=True,
                                         export_class_names=True)

X_train, y_train = train.drop(['filepaths', 'classes'], axis=1), train['classes']
X_test, y_test = test.drop(['filepaths', 'classes'], axis=1), test['classes']
lr = LogisticRegression().fit(X_train, y_train)
```

    Found 2000 images belonging to 2 classes.


    2021-10-06 11:27:40.528937: I tensorflow/core/grappler/optimizers/custom_graph_optimizer_registry.cc:112] Plugin optimizer for device_type GPU is enabled.


    63/63 [==============================] - 22s 350ms/step
    Found 400 images belonging to 2 classes.
    13/13 [==============================] - 4s 351ms/step


    /Users/wittmann/miniforge3/envs/mlp/lib/python3.8/site-packages/sklearn/linear_model/_logistic.py:814: ConvergenceWarning: lbfgs failed to converge (status=1):
    STOP: TOTAL NO. of ITERATIONS REACHED LIMIT.
    
    Increase the number of iterations (max_iter) or scale the data as shown in:
        https://scikit-learn.org/stable/modules/preprocessing.html
    Please also refer to the documentation for alternative solver options:
        https://scikit-learn.org/stable/modules/linear_model.html#logistic-regression
      n_iter_i = _check_optimize_result(


```python
roc_auc_score(y_test, lr.predict_proba(X_test)[:, 1])
```




    0.9996



```python
import seaborn as sns
cm=confusion_matrix(y_test, lr.predict(X_test))
sns.heatmap(cm, annot=True, fmt='g')
```




    <AxesSubplot:>




![png](docs/images/output_25_1.png)


Even though the code is smaller, is still as powerful as the keras code and also very flexible. The most important part is the feature extraction, which `deepfeatx` take care for us, and the rest can be performed as any other ML problem.
