"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ShellScriptAction = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
/**
 * Validate a revision using shell commands
 */
class ShellScriptAction {
    constructor(props) {
        var _a, _b;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: props.actionName,
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [],
            outputs: [],
        };
        if (Object.keys((_a = props.useOutputs) !== null && _a !== void 0 ? _a : {}).length + ((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).length === 0) {
            throw new Error('You must supply either \'useOutputs\' or \'additionalArtifacts\', since a CodeBuild Action must always have at least one input artifact.');
        }
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d;
        const inputs = new Array();
        inputs.push(...(_a = this.props.additionalArtifacts) !== null && _a !== void 0 ? _a : []);
        const envVarCommands = new Array();
        const bashOptions = (_b = this.props.bashOptions) !== null && _b !== void 0 ? _b : '-eu';
        if (bashOptions) {
            envVarCommands.push(`set ${bashOptions}`);
        }
        for (const [varName, output] of Object.entries((_c = this.props.useOutputs) !== null && _c !== void 0 ? _c : {})) {
            const outputArtifact = output.artifactFile;
            // Add the artifact to the list of inputs, if it's not in there already. Determine
            // the location where CodeBuild is going to stick it based on whether it's the first (primary)
            // input or an 'extra input', then parse.
            let artifactIndex = inputs.findIndex(a => a.artifactName === outputArtifact.artifact.artifactName);
            if (artifactIndex === -1) {
                artifactIndex = inputs.push(outputArtifact.artifact) - 1;
            }
            const dirEnv = artifactIndex === 0 ? 'CODEBUILD_SRC_DIR' : `CODEBUILD_SRC_DIR_${outputArtifact.artifact.artifactName}`;
            envVarCommands.push(`export ${varName}="$(node -pe 'require(process.env.${dirEnv} + "/${outputArtifact.fileName}")["${output.outputName}"]')"`);
        }
        this._project = new codebuild.PipelineProject(scope, 'Project', {
            environment: { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0 },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    build: {
                        commands: [
                            ...envVarCommands,
                            ...this.props.commands,
                        ],
                    },
                },
            }),
        });
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.props.actionName,
            input: inputs[0],
            extraInputs: inputs.slice(1),
            runOrder: (_d = this.props.runOrder) !== null && _d !== void 0 ? _d : 100,
            project: this._project,
        });
        // Replace the placeholder actionProperties at the last minute
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
    }
    /**
     * Project generated to run the shell script in
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after ShellScriptAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.ShellScriptAction = ShellScriptAction;
//# sourceMappingURL=data:application/json;base64,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