import * as codebuild from "../../../aws-codebuild";
import * as codepipeline from "../../../aws-codepipeline";
import * as events from "../../../aws-events";
import { PolicyStatement } from "../../../aws-iam";
import { Construct } from "../../../core";
/**
 * Configuration options for a SimpleSynth
 */
export interface SimpleSynthOptions {
    /**
     * The source artifact of the CodePipeline
     */
    readonly sourceArtifact: codepipeline.Artifact;
    /**
     * The artifact where the CloudAssembly should be emitted
     */
    readonly cloudAssemblyArtifact: codepipeline.Artifact;
    /**
     * Environment variables to send into build
     *
     * @default - No additional environment variables
     */
    readonly environmentVariables?: Record<string, codebuild.BuildEnvironmentVariable>;
    /**
     * Environment variables to copy over from parent env
     *
     * These are environment variables that are being used by the build.
     *
     * @default - No environment variables copied
     */
    readonly copyEnvironmentVariables?: string[];
    /**
     * Name of the build action
     *
     * @default 'Synth'
     */
    readonly actionName?: string;
    /**
     * Name of the CodeBuild project
     *
     * @default - Automatically generated
     */
    readonly projectName?: string;
    /**
     * Build environment to use for CodeBuild job
     *
     * @default BuildEnvironment.LinuxBuildImage.STANDARD_4_0
     */
    readonly environment?: codebuild.BuildEnvironment;
    /**
     * Directory inside the source where package.json and cdk.json are located
     *
     * @default - Repository root
     */
    readonly subdirectory?: string;
    /**
     * Produce additional output artifacts after the build based on the given directories
     *
     * Can be used to produce additional artifacts during the build step,
     * separate from the cloud assembly, which can be used further on in the
     * pipeline.
     *
     * Directories are evaluated with respect to `subdirectory`.
     *
     * @default - No additional artifacts generated
     */
    readonly additionalArtifacts?: AdditionalArtifact[];
    /**
     * Policy statements to add to role used during the synth
     *
     * Can be used to add acces to a CodeArtifact repository etc.
     *
     * @default - No policy statements added to CodeBuild Project Role
     */
    readonly rolePolicyStatements?: PolicyStatement[];
}
/**
 * Construction props for SimpleSynthAction
 */
export interface SimpleSynthActionProps extends SimpleSynthOptions {
    /**
     * The synth command
     */
    readonly synthCommand: string;
    /**
     * The install command
     *
     * @default - No install required
     */
    readonly installCommand?: string;
    /**
     * The build command
     *
     * By default, we assume NPM projects are either written in JavaScript or are
     * using `ts-node`, so don't need a build command.
     *
     * Otherwise, put the build command here, for example `npm run build`.
     *
     * @default - No build required
     */
    readonly buildCommand?: string;
}
/**
 * Specification of an additional artifact to generate
 */
export interface AdditionalArtifact {
    /**
     * Directory to be packaged
     */
    readonly directory: string;
    /**
     * Artifact to represent the build directory in the pipeline
     */
    readonly artifact: codepipeline.Artifact;
}
/**
 * A standard synth with a generated buildspec
 */
export declare class SimpleSynthAction implements codepipeline.IAction {
    private readonly props;
    /**
     * Create a standard NPM synth action
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     */
    static standardNpmSynth(options: StandardNpmSynthOptions): SimpleSynthAction;
    /**
     * Create a standard Yarn synth action
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     */
    static standardYarnSynth(options: StandardYarnSynthOptions): SimpleSynthAction;
    private _action?;
    private _actionProperties;
    constructor(props: SimpleSynthActionProps);
    /**
     * Exists to implement IAction
     */
    get actionProperties(): codepipeline.ActionProperties;
    /**
     * Exists to implement IAction
     */
    bind(scope: Construct, stage: codepipeline.IStage, options: codepipeline.ActionBindOptions): codepipeline.ActionConfig;
    /**
     * Exists to implement IAction
     */
    onStateChange(name: string, target?: events.IRuleTarget, options?: events.RuleProps): events.Rule;
}
/**
 * Options for a convention-based synth using NPM
 */
export interface StandardNpmSynthOptions extends SimpleSynthOptions {
    /**
     * The install command
     *
     * @default 'npm ci'
     */
    readonly installCommand?: string;
    /**
     * The build command
     *
     * By default, we assume NPM projects are either written in JavaScript or are
     * using `ts-node`, so don't need a build command.
     *
     * Otherwise, put the build command here, for example `npm run build`.
     *
     * @default - No build required
     */
    readonly buildCommand?: string;
    /**
     * The synth command
     *
     * @default 'npx cdk synth'
     */
    readonly synthCommand?: string;
}
/**
 * Options for a convention-based synth using Yarn
 */
export interface StandardYarnSynthOptions extends SimpleSynthOptions {
    /**
     * The install command
     *
     * @default 'yarn install --frozen-lockfile'
     */
    readonly installCommand?: string;
    /**
     * The build command
     *
     * By default, we assume NPM projects are either written in JavaScript or are
     * using `ts-node`, so don't need a build command.
     *
     * Otherwise, put the build command here, for example `npm run build`.
     *
     * @default - No build required
     */
    readonly buildCommand?: string;
    /**
     * The synth command
     *
     * @default 'npx cdk synth'
     */
    readonly synthCommand?: string;
}
