import * as codepipeline from "../../aws-codepipeline";
import { Construct, Stage } from "../../core";
import * as cxapi from "../../cx-api";
import { AssetType } from './actions';
/**
 * Construction properties for a CdkStage
 */
export interface CdkStageProps {
    /**
     * Name of the stage that should be created
     */
    readonly stageName: string;
    /**
     * The underlying Pipeline Stage associated with thisCdkStage
     */
    readonly pipelineStage: codepipeline.IStage;
    /**
     * The CodePipeline Artifact with the Cloud Assembly
     */
    readonly cloudAssemblyArtifact: codepipeline.Artifact;
    /**
     * Features the Stage needs from its environment
     */
    readonly host: IStageHost;
}
/**
 * Stage in a CdkPipeline
 *
 * You don't need to instantiate this class directly. Use
 * `cdkPipeline.addStage()` instead.
 */
export declare class CdkStage extends Construct {
    private _nextSequentialRunOrder;
    private _manualApprovalCounter;
    private readonly pipelineStage;
    private readonly cloudAssemblyArtifact;
    private readonly stacksToDeploy;
    private readonly stageName;
    private readonly host;
    private _prepared;
    constructor(scope: Construct, id: string, props: CdkStageProps);
    /**
     * Add all stacks in the application Stage to this stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplication(appStage: Stage, options?: AddStageOptions): void;
    /**
     * Add a deployment action based on a stack artifact
     */
    addStackArtifactDeployment(stackArtifact: cxapi.CloudFormationStackArtifact, options?: AddStackOptions): void;
    /**
     * Add a manual approval action
     *
     * If you need more flexibility than what this method offers,
     * use `addAction` with a `ManualApprovalAction`.
     */
    addManualApprovalAction(options?: AddManualApprovalOptions): void;
    /**
     * Add one or more CodePipeline Actions
     *
     * You need to make sure it is created with the right runOrder. Call `nextSequentialRunOrder()`
     * for every action to get actions to execute in sequence.
     */
    addActions(...actions: codepipeline.IAction[]): void;
    /**
     * Return the runOrder number necessary to run the next Action in sequence with the rest
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    nextSequentialRunOrder(count?: number): number;
    /**
     * Whether this Stage contains an action to deploy the given stack, identified by its artifact ID
     */
    deploysStack(artifactId: string): boolean;
    /**
     * Actually add all the DeployStack actions to the stage.
     *
     * We do this late because before we can render the actual DeployActions,
     * we need to know whether or not we need to capture the stack outputs.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    private prepareStage;
    /**
     * Advance the runorder counter so that the next sequential number is higher than the given one
     */
    private advanceRunOrderPast;
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    private simplifyStackName;
    /**
     * Make sure all assets depended on by this stack are published in this pipeline
     *
     * Taking care to exclude the stack template itself -- it is being published
     * as an asset because the CLI needs to know the asset publishing role when
     * pushing the template to S3, but in the case of CodePipeline we always
     * reference the template from the artifact bucket.
     *
     * (NOTE: this is only true for top-level stacks, not nested stacks. Nested
     * Stack templates are always published as assets).
     */
    private publishAssetDependencies;
}
/**
 * Additional options for adding a stack deployment
 */
export interface AddStackOptions {
    /**
     * Base runorder
     *
     * @default - Next sequential runorder
     */
    readonly runOrder?: number;
    /**
     * Base runorder
     *
     * @default - runOrder + 1
     */
    readonly executeRunOrder?: number;
}
/**
 * A single output of a Stack
 */
export declare class StackOutput {
    /**
     * The artifact and file the output is stored in
     */
    readonly artifactFile: codepipeline.ArtifactPath;
    /**
     * The name of the output in the JSON object in the file
     */
    readonly outputName: string;
    /**
     * Build a StackOutput from a known artifact and an output name
     */
    constructor(artifactFile: codepipeline.ArtifactPath, outputName: string);
}
/**
 * Features that the Stage needs from its environment
 */
export interface IStageHost {
    /**
     * Make sure all the assets from the given manifest are published
     */
    publishAsset(command: AssetPublishingCommand): void;
    /**
     * Return the Artifact the given stack has to emit its outputs into, if any
     */
    stackOutputArtifact(stackArtifactId: string): codepipeline.Artifact | undefined;
}
/**
 * Instructions to publish certain assets
 */
export interface AssetPublishingCommand {
    /**
     * Asset manifest path
     */
    readonly assetManifestPath: string;
    /**
     * Asset identifier
     */
    readonly assetId: string;
    /**
     * Asset selector to pass to `cdk-assets`.
     */
    readonly assetSelector: string;
    /**
     * Type of asset to publish
     */
    readonly assetType: AssetType;
}
/**
 * Options for adding an application stage to a pipeline
 */
export interface AddStageOptions {
    /**
     * Add manual approvals before executing change sets
     *
     * This gives humans the opportunity to confirm the change set looks alright
     * before deploying it.
     *
     * @default false
     */
    readonly manualApprovals?: boolean;
}
/**
 * Options for addManualApproval
 */
export interface AddManualApprovalOptions {
    /**
     * The name of the manual approval action
     *
     * @default 'ManualApproval' with a rolling counter
     */
    readonly actionName?: string;
    /**
     * The runOrder for this action
     *
     * @default - The next sequential runOrder
     */
    readonly runOrder?: number;
}
