import * as codepipeline from "../../aws-codepipeline";
import { CfnOutput, Construct, Stage } from "../../core";
import { AddStageOptions, CdkStage, StackOutput } from './stage';
/**
 * Properties for a CdkPipeline
 */
export interface CdkPipelineProps {
    /**
     * The CodePipeline action used to retrieve the CDK app's source
     */
    readonly sourceAction: codepipeline.IAction;
    /**
     * The CodePipeline action build and synthesis step of the CDK app
     */
    readonly synthAction: codepipeline.IAction;
    /**
     * The artifact you have defined to be the artifact to hold the cloudAssemblyArtifact for the synth action
     */
    readonly cloudAssemblyArtifact: codepipeline.Artifact;
    /**
     * Name of the pipeline
     *
     * @default - A name is automatically generated
     */
    readonly pipelineName?: string;
    /**
     * CDK CLI version to use in pipeline
     *
     * Some Actions in the pipeline will download and run a version of the CDK
     * CLI. Specify the version here.
     *
     * @default - Latest version
     */
    readonly cdkCliVersion?: string;
}
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
export declare class CdkPipeline extends Construct {
    private readonly _pipeline;
    private readonly _assets;
    private readonly _stages;
    private readonly _outputArtifacts;
    private readonly _cloudAssemblyArtifact;
    constructor(scope: Construct, id: string, props: CdkPipelineProps);
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage: Stage, options?: AddStageOptions): CdkStage;
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName: string): CdkStage;
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput: CfnOutput): StackOutput;
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    protected validate(): string[];
    /**
     * Return all StackDeployActions in an ordered list
     */
    private get stackActions();
    private validateDeployOrder;
    private validateRequestedOutputs;
}
