"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishAssetsAction = exports.AssetType = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Type of the asset that is being published
 */
var AssetType;
(function (AssetType) {
    /**
     * A file
     */
    AssetType["FILE"] = "file";
    /**
     * A Docker image
     */
    AssetType["DOCKER_IMAGE"] = "docker-image";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
/**
 * Action to publish an asset in the pipeline
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to prepare and publish the asset.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline when you add stacks that use assets.
 */
class PublishAssetsAction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.commands = new Array();
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const project = new codebuild.PipelineProject(this, 'Default', {
            projectName: this.props.projectName,
            environment: {
                buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                privileged: (props.assetType === AssetType.DOCKER_IMAGE) ? true : undefined,
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g cdk-assets${installSuffix}`,
                    },
                    build: {
                        commands: core_1.Lazy.listValue({ produce: () => this.commands }),
                    },
                },
            }),
            role: props.role,
        });
        const rolePattern = props.assetType === AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        this.action = new codepipeline_actions.CodeBuildAction({
            actionName: props.actionName,
            project,
            input: this.props.cloudAssemblyInput,
            role: props.role,
        });
    }
    /**
     * Add a single publishing command
     *
     * Manifest path should be relative to the root Cloud Assembly.
     */
    addPublishCommand(relativeManifestPath, assetSelector) {
        const command = `cdk-assets --path "${relativeManifestPath}" --verbose publish "${assetSelector}"`;
        if (!this.commands.includes(command)) {
            this.commands.push(command);
        }
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.PublishAssetsAction = PublishAssetsAction;
//# sourceMappingURL=data:application/json;base64,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