"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagManager = void 0;
const cfn_resource_1 = require("./cfn-resource");
/**
 * Standard tags are a list of { key, value } objects
 */
class StandardFormatter {
    parseTags(cfnPropertyTags, priority) {
        if (!Array.isArray(cfnPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        const tags = [];
        for (const tag of cfnPropertyTags) {
            if (tag.key === undefined || tag.value === undefined) {
                throw new Error(`Invalid tag input expected {key, value} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = [];
        for (const tag of tags) {
            cfnTags.push({
                key: tag.key,
                value: tag.value,
            });
        }
        return cfnTags.length === 0 ? undefined : cfnTags;
    }
}
/**
 * ASG tags are a list of { key, value, propagateAtLaunch } objects
 */
class AsgFormatter {
    parseTags(cfnPropertyTags, priority) {
        const tags = [];
        if (!Array.isArray(cfnPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value, propagateAtLaunch} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        for (const tag of cfnPropertyTags) {
            if (tag.key === undefined ||
                tag.value === undefined ||
                tag.propagateAtLaunch === undefined) {
                throw new Error(`Invalid tag input expected {key, value, propagateAtLaunch} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
                applyToLaunchedInstances: !!tag.propagateAtLaunch,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = [];
        for (const tag of tags) {
            cfnTags.push({
                key: tag.key,
                value: tag.value,
                propagateAtLaunch: tag.applyToLaunchedInstances !== false,
            });
        }
        return cfnTags.length === 0 ? undefined : cfnTags;
    }
}
/**
 * Some CloudFormation constructs use a { key: value } map for tags
 */
class MapFormatter {
    parseTags(cfnPropertyTags, priority) {
        const tags = [];
        if (Array.isArray(cfnPropertyTags) || typeof (cfnPropertyTags) !== 'object') {
            throw new Error(`Invalid tag input expected map of {key: value} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        for (const [key, value] of Object.entries(cfnPropertyTags)) {
            tags.push({
                key,
                value: `${value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = {};
        for (const tag of tags) {
            cfnTags[`${tag.key}`] = `${tag.value}`;
        }
        return Object.keys(cfnTags).length === 0 ? undefined : cfnTags;
    }
}
/**
 * StackTags are of the format { Key: key, Value: value }
 */
class KeyValueFormatter {
    parseTags(keyValueTags, priority) {
        const tags = [];
        for (const key in keyValueTags) {
            if (keyValueTags.hasOwnProperty(key)) {
                const value = keyValueTags[key];
                tags.push({
                    key,
                    value,
                    priority,
                });
            }
        }
        return tags;
    }
    formatTags(unformattedTags) {
        const tags = [];
        unformattedTags.forEach(tag => {
            tags.push({
                Key: tag.key,
                Value: tag.value,
            });
        });
        return tags;
    }
}
class NoFormat {
    parseTags(_cfnPropertyTags) {
        return [];
    }
    formatTags(_tags) {
        return undefined;
    }
}
const TAG_FORMATTERS = {
    [cfn_resource_1.TagType.AUTOSCALING_GROUP]: new AsgFormatter(),
    [cfn_resource_1.TagType.STANDARD]: new StandardFormatter(),
    [cfn_resource_1.TagType.MAP]: new MapFormatter(),
    [cfn_resource_1.TagType.KEY_VALUE]: new KeyValueFormatter(),
    [cfn_resource_1.TagType.NOT_TAGGABLE]: new NoFormat(),
};
/**
 * TagManager facilitates a common implementation of tagging for Constructs.
 */
class TagManager {
    constructor(tagType, resourceTypeName, tagStructure, options = {}) {
        this.tags = new Map();
        this.priorities = new Map();
        this.initialTagPriority = 50;
        this.resourceTypeName = resourceTypeName;
        this.tagFormatter = TAG_FORMATTERS[tagType];
        if (tagStructure !== undefined) {
            this._setTag(...this.tagFormatter.parseTags(tagStructure, this.initialTagPriority));
        }
        this.tagPropertyName = options.tagPropertyName || 'tags';
    }
    /**
     * Check whether the given construct is Taggable
     */
    static isTaggable(construct) {
        return construct.tags !== undefined;
    }
    /**
     * Adds the specified tag to the array of tags
     *
     */
    setTag(key, value, priority = 0, applyToLaunchedInstances = true) {
        // This method mostly exists because we don't want to expose the 'Tag' type used (it will be confusing
        // to users).
        this._setTag({ key, value, priority, applyToLaunchedInstances });
    }
    /**
     * Removes the specified tag from the array if it exists
     *
     * @param key The tag to remove
     * @param priority The priority of the remove operation
     */
    removeTag(key, priority) {
        if (priority >= (this.priorities.get(key) || 0)) {
            this.tags.delete(key);
            this.priorities.set(key, priority);
        }
    }
    /**
     * Renders tags into the proper format based on TagType
     */
    renderTags() {
        const sortedTags = Array.from(this.tags.values()).sort((a, b) => a.key.localeCompare(b.key));
        return this.tagFormatter.formatTags(sortedTags);
    }
    /**
     * Determine if the aspect applies here
     *
     * Looks at the include and exclude resourceTypeName arrays to determine if
     * the aspect applies here
     */
    applyTagAspectHere(include, exclude) {
        if (exclude && exclude.length > 0 && exclude.indexOf(this.resourceTypeName) !== -1) {
            return false;
        }
        if (include && include.length > 0 && include.indexOf(this.resourceTypeName) === -1) {
            return false;
        }
        return true;
    }
    /**
     * Returns true if there are any tags defined
     */
    hasTags() {
        return this.tags.size > 0;
    }
    _setTag(...tags) {
        for (const tag of tags) {
            if (tag.priority >= (this.priorities.get(tag.key) || 0)) {
                this.tags.set(tag.key, tag);
                this.priorities.set(tag.key, tag.priority);
            }
        }
    }
}
exports.TagManager = TagManager;
//# sourceMappingURL=data:application/json;base64,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