"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs-extra");
const assets_1 = require("./assets");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
const stage_1 = require("./stage");
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        const outdir = (_a = stage_1.Stage.of(this)) === null || _a === void 0 ? void 0 : _a.outdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        const hashType = determineHashType(props.assetHashType, props.assetHash);
        if (props.bundling) {
            // Determine the source hash in advance of bundling if the asset hash type
            // is SOURCE so that the bundler can opt to re-use its previous output.
            const sourceHash = hashType === assets_1.AssetHashType.SOURCE
                ? this.calculateHash(hashType, props.assetHash, props.bundling)
                : undefined;
            this.bundleDir = this.bundle(props.bundling, outdir, sourceHash);
            this.assetHash = sourceHash !== null && sourceHash !== void 0 ? sourceHash : this.calculateHash(hashType, props.assetHash, props.bundling);
            this.relativePath = renderAssetFilename(this.assetHash);
            this.stagedPath = this.relativePath;
        }
        else {
            this.assetHash = this.calculateHash(hashType, props.assetHash);
            const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
            if (stagingDisabled) {
                this.stagedPath = this.sourcePath;
            }
            else {
                this.relativePath = renderAssetFilename(this.assetHash, path.extname(this.sourcePath));
                this.stagedPath = this.relativePath;
            }
        }
        this.sourceHash = this.assetHash;
        this.stageAsset(outdir);
    }
    stageAsset(outdir) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(outdir, this.relativePath);
        // Staging the bundling asset.
        if (this.bundleDir) {
            const isAlreadyStaged = fs.existsSync(targetPath);
            if (isAlreadyStaged && path.resolve(this.bundleDir) !== path.resolve(targetPath)) {
                // When an identical asset is already staged and the bundler used an
                // intermediate bundling directory, we remove the extra directory.
                fs.removeSync(this.bundleDir);
            }
            else if (!isAlreadyStaged) {
                fs.renameSync(this.bundleDir, targetPath);
            }
            return;
        }
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    /**
     * Bundles an asset and provides the emitted asset's directory in return.
     *
     * @param options Bundling options
     * @param outdir Parent directory to create the bundle output directory in
     * @param sourceHash The asset source hash if known in advance. If this field
     * is provided, the bundler may opt to skip bundling, providing any already-
     * emitted bundle. If this field is not provided, the bundler uses an
     * intermediate directory in outdir.
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, outdir, sourceHash) {
        var _a, _b, _c;
        let bundleDir;
        if (sourceHash) {
            // When an asset hash is known in advance of bundling, the bundler outputs
            // directly to the assembly output directory.
            bundleDir = path.resolve(path.join(outdir, renderAssetFilename(sourceHash)));
            if (fs.existsSync(bundleDir)) {
                // Pre-existing bundle directory. The bundle has already been generated
                // once before, so we'll give the caller nothing.
                return bundleDir;
            }
            fs.ensureDirSync(bundleDir);
        }
        else {
            // When the asset hash isn't known in advance, bundler outputs to an
            // intermediate directory.
            // Create temp directory for bundling inside the temp staging directory
            bundleDir = path.resolve(fs.mkdtempSync(path.join(outdir, 'bundling-temp-')));
            // Chmod the bundleDir to full access.
            fs.chmodSync(bundleDir, 0o777);
        }
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_b = options.local) === null || _b === void 0 ? void 0 : _b.tryBundle(bundleDir, options);
            if (!localBundling) {
                options.image._run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_c = options.workingDirectory) !== null && _c !== void 0 ? _c : AssetStaging.BUNDLING_INPUT_DIR,
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
        return bundleDir;
    }
    calculateHash(hashType, assetHash, bundling) {
        if (hashType === assets_1.AssetHashType.CUSTOM && !assetHash) {
            throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
        }
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update(assetHash !== null && assetHash !== void 0 ? assetHash : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param assetHash Asset hash given in the construct props
 */
function determineHashType(assetHashType, assetHash) {
    if (assetHash) {
        if (assetHashType && assetHashType !== assets_1.AssetHashType.CUSTOM) {
            throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
        }
        return assets_1.AssetHashType.CUSTOM;
    }
    else if (assetHashType) {
        return assetHashType;
    }
    else {
        return assets_1.AssetHashType.SOURCE;
    }
}
//# sourceMappingURL=data:application/json;base64,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