"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wait = exports.WaitTime = void 0;
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Represents the Wait state which delays a state machine from continuing for a specified time
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-wait-state.html
 */
class WaitTime {
    constructor(json) {
        this.json = json;
    }
    /**
     * Wait a fixed amount of time.
     */
    static duration(duration) { return new WaitTime({ Seconds: duration.toSeconds() }); }
    /**
     * Wait until the given ISO8601 timestamp
     *
     * @example 2016-03-14T01:59:00Z
     */
    static timestamp(timestamp) { return new WaitTime({ Timestamp: timestamp }); }
    /**
     * Wait for a number of seconds stored in the state object.
     *
     * @example $.waitSeconds
     */
    static secondsPath(path) { return new WaitTime({ SecondsPath: path }); }
    /**
     * Wait until a timestamp found in the state object.
     *
     * @example $.waitTimestamp
     */
    static timestampPath(path) { return new WaitTime({ TimestampPath: path }); }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.WaitTime = WaitTime;
/**
 * Define a Wait state in the state machine
 *
 * A Wait state can be used to delay execution of the state machine for a while.
 */
class Wait extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.time = props.time;
        this.endStates = [this];
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.WAIT,
            Comment: this.comment,
            ...this.time._json,
            ...this.renderNextEnd(),
        };
    }
}
exports.Wait = Wait;
//# sourceMappingURL=data:application/json;base64,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