"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Choice = void 0;
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Choice in the state machine
 *
 * A choice state can be used to make decisions based on the execution
 * state.
 */
class Choice extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [];
    }
    /**
     * If the given condition matches, continue execution with the given state
     */
    when(condition, next) {
        super.addChoice(condition, next.startState);
        return this;
    }
    /**
     * If none of the given conditions match, continue execution with the given state
     *
     * If no conditions match and no otherwise() has been given, an execution
     * error will be raised.
     */
    otherwise(def) {
        super.makeDefault(def.startState);
        return this;
    }
    /**
     * Return a Chain that contains all reachable end states from this Choice
     *
     * Use this to combine all possible choice paths back.
     */
    afterwards(options = {}) {
        const endStates = state_1.State.filterNextables(state_1.State.findReachableEndStates(this, { includeErrorHandlers: options.includeErrorHandlers }));
        if (options.includeOtherwise && this.defaultChoice) {
            throw new Error(`'includeOtherwise' set but Choice state ${this.stateId} already has an 'otherwise' transition`);
        }
        if (options.includeOtherwise) {
            endStates.push(new DefaultAsNext(this));
        }
        return chain_1.Chain.custom(this, endStates, this);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.CHOICE,
            Comment: this.comment,
            ...this.renderInputOutput(),
            ...this.renderChoices(),
        };
    }
}
exports.Choice = Choice;
/**
 * Adapter to make the .otherwise() transition settable through .next()
 */
class DefaultAsNext {
    constructor(choice) {
        this.choice = choice;
    }
    next(state) {
        this.choice.otherwise(state);
        return chain_1.Chain.sequence(this.choice, state);
    }
}
//# sourceMappingURL=data:application/json;base64,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