"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrCreateCluster = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
const cluster_utils_1 = require("./private/cluster-utils");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @experimental
 */
class EmrCreateCluster extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.visibleToAllUsers = this.props.visibleToAllUsers !== undefined ? this.props.visibleToAllUsers : true;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this._autoScalingRole = this.props.autoScalingRole;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = (_a = this.props.serviceRole) !== null && _a !== void 0 ? _a : this.createServiceRole();
        this._clusterRole = (_b = this.props.clusterRole) !== null && _b !== void 0 ? _b : this.createClusterRole();
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole();
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        this.taskPolicies = this.createPolicyStatements(this._serviceRole, this._clusterRole, this._autoScalingRole);
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    /**
     * @internal
     */
    _renderTask() {
        var _a, _b, _c;
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Instances: cluster_utils_1.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(cluster_utils_1.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_a = this._autoScalingRole) === null || _a === void 0 ? void 0 : _a.roleName),
                BootstrapActions: cdk.listMapper(cluster_utils_1.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(cluster_utils_1.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: (_b = this.props.ebsRootVolumeSize) === null || _b === void 0 ? void 0 : _b.toGibibytes(),
                KerberosAttributes: this.props.kerberosAttributes ? cluster_utils_1.KerberosAttributesPropertyToJson(this.props.kerberosAttributes) : undefined,
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_c = this.props.scaleDownBehavior) === null || _c === void 0 ? void 0 : _c.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                ...(this.props.tags ? this.renderTags(this.props.tags) : undefined),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            }),
        };
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['elasticmapreduce:RunJobFlow', 'elasticmapreduce:DescribeCluster', 'elasticmapreduce:TerminateJobFlows'],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [serviceRole.roleArn, clusterRole.roleArn],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole() {
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole() {
        const role = new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole() {
        var _a;
        const role = new iam.Role(this, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')],
        });
        (_a = role.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('application-autoscaling.amazonaws.com')],
            actions: ['sts:AssumeRole'],
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
(function (EmrCreateCluster) {
    /**
     * Valid valus for the Cluster ScaleDownBehavior
     *
     * @experimental
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR blacklists and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     * @experimental
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     * @experimental
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Spot Timeout Actions
     *
     * @experimental
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**\
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * CloudWatch Alarm Comparison Operators
     *
     * @experimental
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     * @experimental
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     * @experimental
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * EC2 Instance Market
     *
     * @experimental
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     * @experimental
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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