"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildStartBuild = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Start a CodeBuild Build as a task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html
 */
class CodeBuildStartBuild extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, CodeBuildStartBuild.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskMetrics = {
            metricPrefixSingular: 'CodeBuildProject',
            metricPrefixPlural: 'CodeBuildProjects',
            metricDimensions: {
                ProjectArn: this.props.project.projectArn,
            },
        };
        this.taskPolicies = this.configurePolicyStatements();
    }
    configurePolicyStatements() {
        let policyStatements = [
            new iam.PolicyStatement({
                resources: [this.props.project.projectArn],
                actions: [
                    'codebuild:StartBuild',
                    'codebuild:StopBuild',
                ],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    cdk.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventForCodeBuildStartBuildRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Provides the CodeBuild StartBuild service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('codebuild', 'startBuild', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ProjectName: this.props.project.projectName,
                EnvironmentVariablesOverride: this.props.environmentVariablesOverride
                    ? this.serializeEnvVariables(this.props.environmentVariablesOverride)
                    : undefined,
            }),
        };
    }
    serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            Name: name,
            Type: environmentVariables[name].type || codebuild.BuildEnvironmentVariableType.PLAINTEXT,
            Value: environmentVariables[name].value,
        }));
    }
}
exports.CodeBuildStartBuild = CodeBuildStartBuild;
CodeBuildStartBuild.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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