"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TopicBase = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is imported (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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