"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UrlSubscription = void 0;
const sns = require("../../aws-sns"); // Automatically re-written from '@aws-cdk/aws-sns'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Use a URL as a subscription target
 *
 * The message will be POSTed to the given URL.
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-http-https-endpoint-as-subscriber.html
 */
class UrlSubscription {
    constructor(url, props = {}) {
        this.url = url;
        this.props = props;
        this.unresolvedUrl = core_1.Token.isUnresolved(url);
        if (!this.unresolvedUrl && !url.startsWith('http://') && !url.startsWith('https://')) {
            throw new Error('URL must start with either http:// or https://');
        }
        if (this.unresolvedUrl && props.protocol === undefined) {
            throw new Error('Must provide protocol if url is unresolved');
        }
        if (this.unresolvedUrl) {
            this.protocol = props.protocol;
        }
        else {
            this.protocol = this.url.startsWith('https:') ? sns.SubscriptionProtocol.HTTPS : sns.SubscriptionProtocol.HTTP;
        }
    }
    /**
     * Returns a configuration for a URL to subscribe to an SNS topic
     */
    bind(_topic) {
        return {
            subscriberId: this.url,
            endpoint: this.url,
            protocol: this.protocol,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
}
exports.UrlSubscription = UrlSubscription;
//# sourceMappingURL=data:application/json;base64,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