"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(_) {
        let engine;
        if (this.dbCluster && this.dbInstance) {
            throw new Error('Proxy cannot target both database cluster and database instance.');
        }
        else if (this.dbCluster) {
            engine = this.dbCluster.node.defaultChild.engine;
        }
        else if (this.dbInstance) {
            engine = this.dbInstance.node.defaultChild.engine;
        }
        let engineFamily;
        switch (engine) {
            case 'aurora':
            case 'aurora-mysql':
            case 'mysql':
                engineFamily = 'MYSQL';
                break;
            case 'aurora-postgresql':
            case 'postgres':
                engineFamily = 'POSTGRESQL';
                break;
            default:
                throw new Error(`Unsupported engine type - ${engine}`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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