"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceEngine = exports.SqlServerEngineVersion = exports.OracleEngineVersion = exports.OracleLegacyEngineVersion = exports.PostgresEngineVersion = exports.MysqlEngineVersion = exports.MariaDbEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
class InstanceEngineBase {
    constructor(props) {
        var _a;
        this.engineType = props.engineType;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.engineVersion = props.version;
        this.parameterGroupFamily = (_a = props.parameterGroupFamily) !== null && _a !== void 0 ? _a : (this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined);
    }
    bindToInstance(_scope, options) {
        if (options.timezone) {
            throw new Error(`timezone property can be configured only for Microsoft SQL Server, not ${this.engineType}`);
        }
        return {};
    }
}
/**
 * The versions for the MariaDB instance engines
 * (those returned by {@link DatabaseInstanceEngine.mariaDb}).
 */
class MariaDbEngineVersion {
    constructor(mariaDbFullVersion, mariaDbMajorVersion) {
        this.mariaDbFullVersion = mariaDbFullVersion;
        this.mariaDbMajorVersion = mariaDbMajorVersion;
    }
    /**
     * Create a new MariaDbEngineVersion with an arbitrary version.
     *
     * @param mariaDbFullVersion the full version string,
     *   for example "10.5.28"
     * @param mariaDbMajorVersion the major version of the engine,
     *   for example "10.5"
     */
    static of(mariaDbFullVersion, mariaDbMajorVersion) {
        return new MariaDbEngineVersion(mariaDbFullVersion, mariaDbMajorVersion);
    }
}
exports.MariaDbEngineVersion = MariaDbEngineVersion;
/** Version "10.0" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_0 = MariaDbEngineVersion.of('10.0', '10.0');
/** Version "10.0.17". */
MariaDbEngineVersion.VER_10_0_17 = MariaDbEngineVersion.of('10.0.17', '10.0');
/** Version "10.0.24". */
MariaDbEngineVersion.VER_10_0_24 = MariaDbEngineVersion.of('10.0.24', '10.0');
/** Version "10.0.28". */
MariaDbEngineVersion.VER_10_0_28 = MariaDbEngineVersion.of('10.0.28', '10.0');
/** Version "10.0.31". */
MariaDbEngineVersion.VER_10_0_31 = MariaDbEngineVersion.of('10.0.31', '10.0');
/** Version "10.0.32". */
MariaDbEngineVersion.VER_10_0_32 = MariaDbEngineVersion.of('10.0.32', '10.0');
/** Version "10.0.34". */
MariaDbEngineVersion.VER_10_0_34 = MariaDbEngineVersion.of('10.0.34', '10.0');
/** Version "10.0.35". */
MariaDbEngineVersion.VER_10_0_35 = MariaDbEngineVersion.of('10.0.35', '10.0');
/** Version "10.1" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_1 = MariaDbEngineVersion.of('10.1', '10.1');
/** Version "10.1.14". */
MariaDbEngineVersion.VER_10_1_14 = MariaDbEngineVersion.of('10.1.14', '10.1');
/** Version "10.1.19". */
MariaDbEngineVersion.VER_10_1_19 = MariaDbEngineVersion.of('10.1.19', '10.1');
/** Version "10.1.23". */
MariaDbEngineVersion.VER_10_1_23 = MariaDbEngineVersion.of('10.1.23', '10.1');
/** Version "10.1.26". */
MariaDbEngineVersion.VER_10_1_26 = MariaDbEngineVersion.of('10.1.26', '10.1');
/** Version "10.1.31". */
MariaDbEngineVersion.VER_10_1_31 = MariaDbEngineVersion.of('10.1.31', '10.1');
/** Version "10.1.34". */
MariaDbEngineVersion.VER_10_1_34 = MariaDbEngineVersion.of('10.1.34', '10.1');
/** Version "10.2" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_2 = MariaDbEngineVersion.of('10.2', '10.2');
/** Version "10.2.11". */
MariaDbEngineVersion.VER_10_2_11 = MariaDbEngineVersion.of('10.2.11', '10.2');
/** Version "10.2.12". */
MariaDbEngineVersion.VER_10_2_12 = MariaDbEngineVersion.of('10.2.12', '10.2');
/** Version "10.2.15". */
MariaDbEngineVersion.VER_10_2_15 = MariaDbEngineVersion.of('10.2.15', '10.2');
/** Version "10.2.21". */
MariaDbEngineVersion.VER_10_2_21 = MariaDbEngineVersion.of('10.2.21', '10.2');
/** Version "10.3" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_3 = MariaDbEngineVersion.of('10.3', '10.3');
/** Version "10.3.8". */
MariaDbEngineVersion.VER_10_3_8 = MariaDbEngineVersion.of('10.3.8', '10.3');
/** Version "10.3.13". */
MariaDbEngineVersion.VER_10_3_13 = MariaDbEngineVersion.of('10.3.13', '10.3');
/** Version "10.3.20". */
MariaDbEngineVersion.VER_10_3_20 = MariaDbEngineVersion.of('10.3.20', '10.3');
/** Version "10.3.23". */
MariaDbEngineVersion.VER_10_3_23 = MariaDbEngineVersion.of('10.3.23', '10.3');
/** Version "10.4" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_4 = MariaDbEngineVersion.of('10.4', '10.4');
/** Version "10.4.8". */
MariaDbEngineVersion.VER_10_4_8 = MariaDbEngineVersion.of('10.4.8', '10.4');
/** Version "10.4.13". */
MariaDbEngineVersion.VER_10_4_13 = MariaDbEngineVersion.of('10.4.13', '10.4');
class MariaDbInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mariadb',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mariaDbFullVersion,
                    majorVersion: version.mariaDbMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the MySQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.mysql}).
 */
class MysqlEngineVersion {
    constructor(mysqlFullVersion, mysqlMajorVersion) {
        this.mysqlFullVersion = mysqlFullVersion;
        this.mysqlMajorVersion = mysqlMajorVersion;
    }
    /**
     * Create a new MysqlEngineVersion with an arbitrary version.
     *
     * @param mysqlFullVersion the full version string,
     *   for example "8.1.43"
     * @param mysqlMajorVersion the major version of the engine,
     *   for example "8.1"
     */
    static of(mysqlFullVersion, mysqlMajorVersion) {
        return new MysqlEngineVersion(mysqlFullVersion, mysqlMajorVersion);
    }
}
exports.MysqlEngineVersion = MysqlEngineVersion;
/** Version "5.5" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_5 = MysqlEngineVersion.of('5.5', '5.5');
/** Version "5.5.46". */
MysqlEngineVersion.VER_5_5_46 = MysqlEngineVersion.of('5.5.46', '5.5');
/** Version "5.5.53". */
MysqlEngineVersion.VER_5_5_53 = MysqlEngineVersion.of('5.5.53', '5.5');
/** Version "5.5.57". */
MysqlEngineVersion.VER_5_5_57 = MysqlEngineVersion.of('5.5.57', '5.5');
/** Version "5.5.59". */
MysqlEngineVersion.VER_5_5_59 = MysqlEngineVersion.of('5.5.59', '5.5');
/** Version "5.5.61". */
MysqlEngineVersion.VER_5_5_61 = MysqlEngineVersion.of('5.5.61', '5.5');
/** Version "5.6" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_6 = MysqlEngineVersion.of('5.6', '5.6');
/** Version "5.6.34". */
MysqlEngineVersion.VER_5_6_34 = MysqlEngineVersion.of('5.6.34', '5.6');
/** Version "5.6.35". */
MysqlEngineVersion.VER_5_6_35 = MysqlEngineVersion.of('5.6.35', '5.6');
/** Version "5.6.37". */
MysqlEngineVersion.VER_5_6_37 = MysqlEngineVersion.of('5.6.37', '5.6');
/** Version "5.6.39". */
MysqlEngineVersion.VER_5_6_39 = MysqlEngineVersion.of('5.6.39', '5.6');
/** Version "5.6.40". */
MysqlEngineVersion.VER_5_6_40 = MysqlEngineVersion.of('5.6.40', '5.6');
/** Version "5.6.41". */
MysqlEngineVersion.VER_5_6_41 = MysqlEngineVersion.of('5.6.41', '5.6');
/** Version "5.6.43". */
MysqlEngineVersion.VER_5_6_43 = MysqlEngineVersion.of('5.6.43', '5.6');
/** Version "5.6.44". */
MysqlEngineVersion.VER_5_6_44 = MysqlEngineVersion.of('5.6.44', '5.6');
/** Version "5.6.46". */
MysqlEngineVersion.VER_5_6_46 = MysqlEngineVersion.of('5.6.46', '5.6');
/** Version "5.6.48". */
MysqlEngineVersion.VER_5_6_48 = MysqlEngineVersion.of('5.6.48', '5.6');
/** Version "5.7" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_7 = MysqlEngineVersion.of('5.7', '5.7');
/** Version "5.7.16". */
MysqlEngineVersion.VER_5_7_16 = MysqlEngineVersion.of('5.7.16', '5.7');
/** Version "5.7.17". */
MysqlEngineVersion.VER_5_7_17 = MysqlEngineVersion.of('5.7.17', '5.7');
/** Version "5.7.19". */
MysqlEngineVersion.VER_5_7_19 = MysqlEngineVersion.of('5.7.19', '5.7');
/** Version "5.7.21". */
MysqlEngineVersion.VER_5_7_21 = MysqlEngineVersion.of('5.7.21', '5.7');
/** Version "5.7.22". */
MysqlEngineVersion.VER_5_7_22 = MysqlEngineVersion.of('5.7.22', '5.7');
/** Version "5.7.23". */
MysqlEngineVersion.VER_5_7_23 = MysqlEngineVersion.of('5.7.23', '5.7');
/** Version "5.7.24". */
MysqlEngineVersion.VER_5_7_24 = MysqlEngineVersion.of('5.7.24', '5.7');
/** Version "5.7.25". */
MysqlEngineVersion.VER_5_7_25 = MysqlEngineVersion.of('5.7.25', '5.7');
/** Version "5.7.26". */
MysqlEngineVersion.VER_5_7_26 = MysqlEngineVersion.of('5.7.26', '5.7');
/** Version "5.7.28". */
MysqlEngineVersion.VER_5_7_28 = MysqlEngineVersion.of('5.7.28', '5.7');
/** Version "5.7.30". */
MysqlEngineVersion.VER_5_7_30 = MysqlEngineVersion.of('5.7.30', '5.7');
/** Version "8.0" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_8_0 = MysqlEngineVersion.of('8.0', '8.0');
/** Version "8.0.11". */
MysqlEngineVersion.VER_8_0_11 = MysqlEngineVersion.of('8.0.11', '8.0');
/** Version "8.0.13". */
MysqlEngineVersion.VER_8_0_13 = MysqlEngineVersion.of('8.0.13', '8.0');
/** Version "8.0.15". */
MysqlEngineVersion.VER_8_0_15 = MysqlEngineVersion.of('8.0.15', '8.0');
/** Version "8.0.16". */
MysqlEngineVersion.VER_8_0_16 = MysqlEngineVersion.of('8.0.16', '8.0');
/** Version "8.0.17". */
MysqlEngineVersion.VER_8_0_17 = MysqlEngineVersion.of('8.0.17', '8.0');
/** Version "8.0.19". */
MysqlEngineVersion.VER_8_0_19 = MysqlEngineVersion.of('8.0.19', '8.0');
class MySqlInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mysql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mysqlFullVersion,
                    majorVersion: version.mysqlMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the PostgreSQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.postgres}).
 */
class PostgresEngineVersion {
    constructor(postgresFullVersion, postgresMajorVersion) {
        this.postgresFullVersion = postgresFullVersion;
        this.postgresMajorVersion = postgresMajorVersion;
    }
    /**
     * Create a new PostgresEngineVersion with an arbitrary version.
     *
     * @param postgresFullVersion the full version string,
     *   for example "13.11"
     * @param postgresMajorVersion the major version of the engine,
     *   for example "13"
     */
    static of(postgresFullVersion, postgresMajorVersion) {
        return new PostgresEngineVersion(postgresFullVersion, postgresMajorVersion);
    }
}
exports.PostgresEngineVersion = PostgresEngineVersion;
/** Version "9.5" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_5 = PostgresEngineVersion.of('9.5', '9.5');
/** Version "9.5.2". */
PostgresEngineVersion.VER_9_5_2 = PostgresEngineVersion.of('9.5.2', '9.5');
/** Version "9.5.4". */
PostgresEngineVersion.VER_9_5_4 = PostgresEngineVersion.of('9.5.4', '9.5');
/** Version "9.5.6". */
PostgresEngineVersion.VER_9_5_6 = PostgresEngineVersion.of('9.5.6', '9.5');
/** Version "9.5.7". */
PostgresEngineVersion.VER_9_5_7 = PostgresEngineVersion.of('9.5.7', '9.5');
/** Version "9.5.9". */
PostgresEngineVersion.VER_9_5_9 = PostgresEngineVersion.of('9.5.9', '9.5');
/** Version "9.5.10". */
PostgresEngineVersion.VER_9_5_10 = PostgresEngineVersion.of('9.5.10', '9.5');
/** Version "9.5.12". */
PostgresEngineVersion.VER_9_5_12 = PostgresEngineVersion.of('9.5.12', '9.5');
/** Version "9.5.13". */
PostgresEngineVersion.VER_9_5_13 = PostgresEngineVersion.of('9.5.13', '9.5');
/** Version "9.5.14". */
PostgresEngineVersion.VER_9_5_14 = PostgresEngineVersion.of('9.5.14', '9.5');
/** Version "9.5.15". */
PostgresEngineVersion.VER_9_5_15 = PostgresEngineVersion.of('9.5.15', '9.5');
/** Version "9.5.16". */
PostgresEngineVersion.VER_9_5_16 = PostgresEngineVersion.of('9.5.16', '9.5');
/** Version "9.5.18". */
PostgresEngineVersion.VER_9_5_18 = PostgresEngineVersion.of('9.5.18', '9.5');
/** Version "9.5.19". */
PostgresEngineVersion.VER_9_5_19 = PostgresEngineVersion.of('9.5.19', '9.5');
/** Version "9.5.20". */
PostgresEngineVersion.VER_9_5_20 = PostgresEngineVersion.of('9.5.20', '9.5');
/** Version "9.5.21". */
PostgresEngineVersion.VER_9_5_21 = PostgresEngineVersion.of('9.5.21', '9.5');
/** Version "9.5.22". */
PostgresEngineVersion.VER_9_5_22 = PostgresEngineVersion.of('9.5.22', '9.5');
/** Version "9.6" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_6 = PostgresEngineVersion.of('9.6', '9.6');
/** Version "9.6.1". */
PostgresEngineVersion.VER_9_6_1 = PostgresEngineVersion.of('9.6.1', '9.6');
/** Version "9.6.2". */
PostgresEngineVersion.VER_9_6_2 = PostgresEngineVersion.of('9.6.2', '9.6');
/** Version "9.6.3". */
PostgresEngineVersion.VER_9_6_3 = PostgresEngineVersion.of('9.6.3', '9.6');
/** Version "9.6.5". */
PostgresEngineVersion.VER_9_6_5 = PostgresEngineVersion.of('9.6.5', '9.6');
/** Version "9.6.6". */
PostgresEngineVersion.VER_9_6_6 = PostgresEngineVersion.of('9.6.6', '9.6');
/** Version "9.6.8". */
PostgresEngineVersion.VER_9_6_8 = PostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
PostgresEngineVersion.VER_9_6_9 = PostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.10". */
PostgresEngineVersion.VER_9_6_10 = PostgresEngineVersion.of('9.6.10', '9.6');
/** Version "9.6.11". */
PostgresEngineVersion.VER_9_6_11 = PostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
PostgresEngineVersion.VER_9_6_12 = PostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.14". */
PostgresEngineVersion.VER_9_6_14 = PostgresEngineVersion.of('9.6.14', '9.6');
/** Version "9.6.15". */
PostgresEngineVersion.VER_9_6_15 = PostgresEngineVersion.of('9.6.15', '9.6');
/** Version "9.6.16". */
PostgresEngineVersion.VER_9_6_16 = PostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
PostgresEngineVersion.VER_9_6_17 = PostgresEngineVersion.of('9.6.17', '9.6');
/** Version "9.6.18". */
PostgresEngineVersion.VER_9_6_18 = PostgresEngineVersion.of('9.6.18', '9.6');
/** Version "10" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_10 = PostgresEngineVersion.of('10', '10');
/** Version "10.1". */
PostgresEngineVersion.VER_10_1 = PostgresEngineVersion.of('10.1', '10');
/** Version "10.3". */
PostgresEngineVersion.VER_10_3 = PostgresEngineVersion.of('10.3', '10');
/** Version "10.4". */
PostgresEngineVersion.VER_10_4 = PostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
PostgresEngineVersion.VER_10_5 = PostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
PostgresEngineVersion.VER_10_6 = PostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
PostgresEngineVersion.VER_10_7 = PostgresEngineVersion.of('10.7', '10');
/** Version "10.9". */
PostgresEngineVersion.VER_10_9 = PostgresEngineVersion.of('10.9', '10');
/** Version "10.10". */
PostgresEngineVersion.VER_10_10 = PostgresEngineVersion.of('10.10', '10');
/** Version "10.11". */
PostgresEngineVersion.VER_10_11 = PostgresEngineVersion.of('10.11', '10');
/** Version "10.12". */
PostgresEngineVersion.VER_10_12 = PostgresEngineVersion.of('10.12', '10');
/** Version "10.13". */
PostgresEngineVersion.VER_10_13 = PostgresEngineVersion.of('10.13', '10');
/** Version "11" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_11 = PostgresEngineVersion.of('11', '11');
/** Version "11.1". */
PostgresEngineVersion.VER_11_1 = PostgresEngineVersion.of('11.1', '11');
/** Version "11.2". */
PostgresEngineVersion.VER_11_2 = PostgresEngineVersion.of('11.2', '11');
/** Version "11.4". */
PostgresEngineVersion.VER_11_4 = PostgresEngineVersion.of('11.4', '11');
/** Version "11.5". */
PostgresEngineVersion.VER_11_5 = PostgresEngineVersion.of('11.5', '11');
/** Version "11.6". */
PostgresEngineVersion.VER_11_6 = PostgresEngineVersion.of('11.6', '11');
/** Version "11.7". */
PostgresEngineVersion.VER_11_7 = PostgresEngineVersion.of('11.7', '11');
/** Version "11.8". */
PostgresEngineVersion.VER_11_8 = PostgresEngineVersion.of('11.8', '11');
/** Version "12" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_12 = PostgresEngineVersion.of('12', '12');
/** Version "12.2". */
PostgresEngineVersion.VER_12_2 = PostgresEngineVersion.of('12.2', '12');
/** Version "12.3". */
PostgresEngineVersion.VER_12_3 = PostgresEngineVersion.of('12.3', '12');
/**
 * The instance engine for PostgreSQL.
 */
class PostgresInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'postgres',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.postgresFullVersion,
                    majorVersion: version.postgresMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the legacy Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe}
 * and {@link DatabaseInstanceEngine.oracleSe1}).
 * Note: RDS will stop allowing creating new databases with this version in August 2020.
 */
class OracleLegacyEngineVersion {
    constructor(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        this.oracleLegacyFullVersion = oracleLegacyFullVersion;
        this.oracleLegacyMajorVersion = oracleLegacyMajorVersion;
    }
    static of(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        return new OracleLegacyEngineVersion(oracleLegacyFullVersion, oracleLegacyMajorVersion);
    }
}
exports.OracleLegacyEngineVersion = OracleLegacyEngineVersion;
/** Version "11.2" (only a major version, without a specific minor version). */
OracleLegacyEngineVersion.VER_11_2 = OracleLegacyEngineVersion.of('11.2', '11.2');
/** Version "11.2.0.2.v2". */
OracleLegacyEngineVersion.VER_11_2_0_2_V2 = OracleLegacyEngineVersion.of('11.2.0.2.v2', '11.2');
/** Version "11.2.0.4.v1". */
OracleLegacyEngineVersion.VER_11_2_0_4_V1 = OracleLegacyEngineVersion.of('11.2.0.4.v1', '11.2');
/** Version "11.2.0.4.v3". */
OracleLegacyEngineVersion.VER_11_2_0_4_V3 = OracleLegacyEngineVersion.of('11.2.0.4.v3', '11.2');
/** Version "11.2.0.4.v4". */
OracleLegacyEngineVersion.VER_11_2_0_4_V4 = OracleLegacyEngineVersion.of('11.2.0.4.v4', '11.2');
/** Version "11.2.0.4.v5". */
OracleLegacyEngineVersion.VER_11_2_0_4_V5 = OracleLegacyEngineVersion.of('11.2.0.4.v5', '11.2');
/** Version "11.2.0.4.v6". */
OracleLegacyEngineVersion.VER_11_2_0_4_V6 = OracleLegacyEngineVersion.of('11.2.0.4.v6', '11.2');
/** Version "11.2.0.4.v7". */
OracleLegacyEngineVersion.VER_11_2_0_4_V7 = OracleLegacyEngineVersion.of('11.2.0.4.v7', '11.2');
/** Version "11.2.0.4.v8". */
OracleLegacyEngineVersion.VER_11_2_0_4_V8 = OracleLegacyEngineVersion.of('11.2.0.4.v8', '11.2');
/** Version "11.2.0.4.v9". */
OracleLegacyEngineVersion.VER_11_2_0_4_V9 = OracleLegacyEngineVersion.of('11.2.0.4.v9', '11.2');
/** Version "11.2.0.4.v10". */
OracleLegacyEngineVersion.VER_11_2_0_4_V10 = OracleLegacyEngineVersion.of('11.2.0.4.v10', '11.2');
/** Version "11.2.0.4.v11". */
OracleLegacyEngineVersion.VER_11_2_0_4_V11 = OracleLegacyEngineVersion.of('11.2.0.4.v11', '11.2');
/** Version "11.2.0.4.v12". */
OracleLegacyEngineVersion.VER_11_2_0_4_V12 = OracleLegacyEngineVersion.of('11.2.0.4.v12', '11.2');
/** Version "11.2.0.4.v13". */
OracleLegacyEngineVersion.VER_11_2_0_4_V13 = OracleLegacyEngineVersion.of('11.2.0.4.v13', '11.2');
/** Version "11.2.0.4.v14". */
OracleLegacyEngineVersion.VER_11_2_0_4_V14 = OracleLegacyEngineVersion.of('11.2.0.4.v14', '11.2');
/** Version "11.2.0.4.v15". */
OracleLegacyEngineVersion.VER_11_2_0_4_V15 = OracleLegacyEngineVersion.of('11.2.0.4.v15', '11.2');
/** Version "11.2.0.4.v16". */
OracleLegacyEngineVersion.VER_11_2_0_4_V16 = OracleLegacyEngineVersion.of('11.2.0.4.v16', '11.2');
/** Version "11.2.0.4.v17". */
OracleLegacyEngineVersion.VER_11_2_0_4_V17 = OracleLegacyEngineVersion.of('11.2.0.4.v17', '11.2');
/** Version "11.2.0.4.v18". */
OracleLegacyEngineVersion.VER_11_2_0_4_V18 = OracleLegacyEngineVersion.of('11.2.0.4.v18', '11.2');
/** Version "11.2.0.4.v19". */
OracleLegacyEngineVersion.VER_11_2_0_4_V19 = OracleLegacyEngineVersion.of('11.2.0.4.v19', '11.2');
/** Version "11.2.0.4.v20". */
OracleLegacyEngineVersion.VER_11_2_0_4_V20 = OracleLegacyEngineVersion.of('11.2.0.4.v20', '11.2');
/** Version "11.2.0.4.v21". */
OracleLegacyEngineVersion.VER_11_2_0_4_V21 = OracleLegacyEngineVersion.of('11.2.0.4.v21', '11.2');
/** Version "11.2.0.4.v22". */
OracleLegacyEngineVersion.VER_11_2_0_4_V22 = OracleLegacyEngineVersion.of('11.2.0.4.v22', '11.2');
/** Version "11.2.0.4.v23". */
OracleLegacyEngineVersion.VER_11_2_0_4_V23 = OracleLegacyEngineVersion.of('11.2.0.4.v23', '11.2');
/** Version "11.2.0.4.v24". */
OracleLegacyEngineVersion.VER_11_2_0_4_V24 = OracleLegacyEngineVersion.of('11.2.0.4.v24', '11.2');
/**
 * The versions for the Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe2} and
 * {@link DatabaseInstanceEngine.oracleEe}).
 */
class OracleEngineVersion {
    constructor(oracleFullVersion, oracleMajorVersion) {
        this.oracleFullVersion = oracleFullVersion;
        this.oracleMajorVersion = oracleMajorVersion;
    }
    /**
     * Creates a new OracleEngineVersion with an arbitrary version.
     *
     * @param oracleFullVersion the full version string,
     *   for example "19.0.0.0.ru-2019-10.rur-2019-10.r1"
     * @param oracleMajorVersion the major version of the engine,
     *   for example "19"
     */
    static of(oracleFullVersion, oracleMajorVersion) {
        return new OracleEngineVersion(oracleFullVersion, oracleMajorVersion);
    }
}
exports.OracleEngineVersion = OracleEngineVersion;
/** Version "12.1" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_1 = OracleEngineVersion.of('12.1', '12.1');
/** Version "12.1.0.2.v1". */
OracleEngineVersion.VER_12_1_0_2_V1 = OracleEngineVersion.of('12.1.0.2.v1', '12.1');
/** Version "12.1.0.2.v2". */
OracleEngineVersion.VER_12_1_0_2_V2 = OracleEngineVersion.of('12.1.0.2.v2', '12.1');
/** Version "12.1.0.2.v3". */
OracleEngineVersion.VER_12_1_0_2_V3 = OracleEngineVersion.of('12.1.0.2.v3', '12.1');
/** Version "12.1.0.2.v4". */
OracleEngineVersion.VER_12_1_0_2_V4 = OracleEngineVersion.of('12.1.0.2.v4', '12.1');
/** Version "12.1.0.2.v5". */
OracleEngineVersion.VER_12_1_0_2_V5 = OracleEngineVersion.of('12.1.0.2.v5', '12.1');
/** Version "12.1.0.2.v6". */
OracleEngineVersion.VER_12_1_0_2_V6 = OracleEngineVersion.of('12.1.0.2.v6', '12.1');
/** Version "12.1.0.2.v7". */
OracleEngineVersion.VER_12_1_0_2_V7 = OracleEngineVersion.of('12.1.0.2.v7', '12.1');
/** Version "12.1.0.2.v8". */
OracleEngineVersion.VER_12_1_0_2_V8 = OracleEngineVersion.of('12.1.0.2.v8', '12.1');
/** Version "12.1.0.2.v9". */
OracleEngineVersion.VER_12_1_0_2_V9 = OracleEngineVersion.of('12.1.0.2.v9', '12.1');
/** Version "12.1.0.2.v10". */
OracleEngineVersion.VER_12_1_0_2_V10 = OracleEngineVersion.of('12.1.0.2.v10', '12.1');
/** Version "12.1.0.2.v11". */
OracleEngineVersion.VER_12_1_0_2_V11 = OracleEngineVersion.of('12.1.0.2.v11', '12.1');
/** Version "12.1.0.2.v12". */
OracleEngineVersion.VER_12_1_0_2_V12 = OracleEngineVersion.of('12.1.0.2.v12', '12.1');
/** Version "12.1.0.2.v13". */
OracleEngineVersion.VER_12_1_0_2_V13 = OracleEngineVersion.of('12.1.0.2.v13', '12.1');
/** Version "12.1.0.2.v14". */
OracleEngineVersion.VER_12_1_0_2_V14 = OracleEngineVersion.of('12.1.0.2.v14', '12.1');
/** Version "12.1.0.2.v15". */
OracleEngineVersion.VER_12_1_0_2_V15 = OracleEngineVersion.of('12.1.0.2.v15', '12.1');
/** Version "12.1.0.2.v16". */
OracleEngineVersion.VER_12_1_0_2_V16 = OracleEngineVersion.of('12.1.0.2.v16', '12.1');
/** Version "12.1.0.2.v17". */
OracleEngineVersion.VER_12_1_0_2_V17 = OracleEngineVersion.of('12.1.0.2.v17', '12.1');
/** Version "12.1.0.2.v18". */
OracleEngineVersion.VER_12_1_0_2_V18 = OracleEngineVersion.of('12.1.0.2.v18', '12.1');
/** Version "12.1.0.2.v19". */
OracleEngineVersion.VER_12_1_0_2_V19 = OracleEngineVersion.of('12.1.0.2.v19', '12.1');
/** Version "12.1.0.2.v20". */
OracleEngineVersion.VER_12_1_0_2_V20 = OracleEngineVersion.of('12.1.0.2.v20', '12.1');
/** Version "12.2" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_2 = OracleEngineVersion.of('12.2', '12.2');
/** Version "12.2.0.1.ru-2018-10.rur-2018-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2018_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2018-10.rur-2018-10.r1', '12.2');
/** Version "12.2.0.1.ru-2019-01.rur-2019-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-01.rur-2019-01.r1', '12.2');
/** Version "12.2.0.1.ru-2019-04.rur-2019-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-04.rur-2019-04.r1', '12.2');
/** Version "12.2.0.1.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_07_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-07.rur-2019-07.r1', '12.2');
/** Version "12.2.0.1.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-10.rur-2019-10.r1', '12.2');
/** Version "12.2.0.1.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-01.rur-2020-01.r1', '12.2');
/** Version "12.2.0.1.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-04.rur-2020-04.r1', '12.2');
/** Version "18" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_18 = OracleEngineVersion.of('18', '18');
/** Version "18.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_07_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-07.rur-2019-07.r1', '18');
/** Version "18.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_10_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-10.rur-2019-10.r1', '18');
/** Version "18.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_01_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-01.rur-2020-01.r1', '18');
/** Version "18.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_04_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-04.rur-2020-04.r1', '18');
/** Version "19" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_19 = OracleEngineVersion.of('19', '19');
/** Version "19.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_07_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-07.rur-2019-07.r1', '19');
/** Version "19.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_10_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-10.rur-2019-10.r1', '19');
/** Version "19.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_01_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-01.rur-2020-01.r1', '19');
/** Version "19.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_04_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-04.rur-2020-04.r1', '19');
class OracleInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER,
            parameterGroupFamily: props.version ? `${props.engineType}-${props.version.majorVersion}` : undefined,
        });
    }
}
class OracleSeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe1InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se1',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe2InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se2',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
class OracleEeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-ee',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the SQL Server instance engines
 * (those returned by {@link DatabaseInstanceEngine.sqlServerSe},
 * {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb}
 * and {@link DatabaseInstanceEngine.sqlServerEe}).
 */
class SqlServerEngineVersion {
    constructor(sqlServerFullVersion, sqlServerMajorVersion) {
        this.sqlServerFullVersion = sqlServerFullVersion;
        this.sqlServerMajorVersion = sqlServerMajorVersion;
    }
    /**
     * Create a new SqlServerEngineVersion with an arbitrary version.
     *
     * @param sqlServerFullVersion the full version string,
     *   for example "15.00.3049.1.v1"
     * @param sqlServerMajorVersion the major version of the engine,
     *   for example "15.00"
     */
    static of(sqlServerFullVersion, sqlServerMajorVersion) {
        return new SqlServerEngineVersion(sqlServerFullVersion, sqlServerMajorVersion);
    }
}
exports.SqlServerEngineVersion = SqlServerEngineVersion;
/** Version "11.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_11 = SqlServerEngineVersion.of('11.00', '11.00');
/** Version "11.00.5058.0.v1". */
SqlServerEngineVersion.VER_11_00_5058_0_V1 = SqlServerEngineVersion.of('11.00.5058.0.v1', '11.00');
/** Version "11.00.6020.0.v1". */
SqlServerEngineVersion.VER_11_00_6020_0_V1 = SqlServerEngineVersion.of('11.00.6020.0.v1', '11.00');
/** Version "11.00.6594.0.v1". */
SqlServerEngineVersion.VER_11_00_6594_0_V1 = SqlServerEngineVersion.of('11.00.6594.0.v1', '11.00');
/** Version "11.00.7462.6.v1". */
SqlServerEngineVersion.VER_11_00_7462_6_V1 = SqlServerEngineVersion.of('11.00.7462.6.v1', '11.00');
/** Version "11.00.7493.4.v1". */
SqlServerEngineVersion.VER_11_00_7493_4_V1 = SqlServerEngineVersion.of('11.00.7493.4.v1', '11.00');
/** Version "12.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_12 = SqlServerEngineVersion.of('12.00', '12.00');
/** Version "12.00.5000.0.v1". */
SqlServerEngineVersion.VER_12_00_5000_0_V1 = SqlServerEngineVersion.of('12.00.5000.0.v1', '12.00');
/** Version "12.00.5546.0.v1". */
SqlServerEngineVersion.VER_12_00_5546_0_V1 = SqlServerEngineVersion.of('12.00.5546.0.v1', '12.00');
/** Version "12.00.5571.0.v1". */
SqlServerEngineVersion.VER_12_00_5571_0_V1 = SqlServerEngineVersion.of('12.00.5571.0.v1', '12.00');
/** Version "12.00.6293.0.v1". */
SqlServerEngineVersion.VER_12_00_6293_0_V1 = SqlServerEngineVersion.of('12.00.6293.0.v1', '12.00');
/** Version "12.00.6329.1.v1". */
SqlServerEngineVersion.VER_12_00_6329_1_V1 = SqlServerEngineVersion.of('12.00.6329.1.v1', '12.00');
/** Version "13.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_13 = SqlServerEngineVersion.of('13.00', '13.00');
/** Version "13.00.2164.0.v1". */
SqlServerEngineVersion.VER_13_00_2164_0_V1 = SqlServerEngineVersion.of('13.00.2164.0.v1', '13.00');
/** Version "13.00.4422.0.v1". */
SqlServerEngineVersion.VER_13_00_4422_0_V1 = SqlServerEngineVersion.of('13.00.4422.0.v1', '13.00');
/** Version "13.00.4451.0.v1". */
SqlServerEngineVersion.VER_13_00_4451_0_V1 = SqlServerEngineVersion.of('13.00.4451.0.v1', '13.00');
/** Version "13.00.4466.4.v1". */
SqlServerEngineVersion.VER_13_00_4466_4_V1 = SqlServerEngineVersion.of('13.00.4466.4.v1', '13.00');
/** Version "13.00.4522.0.v1". */
SqlServerEngineVersion.VER_13_00_4522_0_V1 = SqlServerEngineVersion.of('13.00.4522.0.v1', '13.00');
/** Version "13.00.5216.0.v1". */
SqlServerEngineVersion.VER_13_00_5216_0_V1 = SqlServerEngineVersion.of('13.00.5216.0.v1', '13.00');
/** Version "13.00.5292.0.v1". */
SqlServerEngineVersion.VER_13_00_5292_0_V1 = SqlServerEngineVersion.of('13.00.5292.0.v1', '13.00');
/** Version "13.00.5366.0.v1". */
SqlServerEngineVersion.VER_13_00_5366_0_V1 = SqlServerEngineVersion.of('13.00.5366.0.v1', '13.00');
/** Version "13.00.5426.0.v1". */
SqlServerEngineVersion.VER_13_00_5426_0_V1 = SqlServerEngineVersion.of('13.00.5426.0.v1', '13.00');
/** Version "13.00.5598.27.v1". */
SqlServerEngineVersion.VER_13_00_5598_27_V1 = SqlServerEngineVersion.of('13.00.5598.27.v1', '13.00');
/** Version "14.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_14 = SqlServerEngineVersion.of('14.00', '14.00');
/** Version "14.00.1000.169.v1". */
SqlServerEngineVersion.VER_14_00_1000_169_V1 = SqlServerEngineVersion.of('14.00.1000.169.v1', '14.00');
/** Version "14.00.3015.40.v1". */
SqlServerEngineVersion.VER_14_00_3015_40_V1 = SqlServerEngineVersion.of('14.00.3015.40.v1', '14.00');
/** Version "14.00.3035.2.v1". */
SqlServerEngineVersion.VER_14_00_3035_2_V1 = SqlServerEngineVersion.of('14.00.3035.2.v1', '14.00');
/** Version "14.00.3049.1.v1". */
SqlServerEngineVersion.VER_14_00_3049_1_V1 = SqlServerEngineVersion.of('14.00.3049.1.v1', '14.00');
/** Version "14.00.3192.2.v1". */
SqlServerEngineVersion.VER_14_00_3192_2_V1 = SqlServerEngineVersion.of('14.00.3192.2.v1', '14.00');
class SqlServerInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER,
            version: props.version
                ? {
                    fullVersion: props.version.sqlServerFullVersion,
                    majorVersion: props.version.sqlServerMajorVersion,
                }
                : undefined,
            parameterGroupFamily: props.version
                // for some reason, even though SQL Server major versions usually end in '.00',
                // the ParameterGroup family has to end in '.0'
                ? `${props.engineType}-${props.version.sqlServerMajorVersion.endsWith('.00')
                    ? props.version.sqlServerMajorVersion.slice(0, -1)
                    : props.version.sqlServerMajorVersion}`
                : undefined,
        });
    }
    bindToInstance(_scope, _options) {
        return {};
    }
}
class SqlServerSeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-se',
            version,
        });
    }
}
class SqlServerExInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ex',
            version,
        });
    }
}
class SqlServerWebInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-web',
            version,
        });
    }
}
class SqlServerEeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ee',
            version,
        });
    }
}
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine {
    /** Creates a new MariaDB instance engine. */
    static mariaDb(props) {
        return new MariaDbInstanceEngine(props.version);
    }
    /** Creates a new MySQL instance engine. */
    static mysql(props) {
        return new MySqlInstanceEngine(props.version);
    }
    /** Creates a new PostgreSQL instance engine. */
    static postgres(props) {
        return new PostgresInstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition instance engine. */
    static oracleSe(props) {
        return new OracleSeInstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe1(props) {
        return new OracleSe1InstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe2(props) {
        return new OracleSe2InstanceEngine(props.version);
    }
    /** Creates a new Oracle Enterprise Edition instance engine. */
    static oracleEe(props) {
        return new OracleEeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Standard Edition instance engine. */
    static sqlServerSe(props) {
        return new SqlServerSeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Express Edition instance engine. */
    static sqlServerEx(props) {
        return new SqlServerExInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Web Edition instance engine. */
    static sqlServerWeb(props) {
        return new SqlServerWebInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Enterprise Edition instance engine. */
    static sqlServerEe(props) {
        return new SqlServerEeInstanceEngine(props.version);
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/**
 * The unversioned 'mariadb' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mariaDb()} method
 */
DatabaseInstanceEngine.MARIADB = new MariaDbInstanceEngine();
/**
 * The unversioned 'mysql' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mysql()} method
 */
DatabaseInstanceEngine.MYSQL = new MySqlInstanceEngine();
/**
 * The unversioned 'oracle-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleEe()} method
 */
DatabaseInstanceEngine.ORACLE_EE = new OracleEeInstanceEngine();
/**
 * The unversioned 'oracle-se2' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe2()} method
 */
DatabaseInstanceEngine.ORACLE_SE2 = new OracleSe2InstanceEngine();
/**
 * The unversioned 'oracle-se1' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe1()} method
 */
DatabaseInstanceEngine.ORACLE_SE1 = new OracleSe1InstanceEngine();
/**
 * The unversioned 'oracle-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe()} method
 */
DatabaseInstanceEngine.ORACLE_SE = new OracleSeInstanceEngine();
/**
 * The unversioned 'postgres' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link postgres()} method
 */
DatabaseInstanceEngine.POSTGRES = new PostgresInstanceEngine();
/**
 * The unversioned 'sqlserver-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EE = new SqlServerEeInstanceEngine();
/**
 * The unversioned 'sqlserver-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerSe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_SE = new SqlServerSeInstanceEngine();
/**
 * The unversioned 'sqlserver-ex' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEx()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EX = new SqlServerExInstanceEngine();
/**
 * The unversioned 'sqlserver-web' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerWeb()} method
 */
DatabaseInstanceEngine.SQL_SERVER_WEB = new SqlServerWebInstanceEngine();
//# sourceMappingURL=data:application/json;base64,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