"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogRetention = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const log_group_1 = require("./log-group");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Custom resource provider
        const provider = this.ensureSingletonLogRetentionFunction(props);
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        const retryOptions = props.logRetentionRetryOptions;
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                SdkRetry: retryOptions ? {
                    maxRetries: retryOptions.maxRetries,
                    base: (_a = retryOptions.base) === null || _a === void 0 ? void 0 : _a.toMilliseconds(),
                } : undefined,
                RetentionInDays: props.retention === log_group_1.RetentionDays.INFINITE ? undefined : props.retention,
            },
        });
        const logGroupName = resource.getAtt('LogGroupName').toString();
        // Append ':*' at the end of the ARN to match with how CloudFormation does this for LogGroup ARNs
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values
        this.logGroupArn = cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            resourceName: `${logGroupName}:*`,
            sep: ':',
        });
    }
    /**
     * Helper method to ensure that only one instance of LogRetentionFunction resources are in the stack mimicking the
     * behaviour of @aws-cdk/aws-lambda's SingletonFunction to prevent circular dependencies
     */
    ensureSingletonLogRetentionFunction(props) {
        const functionLogicalId = 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8a';
        const existing = cdk.Stack.of(this).node.tryFindChild(functionLogicalId);
        if (existing) {
            return existing;
        }
        return new LogRetentionFunction(cdk.Stack.of(this), functionLogicalId, props);
    }
}
exports.LogRetention = LogRetention;
/**
 * Private provider Lambda function to support the log retention custom resource.
 */
class LogRetentionFunction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Code
        const asset = new s3_assets.Asset(this, 'Code', {
            path: path.join(__dirname, 'log-retention-provider'),
        });
        // Role
        const role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing its name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Lambda function
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'AWS::Lambda::Function',
            properties: {
                Handler: 'index.handler',
                Runtime: 'nodejs10.x',
                Code: {
                    S3Bucket: asset.s3BucketName,
                    S3Key: asset.s3ObjectKey,
                },
                Role: role.roleArn,
            },
        });
        this.functionArn = resource.getAtt('Arn');
        // Function dependencies
        role.node.children.forEach((child) => {
            if (cdk.CfnResource.isCfnResource(child)) {
                resource.addDependsOn(child);
            }
            if (cdk.Construct.isConstruct(child) && child.node.defaultChild && cdk.CfnResource.isCfnResource(child.node.defaultChild)) {
                resource.addDependsOn(child.node.defaultChild);
            }
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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