"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossAccountDestination = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.stringValue({
            produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,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