"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./util");
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static get runsLocally() {
        if (LocalBundler._runsLocally !== undefined) {
            return LocalBundler._runsLocally;
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel'), ['--version']);
            LocalBundler._runsLocally = /^2/.test(parcel.stdout.toString().trim()); // Cache result to avoid unnecessary spawns
            return LocalBundler._runsLocally;
        }
        catch (_a) {
            return false;
        }
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
            cwd: path.dirname(path.join(this.props.projectRoot, this.props.relativeEntryPath)),
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b;
        const image = props.buildImage
            ? cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_a = props.buildArgs) !== null && _a !== void 0 ? _a : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_b = props.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
                },
            })
            : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)).replace(/\\/g, '/'),
        };
    }
}
exports.DockerBundler = DockerBundler;
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.ts$/, '.js');
    const parcelCommand = chain([
        [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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