"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoEventSource = void 0;
const stream_1 = require("./stream");
/**
 * Use an Amazon DynamoDB stream as an event source for AWS Lambda.
 */
class DynamoEventSource extends stream_1.StreamEventSource {
    constructor(table, props) {
        super(props);
        this.table = table;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 1000)) {
            throw new Error(`Maximum batch size must be between 1 and 1000 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        if (!this.table.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.table.node.path}`);
        }
        const eventSourceMapping = target.addEventSourceMapping(`DynamoDBEventSource:${this.table.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.table.tableStreamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.table.grantStreamRead(target);
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('DynamoEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.DynamoEventSource = DynamoEventSource;
//# sourceMappingURL=data:application/json;base64,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