"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnknownPrincipal = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${this.resource.node.path}, resource imported without a role`);
    }
    addToPrincipalPolicy(statement) {
        const stack = core_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        this.resource.node.addWarning(`Add statement to this resource's role: ${repr}`);
        // Pretend we did the work. The human will do it for us, eventually.
        return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,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