"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ManagedPolicy = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * Managed policy
 *
 */
class ManagedPolicy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.managedPolicyName,
        });
        /**
         * The policy document.
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.description = props.description || '';
        this.path = props.path || '/';
        if (props.document) {
            this.document = props.document;
        }
        const resource = new iam_generated_1.CfnManagedPolicy(this, 'Resource', {
            policyDocument: this.document,
            managedPolicyName: this.physicalName,
            description: this.description,
            path: this.path,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
        // arn:aws:iam::123456789012:policy/teststack-CreateTestDBPolicy-16M23YE3CS700
        this.managedPolicyName = this.getResourceNameAttribute(core_1.Stack.of(this).parseArn(resource.ref, '/').resourceName);
        this.managedPolicyArn = this.getResourceArnAttribute(resource.ref, {
            region: '',
            service: 'iam',
            resource: 'policy',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a customer managed policy from the managedPolicyName.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     */
    static fromManagedPolicyName(scope, id, managedPolicyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = core_1.Stack.of(scope).formatArn({
                    service: 'iam',
                    region: '',
                    account: core_1.Stack.of(scope).account,
                    resource: 'policy',
                    resourceName: managedPolicyName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an external managed policy by ARN.
     *
     * For this managed policy, you only need to know the ARN to be able to use it.
     * This can be useful if you got the ARN from a CloudFormation Export.
     *
     * If the imported Managed Policy ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * managed policy has a `path` (like `arn:...:policy/AdminPolicy/AdminAllow`), the
     * `managedPolicyName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Managed Policy ARN should be supplied without the
     * `path` in order to resolve the correct managed policy resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param managedPolicyArn the ARN of the managed policy to import
     */
    static fromManagedPolicyArn(scope, id, managedPolicyArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = managedPolicyArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a managed policy from one of the policies that AWS manages.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * Some managed policy names start with "service-role/", some start with
     * "job-function/", and some don't start with anything. Do include the
     * prefix when constructing this object.
     */
    static fromAwsManagedPolicyName(managedPolicyName) {
        class AwsManagedPolicy {
            constructor() {
                this.managedPolicyArn = core_1.Lazy.stringValue({
                    produce(ctx) {
                        return core_1.Stack.of(ctx.scope).formatArn({
                            service: 'iam',
                            region: '',
                            account: 'aws',
                            resource: 'policy',
                            resourceName: managedPolicyName,
                        });
                    },
                });
            }
        }
        return new AwsManagedPolicy();
    }
    /**
     * Adds a statement to the policy document.
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
    }
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
    }
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
    }
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            result.push('Managed Policy is empty. You must add statements to the policy');
        }
        result.push(...this.document.validateForIdentityPolicy());
        return result;
    }
}
exports.ManagedPolicy = ManagedPolicy;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFuYWdlZC1wb2xpY3kuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtYW5hZ2VkLXBvbGljeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBK0UsQ0FBQyxnREFBZ0Q7QUFFaEksbURBQW1EO0FBQ25ELHVEQUFtRDtBQUluRCxpQ0FBMEM7QUErRTFDOzs7R0FHRztBQUNILE1BQWEsYUFBYyxTQUFRLGVBQVE7SUFvR3ZDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBNEIsRUFBRTtRQUNwRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNiLFlBQVksRUFBRSxLQUFLLENBQUMsaUJBQWlCO1NBQ3hDLENBQUMsQ0FBQztRQTVCUDs7V0FFRztRQUNhLGFBQVEsR0FBRyxJQUFJLGdDQUFjLEVBQUUsQ0FBQztRQW1CL0IsVUFBSyxHQUFHLElBQUksS0FBSyxFQUFTLENBQUM7UUFDM0IsVUFBSyxHQUFHLElBQUksS0FBSyxFQUFTLENBQUM7UUFDM0IsV0FBTSxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFLMUMsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLEVBQUUsQ0FBQztRQUMzQyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksR0FBRyxDQUFDO1FBQzlCLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNoQixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7U0FDbEM7UUFDRCxNQUFNLFFBQVEsR0FBRyxJQUFJLGdDQUFnQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDcEQsY0FBYyxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQzdCLGlCQUFpQixFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ3BDLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztZQUM3QixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7WUFDZixLQUFLLEVBQUUsdUJBQWdCLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDOUQsS0FBSyxFQUFFLHVCQUFnQixDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzlELE1BQU0sRUFBRSx1QkFBZ0IsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUNwRSxDQUFDLENBQUM7UUFDSCxJQUFJLEtBQUssQ0FBQyxLQUFLLEVBQUU7WUFDYixLQUFLLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNsRDtRQUNELElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUNkLEtBQUssQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3BEO1FBQ0QsSUFBSSxLQUFLLENBQUMsS0FBSyxFQUFFO1lBQ2IsS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDbEQ7UUFDRCxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDbEIsS0FBSyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDeEQ7UUFDRCw4RUFBOEU7UUFDOUUsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDLFlBQWEsQ0FBQyxDQUFDO1FBQ2pILElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRTtZQUMvRCxNQUFNLEVBQUUsRUFBRTtZQUNWLE9BQU8sRUFBRSxLQUFLO1lBQ2QsUUFBUSxFQUFFLFFBQVE7WUFDbEIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2xDLENBQUMsQ0FBQztJQUNQLENBQUM7SUF6SUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsaUJBQXlCO1FBQ3ZGLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNvQixxQkFBZ0IsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQztvQkFDekQsT0FBTyxFQUFFLEtBQUs7b0JBQ2QsTUFBTSxFQUFFLEVBQUU7b0JBQ1YsT0FBTyxFQUFFLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTztvQkFDaEMsUUFBUSxFQUFFLFFBQVE7b0JBQ2xCLFlBQVksRUFBRSxpQkFBaUI7aUJBQ2xDLENBQUMsQ0FBQztZQUNQLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFDRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGdCQUF3QjtRQUNyRixNQUFNLE1BQU8sU0FBUSxlQUFRO1lBQTdCOztnQkFDb0IscUJBQWdCLEdBQUcsZ0JBQWdCLENBQUM7WUFDeEQsQ0FBQztTQUFBO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUNEOzs7Ozs7OztPQVFHO0lBQ0ksTUFBTSxDQUFDLHdCQUF3QixDQUFDLGlCQUF5QjtRQUM1RCxNQUFNLGdCQUFnQjtZQUF0QjtnQkFDb0IscUJBQWdCLEdBQUcsV0FBSSxDQUFDLFdBQVcsQ0FBQztvQkFDaEQsT0FBTyxDQUFDLEdBQW9CO3dCQUN4QixPQUFPLFlBQUssQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQzs0QkFDakMsT0FBTyxFQUFFLEtBQUs7NEJBQ2QsTUFBTSxFQUFFLEVBQUU7NEJBQ1YsT0FBTyxFQUFFLEtBQUs7NEJBQ2QsUUFBUSxFQUFFLFFBQVE7NEJBQ2xCLFlBQVksRUFBRSxpQkFBaUI7eUJBQ2xDLENBQUMsQ0FBQztvQkFDUCxDQUFDO2lCQUNKLENBQUMsQ0FBQztZQUNQLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxnQkFBZ0IsRUFBRSxDQUFDO0lBQ2xDLENBQUM7SUF1RUQ7O09BRUc7SUFDSSxhQUFhLENBQUMsR0FBRyxTQUE0QjtRQUNoRCxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFDRDs7T0FFRztJQUNJLFlBQVksQ0FBQyxJQUFXO1FBQzNCLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssSUFBSSxDQUFDLEVBQUU7WUFDbEMsT0FBTztTQUNWO1FBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUNEOztPQUVHO0lBQ0ksWUFBWSxDQUFDLElBQVc7UUFDM0IsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxJQUFJLENBQUMsRUFBRTtZQUNsQyxPQUFPO1NBQ1Y7UUFDRCxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMxQixDQUFDO0lBQ0Q7O09BRUc7SUFDSSxhQUFhLENBQUMsS0FBYTtRQUM5QixJQUFJLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLEtBQUssQ0FBQyxFQUFFO1lBQ3BDLE9BQU87U0FDVjtRQUNELElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFDUyxRQUFRO1FBQ2QsTUFBTSxNQUFNLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUNuQyxpREFBaUQ7UUFDakQsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRTtZQUN2QixNQUFNLENBQUMsSUFBSSxDQUFDLGdFQUFnRSxDQUFDLENBQUM7U0FDakY7UUFDRCxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyx5QkFBeUIsRUFBRSxDQUFDLENBQUM7UUFDMUQsT0FBTyxNQUFNLENBQUM7SUFDbEIsQ0FBQztDQUNKO0FBckxELHNDQXFMQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbnN0cnVjdCwgSVJlc29sdmVDb250ZXh0LCBMYXp5LCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IElHcm91cCB9IGZyb20gJy4vZ3JvdXAnO1xuaW1wb3J0IHsgQ2ZuTWFuYWdlZFBvbGljeSB9IGZyb20gJy4vaWFtLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBQb2xpY3lEb2N1bWVudCB9IGZyb20gJy4vcG9saWN5LWRvY3VtZW50JztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJy4vcG9saWN5LXN0YXRlbWVudCc7XG5pbXBvcnQgeyBJUm9sZSB9IGZyb20gJy4vcm9sZSc7XG5pbXBvcnQgeyBJVXNlciB9IGZyb20gJy4vdXNlcic7XG5pbXBvcnQgeyB1bmRlZmluZWRJZkVtcHR5IH0gZnJvbSAnLi91dGlsJztcbi8qKlxuICogQSBtYW5hZ2VkIHBvbGljeVxuICovXG5leHBvcnQgaW50ZXJmYWNlIElNYW5hZ2VkUG9saWN5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgQVJOIG9mIHRoZSBtYW5hZ2VkIHBvbGljeVxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSBtYW5hZ2VkUG9saWN5QXJuOiBzdHJpbmc7XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIGFuIElBTSBtYW5hZ2VkIHBvbGljeVxuICovXG5leHBvcnQgaW50ZXJmYWNlIE1hbmFnZWRQb2xpY3lQcm9wcyB7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIG1hbmFnZWQgcG9saWN5LiBJZiB5b3Ugc3BlY2lmeSBtdWx0aXBsZSBwb2xpY2llcyBmb3IgYW4gZW50aXR5LFxuICAgICAqIHNwZWNpZnkgdW5pcXVlIG5hbWVzLiBGb3IgZXhhbXBsZSwgaWYgeW91IHNwZWNpZnkgYSBsaXN0IG9mIHBvbGljaWVzIGZvclxuICAgICAqIGFuIElBTSByb2xlLCBlYWNoIHBvbGljeSBtdXN0IGhhdmUgYSB1bmlxdWUgbmFtZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQSBuYW1lIGlzIGF1dG9tYXRpY2FsbHkgZ2VuZXJhdGVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1hbmFnZWRQb2xpY3lOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEEgZGVzY3JpcHRpb24gb2YgdGhlIG1hbmFnZWQgcG9saWN5LiBUeXBpY2FsbHkgdXNlZCB0byBzdG9yZSBpbmZvcm1hdGlvbiBhYm91dCB0aGVcbiAgICAgKiBwZXJtaXNzaW9ucyBkZWZpbmVkIGluIHRoZSBwb2xpY3kuIEZvciBleGFtcGxlLCBcIkdyYW50cyBhY2Nlc3MgdG8gcHJvZHVjdGlvbiBEeW5hbW9EQiB0YWJsZXMuXCJcbiAgICAgKiBUaGUgcG9saWN5IGRlc2NyaXB0aW9uIGlzIGltbXV0YWJsZS4gQWZ0ZXIgYSB2YWx1ZSBpcyBhc3NpZ25lZCwgaXQgY2Fubm90IGJlIGNoYW5nZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIGVtcHR5XG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHBhdGggZm9yIHRoZSBwb2xpY3kuIFRoaXMgcGFyYW1ldGVyIGFsbG93cyAodGhyb3VnaCBpdHMgcmVnZXggcGF0dGVybikgYSBzdHJpbmcgb2YgY2hhcmFjdGVyc1xuICAgICAqIGNvbnNpc3Rpbmcgb2YgZWl0aGVyIGEgZm9yd2FyZCBzbGFzaCAoLykgYnkgaXRzZWxmIG9yIGEgc3RyaW5nIHRoYXQgbXVzdCBiZWdpbiBhbmQgZW5kIHdpdGggZm9yd2FyZCBzbGFzaGVzLlxuICAgICAqIEluIGFkZGl0aW9uLCBpdCBjYW4gY29udGFpbiBhbnkgQVNDSUkgY2hhcmFjdGVyIGZyb20gdGhlICEgKFxcdTAwMjEpIHRocm91Z2ggdGhlIERFTCBjaGFyYWN0ZXIgKFxcdTAwN0YpLFxuICAgICAqIGluY2x1ZGluZyBtb3N0IHB1bmN0dWF0aW9uIGNoYXJhY3RlcnMsIGRpZ2l0cywgYW5kIHVwcGVyIGFuZCBsb3dlcmNhc2VkIGxldHRlcnMuXG4gICAgICpcbiAgICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBhYm91dCBwYXRocywgc2VlIElBTSBJZGVudGlmaWVycyBpbiB0aGUgSUFNIFVzZXIgR3VpZGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFwiL1wiXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGF0aD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBVc2VycyB0byBhdHRhY2ggdGhpcyBwb2xpY3kgdG8uXG4gICAgICogWW91IGNhbiBhbHNvIHVzZSBgYXR0YWNoVG9Vc2VyKHVzZXIpYCB0byBhdHRhY2ggdGhpcyBwb2xpY3kgdG8gYSB1c2VyLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyB1c2Vycy5cbiAgICAgKi9cbiAgICByZWFkb25seSB1c2Vycz86IElVc2VyW107XG4gICAgLyoqXG4gICAgICogUm9sZXMgdG8gYXR0YWNoIHRoaXMgcG9saWN5IHRvLlxuICAgICAqIFlvdSBjYW4gYWxzbyB1c2UgYGF0dGFjaFRvUm9sZShyb2xlKWAgdG8gYXR0YWNoIHRoaXMgcG9saWN5IHRvIGEgcm9sZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gcm9sZXMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcm9sZXM/OiBJUm9sZVtdO1xuICAgIC8qKlxuICAgICAqIEdyb3VwcyB0byBhdHRhY2ggdGhpcyBwb2xpY3kgdG8uXG4gICAgICogWW91IGNhbiBhbHNvIHVzZSBgYXR0YWNoVG9Hcm91cChncm91cClgIHRvIGF0dGFjaCB0aGlzIHBvbGljeSB0byBhIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBncm91cHMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZ3JvdXBzPzogSUdyb3VwW107XG4gICAgLyoqXG4gICAgICogSW5pdGlhbCBzZXQgb2YgcGVybWlzc2lvbnMgdG8gYWRkIHRvIHRoaXMgcG9saWN5IGRvY3VtZW50LlxuICAgICAqIFlvdSBjYW4gYWxzbyB1c2UgYGFkZFBlcm1pc3Npb24oc3RhdGVtZW50KWAgdG8gYWRkIHBlcm1pc3Npb25zIGxhdGVyLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBzdGF0ZW1lbnRzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0YXRlbWVudHM/OiBQb2xpY3lTdGF0ZW1lbnRbXTtcbiAgICAvKipcbiAgICAgKiBJbml0aWFsIFBvbGljeURvY3VtZW50IHRvIHVzZSBmb3IgdGhpcyBNYW5hZ2VkUG9saWN5LiBJZiBvbWl0ZWQsIGFueVxuICAgICAqIGBQb2xpY3lTdGF0ZW1lbnRgIHByb3ZpZGVkIGluIHRoZSBgc3RhdGVtZW50c2AgcHJvcGVydHkgd2lsbCBiZSBhcHBsaWVkXG4gICAgICogYWdhaW5zdCB0aGUgZW1wdHkgZGVmYXVsdCBgUG9saWN5RG9jdW1lbnRgLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBbiBlbXB0eSBwb2xpY3kuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZG9jdW1lbnQ/OiBQb2xpY3lEb2N1bWVudDtcbn1cbi8qKlxuICogTWFuYWdlZCBwb2xpY3lcbiAqXG4gKi9cbmV4cG9ydCBjbGFzcyBNYW5hZ2VkUG9saWN5IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJTWFuYWdlZFBvbGljeSB7XG4gICAgLyoqXG4gICAgICogSW1wb3J0IGEgY3VzdG9tZXIgbWFuYWdlZCBwb2xpY3kgZnJvbSB0aGUgbWFuYWdlZFBvbGljeU5hbWUuXG4gICAgICpcbiAgICAgKiBGb3IgdGhpcyBtYW5hZ2VkIHBvbGljeSwgeW91IG9ubHkgbmVlZCB0byBrbm93IHRoZSBuYW1lIHRvIGJlIGFibGUgdG8gdXNlIGl0LlxuICAgICAqXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tTWFuYWdlZFBvbGljeU5hbWUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgbWFuYWdlZFBvbGljeU5hbWU6IHN0cmluZyk6IElNYW5hZ2VkUG9saWN5IHtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJTWFuYWdlZFBvbGljeSB7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgbWFuYWdlZFBvbGljeUFybiA9IFN0YWNrLm9mKHNjb3BlKS5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgIHNlcnZpY2U6ICdpYW0nLFxuICAgICAgICAgICAgICAgIHJlZ2lvbjogJycsXG4gICAgICAgICAgICAgICAgYWNjb3VudDogU3RhY2sub2Yoc2NvcGUpLmFjY291bnQsXG4gICAgICAgICAgICAgICAgcmVzb3VyY2U6ICdwb2xpY3knLFxuICAgICAgICAgICAgICAgIHJlc291cmNlTmFtZTogbWFuYWdlZFBvbGljeU5hbWUsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBJbXBvcnQgYW4gZXh0ZXJuYWwgbWFuYWdlZCBwb2xpY3kgYnkgQVJOLlxuICAgICAqXG4gICAgICogRm9yIHRoaXMgbWFuYWdlZCBwb2xpY3ksIHlvdSBvbmx5IG5lZWQgdG8ga25vdyB0aGUgQVJOIHRvIGJlIGFibGUgdG8gdXNlIGl0LlxuICAgICAqIFRoaXMgY2FuIGJlIHVzZWZ1bCBpZiB5b3UgZ290IHRoZSBBUk4gZnJvbSBhIENsb3VkRm9ybWF0aW9uIEV4cG9ydC5cbiAgICAgKlxuICAgICAqIElmIHRoZSBpbXBvcnRlZCBNYW5hZ2VkIFBvbGljeSBBUk4gaXMgYSBUb2tlbiAoc3VjaCBhcyBhXG4gICAgICogYENmblBhcmFtZXRlci52YWx1ZUFzU3RyaW5nYCBvciBhIGBGbi5pbXBvcnRWYWx1ZSgpYCkgKmFuZCogdGhlIHJlZmVyZW5jZWRcbiAgICAgKiBtYW5hZ2VkIHBvbGljeSBoYXMgYSBgcGF0aGAgKGxpa2UgYGFybjouLi46cG9saWN5L0FkbWluUG9saWN5L0FkbWluQWxsb3dgKSwgdGhlXG4gICAgICogYG1hbmFnZWRQb2xpY3lOYW1lYCBwcm9wZXJ0eSB3aWxsIG5vdCByZXNvbHZlIHRvIHRoZSBjb3JyZWN0IHZhbHVlLiBJbnN0ZWFkIGl0XG4gICAgICogd2lsbCByZXNvbHZlIHRvIHRoZSBmaXJzdCBwYXRoIGNvbXBvbmVudC4gV2UgdW5mb3J0dW5hdGVseSBjYW5ub3QgZXhwcmVzc1xuICAgICAqIHRoZSBjb3JyZWN0IGNhbGN1bGF0aW9uIG9mIHRoZSBmdWxsIHBhdGggbmFtZSBhcyBhIENsb3VkRm9ybWF0aW9uXG4gICAgICogZXhwcmVzc2lvbi4gSW4gdGhpcyBzY2VuYXJpbyB0aGUgTWFuYWdlZCBQb2xpY3kgQVJOIHNob3VsZCBiZSBzdXBwbGllZCB3aXRob3V0IHRoZVxuICAgICAqIGBwYXRoYCBpbiBvcmRlciB0byByZXNvbHZlIHRoZSBjb3JyZWN0IG1hbmFnZWQgcG9saWN5IHJlc291cmNlLlxuICAgICAqXG4gICAgICogQHBhcmFtIHNjb3BlIGNvbnN0cnVjdCBzY29wZVxuICAgICAqIEBwYXJhbSBpZCBjb25zdHJ1Y3QgaWRcbiAgICAgKiBAcGFyYW0gbWFuYWdlZFBvbGljeUFybiB0aGUgQVJOIG9mIHRoZSBtYW5hZ2VkIHBvbGljeSB0byBpbXBvcnRcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21NYW5hZ2VkUG9saWN5QXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIG1hbmFnZWRQb2xpY3lBcm46IHN0cmluZyk6IElNYW5hZ2VkUG9saWN5IHtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJTWFuYWdlZFBvbGljeSB7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgbWFuYWdlZFBvbGljeUFybiA9IG1hbmFnZWRQb2xpY3lBcm47XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogSW1wb3J0IGEgbWFuYWdlZCBwb2xpY3kgZnJvbSBvbmUgb2YgdGhlIHBvbGljaWVzIHRoYXQgQVdTIG1hbmFnZXMuXG4gICAgICpcbiAgICAgKiBGb3IgdGhpcyBtYW5hZ2VkIHBvbGljeSwgeW91IG9ubHkgbmVlZCB0byBrbm93IHRoZSBuYW1lIHRvIGJlIGFibGUgdG8gdXNlIGl0LlxuICAgICAqXG4gICAgICogU29tZSBtYW5hZ2VkIHBvbGljeSBuYW1lcyBzdGFydCB3aXRoIFwic2VydmljZS1yb2xlL1wiLCBzb21lIHN0YXJ0IHdpdGhcbiAgICAgKiBcImpvYi1mdW5jdGlvbi9cIiwgYW5kIHNvbWUgZG9uJ3Qgc3RhcnQgd2l0aCBhbnl0aGluZy4gRG8gaW5jbHVkZSB0aGVcbiAgICAgKiBwcmVmaXggd2hlbiBjb25zdHJ1Y3RpbmcgdGhpcyBvYmplY3QuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUobWFuYWdlZFBvbGljeU5hbWU6IHN0cmluZyk6IElNYW5hZ2VkUG9saWN5IHtcbiAgICAgICAgY2xhc3MgQXdzTWFuYWdlZFBvbGljeSBpbXBsZW1lbnRzIElNYW5hZ2VkUG9saWN5IHtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBtYW5hZ2VkUG9saWN5QXJuID0gTGF6eS5zdHJpbmdWYWx1ZSh7XG4gICAgICAgICAgICAgICAgcHJvZHVjZShjdHg6IElSZXNvbHZlQ29udGV4dCkge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gU3RhY2sub2YoY3R4LnNjb3BlKS5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgICAgICAgICAgc2VydmljZTogJ2lhbScsXG4gICAgICAgICAgICAgICAgICAgICAgICByZWdpb246ICcnLFxuICAgICAgICAgICAgICAgICAgICAgICAgYWNjb3VudDogJ2F3cycsXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZTogJ3BvbGljeScsXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6IG1hbmFnZWRQb2xpY3lOYW1lLFxuICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBBd3NNYW5hZ2VkUG9saWN5KCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybnMgdGhlIEFSTiBvZiB0aGlzIG1hbmFnZWQgcG9saWN5LlxuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBtYW5hZ2VkUG9saWN5QXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHBvbGljeSBkb2N1bWVudC5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZG9jdW1lbnQgPSBuZXcgUG9saWN5RG9jdW1lbnQoKTtcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGlzIHBvbGljeS5cbiAgICAgKlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgbWFuYWdlZFBvbGljeU5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgZGVzY3JpcHRpb24gb2YgdGhpcyBwb2xpY3kuXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGRlc2NyaXB0aW9uOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHBhdGggb2YgdGhpcyBwb2xpY3kuXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHBhdGg6IHN0cmluZztcbiAgICBwcml2YXRlIHJlYWRvbmx5IHJvbGVzID0gbmV3IEFycmF5PElSb2xlPigpO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgdXNlcnMgPSBuZXcgQXJyYXk8SVVzZXI+KCk7XG4gICAgcHJpdmF0ZSByZWFkb25seSBncm91cHMgPSBuZXcgQXJyYXk8SUdyb3VwPigpO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBNYW5hZ2VkUG9saWN5UHJvcHMgPSB7fSkge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMubWFuYWdlZFBvbGljeU5hbWUsXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLmRlc2NyaXB0aW9uID0gcHJvcHMuZGVzY3JpcHRpb24gfHwgJyc7XG4gICAgICAgIHRoaXMucGF0aCA9IHByb3BzLnBhdGggfHwgJy8nO1xuICAgICAgICBpZiAocHJvcHMuZG9jdW1lbnQpIHtcbiAgICAgICAgICAgIHRoaXMuZG9jdW1lbnQgPSBwcm9wcy5kb2N1bWVudDtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5NYW5hZ2VkUG9saWN5KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIHBvbGljeURvY3VtZW50OiB0aGlzLmRvY3VtZW50LFxuICAgICAgICAgICAgbWFuYWdlZFBvbGljeU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246IHRoaXMuZGVzY3JpcHRpb24sXG4gICAgICAgICAgICBwYXRoOiB0aGlzLnBhdGgsXG4gICAgICAgICAgICByb2xlczogdW5kZWZpbmVkSWZFbXB0eSgoKSA9PiB0aGlzLnJvbGVzLm1hcChyID0+IHIucm9sZU5hbWUpKSxcbiAgICAgICAgICAgIHVzZXJzOiB1bmRlZmluZWRJZkVtcHR5KCgpID0+IHRoaXMudXNlcnMubWFwKHUgPT4gdS51c2VyTmFtZSkpLFxuICAgICAgICAgICAgZ3JvdXBzOiB1bmRlZmluZWRJZkVtcHR5KCgpID0+IHRoaXMuZ3JvdXBzLm1hcChnID0+IGcuZ3JvdXBOYW1lKSksXG4gICAgICAgIH0pO1xuICAgICAgICBpZiAocHJvcHMudXNlcnMpIHtcbiAgICAgICAgICAgIHByb3BzLnVzZXJzLmZvckVhY2godSA9PiB0aGlzLmF0dGFjaFRvVXNlcih1KSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmdyb3Vwcykge1xuICAgICAgICAgICAgcHJvcHMuZ3JvdXBzLmZvckVhY2goZyA9PiB0aGlzLmF0dGFjaFRvR3JvdXAoZykpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5yb2xlcykge1xuICAgICAgICAgICAgcHJvcHMucm9sZXMuZm9yRWFjaChyID0+IHRoaXMuYXR0YWNoVG9Sb2xlKHIpKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuc3RhdGVtZW50cykge1xuICAgICAgICAgICAgcHJvcHMuc3RhdGVtZW50cy5mb3JFYWNoKHAgPT4gdGhpcy5hZGRTdGF0ZW1lbnRzKHApKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBhcm46YXdzOmlhbTo6MTIzNDU2Nzg5MDEyOnBvbGljeS90ZXN0c3RhY2stQ3JlYXRlVGVzdERCUG9saWN5LTE2TTIzWUUzQ1M3MDBcbiAgICAgICAgdGhpcy5tYW5hZ2VkUG9saWN5TmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKFN0YWNrLm9mKHRoaXMpLnBhcnNlQXJuKHJlc291cmNlLnJlZiwgJy8nKS5yZXNvdXJjZU5hbWUhKTtcbiAgICAgICAgdGhpcy5tYW5hZ2VkUG9saWN5QXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShyZXNvdXJjZS5yZWYsIHtcbiAgICAgICAgICAgIHJlZ2lvbjogJycsXG4gICAgICAgICAgICBzZXJ2aWNlOiAnaWFtJyxcbiAgICAgICAgICAgIHJlc291cmNlOiAncG9saWN5JyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGRzIGEgc3RhdGVtZW50IHRvIHRoZSBwb2xpY3kgZG9jdW1lbnQuXG4gICAgICovXG4gICAgcHVibGljIGFkZFN0YXRlbWVudHMoLi4uc3RhdGVtZW50OiBQb2xpY3lTdGF0ZW1lbnRbXSkge1xuICAgICAgICB0aGlzLmRvY3VtZW50LmFkZFN0YXRlbWVudHMoLi4uc3RhdGVtZW50KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQXR0YWNoZXMgdGhpcyBwb2xpY3kgdG8gYSB1c2VyLlxuICAgICAqL1xuICAgIHB1YmxpYyBhdHRhY2hUb1VzZXIodXNlcjogSVVzZXIpIHtcbiAgICAgICAgaWYgKHRoaXMudXNlcnMuZmluZCh1ID0+IHUgPT09IHVzZXIpKSB7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy51c2Vycy5wdXNoKHVzZXIpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBdHRhY2hlcyB0aGlzIHBvbGljeSB0byBhIHJvbGUuXG4gICAgICovXG4gICAgcHVibGljIGF0dGFjaFRvUm9sZShyb2xlOiBJUm9sZSkge1xuICAgICAgICBpZiAodGhpcy5yb2xlcy5maW5kKHIgPT4gciA9PT0gcm9sZSkpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnJvbGVzLnB1c2gocm9sZSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEF0dGFjaGVzIHRoaXMgcG9saWN5IHRvIGEgZ3JvdXAuXG4gICAgICovXG4gICAgcHVibGljIGF0dGFjaFRvR3JvdXAoZ3JvdXA6IElHcm91cCkge1xuICAgICAgICBpZiAodGhpcy5ncm91cHMuZmluZChnID0+IGcgPT09IGdyb3VwKSkge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuZ3JvdXBzLnB1c2goZ3JvdXApO1xuICAgIH1cbiAgICBwcm90ZWN0ZWQgdmFsaWRhdGUoKTogc3RyaW5nW10ge1xuICAgICAgICBjb25zdCByZXN1bHQgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICAgICAgICAvLyB2YWxpZGF0ZSB0aGF0IHRoZSBwb2xpY3kgZG9jdW1lbnQgaXMgbm90IGVtcHR5XG4gICAgICAgIGlmICh0aGlzLmRvY3VtZW50LmlzRW1wdHkpIHtcbiAgICAgICAgICAgIHJlc3VsdC5wdXNoKCdNYW5hZ2VkIFBvbGljeSBpcyBlbXB0eS4gWW91IG11c3QgYWRkIHN0YXRlbWVudHMgdG8gdGhlIHBvbGljeScpO1xuICAgICAgICB9XG4gICAgICAgIHJlc3VsdC5wdXNoKC4uLnRoaXMuZG9jdW1lbnQudmFsaWRhdGVGb3JJZGVudGl0eVBvbGljeSgpKTtcbiAgICAgICAgcmV0dXJuIHJlc3VsdDtcbiAgICB9XG59XG4iXX0=