"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeDependable = exports.Grant = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const resourceAndPrincipalAccountComparison = options.grantee.grantPrincipal.principalAccount
            ? cdk.Token.compareStrings(options.resource.env.account, options.grantee.grantPrincipal.principalAccount)
            : undefined;
        // if both accounts are tokens, we assume here they are the same
        const equalOrBothUnresolved = resourceAndPrincipalAccountComparison === cdk.TokenComparison.SAME
            || resourceAndPrincipalAccountComparison == cdk.TokenComparison.BOTH_UNRESOLVED;
        const sameAccount = resourceAndPrincipalAccountComparison
            ? equalOrBothUnresolved
            // if the principal doesn't have an account (for example, a service principal),
            // we should modify the resource's trust policy
            : false;
        // If we added to the principal AND we're in the same account, then we're done.
        // If not, it's a different account and we must also add a trust policy on the resource.
        if (result.success && sameAccount) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // eslint-disable-next-line max-len
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
//# sourceMappingURL=data:application/json;base64,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