import * as iam from "../../aws-iam";
import * as kms from "../../aws-kms";
import * as s3 from "../../aws-s3";
import { Construct, IResource, Resource } from "../../core";
import { DataFormat } from './data-format';
import { IDatabase } from './database';
import { Column } from './schema';
export interface ITable extends IResource {
    /**
     * @attribute
     */
    readonly tableArn: string;
    /**
     * @attribute
     */
    readonly tableName: string;
}
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
export declare enum TableEncryption {
    UNENCRYPTED = "Unencrypted",
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    S3_MANAGED = "SSE-S3",
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    KMS = "SSE-KMS",
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    KMS_MANAGED = "SSE-KMS-MANAGED",
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    CLIENT_SIDE_KMS = "CSE-KMS"
}
export interface TableAttributes {
    readonly tableArn: string;
    readonly tableName: string;
}
export interface TableProps {
    /**
     * Name of the table.
     */
    readonly tableName: string;
    /**
     * Description of the table.
     *
     * @default generated
     */
    readonly description?: string;
    /**
     * Database in which to store the table.
     */
    readonly database: IDatabase;
    /**
     * S3 bucket in which to store data.
     *
     * @default one is created for you
     */
    readonly bucket?: s3.IBucket;
    /**
     * S3 prefix under which table objects are stored.
     *
     * @default - No prefix. The data will be stored under the root of the bucket.
     */
    readonly s3Prefix?: string;
    /**
     * Columns of the table.
     */
    readonly columns: Column[];
    /**
     * Partition columns of the table.
     *
     * @default table is not partitioned
     */
    readonly partitionKeys?: Column[];
    /**
     * Storage type of the table's data.
     */
    readonly dataFormat: DataFormat;
    /**
     * Indicates whether the table's data is compressed or not.
     *
     * @default false
     */
    readonly compressed?: boolean;
    /**
     * The kind of encryption to secure the data with.
     *
     * You can only provide this option if you are not explicitly passing in a bucket.
     *
     * If you choose `SSE-KMS`, you *can* provide an un-managed KMS key with `encryptionKey`.
     * If you choose `CSE-KMS`, you *must* provide an un-managed KMS key with `encryptionKey`.
     *
     * @default Unencrypted
     */
    readonly encryption?: TableEncryption;
    /**
     * External KMS key to use for bucket encryption.
     *
     * The `encryption` property must be `SSE-KMS` or `CSE-KMS`.
     *
     * @default key is managed by KMS.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Indicates whether the table data is stored in subdirectories.
     *
     * @default false
     */
    readonly storedAsSubDirectories?: boolean;
}
/**
 * A Glue table.
 */
export declare class Table extends Resource implements ITable {
    static fromTableArn(scope: Construct, id: string, tableArn: string): ITable;
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope: Construct, id: string, attrs: TableAttributes): ITable;
    /**
     * Database this table belongs to.
     */
    readonly database: IDatabase;
    /**
     * Indicates whether the table's data is compressed or not.
     */
    readonly compressed: boolean;
    /**
     * The type of encryption enabled for the table.
     */
    readonly encryption: TableEncryption;
    /**
     * The KMS key used to secure the data if `encryption` is set to `CSE-KMS` or `SSE-KMS`. Otherwise, `undefined`.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * S3 bucket in which the table's data resides.
     */
    readonly bucket: s3.IBucket;
    /**
     * S3 Key Prefix under which this table's files are stored in S3.
     */
    readonly s3Prefix: string;
    /**
     * Name of this table.
     */
    readonly tableName: string;
    /**
     * ARN of this table.
     */
    readonly tableArn: string;
    /**
     * Format of this table's data files.
     */
    readonly dataFormat: DataFormat;
    /**
     * This table's columns.
     */
    readonly columns: Column[];
    /**
     * This table's partition keys if the table is partitioned.
     */
    readonly partitionKeys?: Column[];
    constructor(scope: Construct, id: string, props: TableProps);
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee: iam.IGrantable): iam.Grant;
    private grant;
}
