"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDeploymentType = void 0;
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    /**
     * Original type for shorter term data processing. Data is not replicated and does not persist on server fail.
     */
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    /**
     * Newer type for shorter term data processing. Data is not replicated and does not persist on server fail.
     * Provides better support for spiky workloads.
     */
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    /**
     * Long term storage. Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_a = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _a === void 0 ? void 0 : _a.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1 deployment type');
        }
        if (![50, 100, 200].includes(perUnitStorageThroughput)) {
            throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB');
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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