"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.accountEventBusTargets = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.anyValue({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.anyValue({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target) {
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = targetProps.role ? targetProps.role.roleArn : undefined;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetStack.account;
            const targetRegion = targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            if (targetRegion !== sourceRegion) {
                throw new Error('Rule and target must be in the same region');
            }
            if (targetAccount !== sourceAccount) {
                // cross-account event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html
                // for cross-account events, we require concrete accounts
                if (core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account events');
                }
                // and the target region has to be concrete as well
                if (core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account
                const exists = this.accountEventBusTargets[targetAccount];
                if (!exists) {
                    this.accountEventBusTargets[targetAccount] = true;
                    this.targets.push({
                        id,
                        arn: targetStack.formatArn({
                            service: 'events',
                            resource: 'event-bus',
                            resourceName: 'default',
                            region: targetRegion,
                            account: targetAccount,
                        }),
                    });
                }
                // Grant the source account permissions to publish events to the event bus of the target account.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (EventBridge verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account targets must be part of a CDK app');
                }
                const targetApp = targetProps.targetResource.node.root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                if (!eventBusPolicyStack) {
                    eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                        env: {
                            account: targetAccount,
                            region: targetRegion,
                        },
                        stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                    });
                    new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                        action: 'events:PutEvents',
                        statementId: `Allow-account-${sourceAccount}`,
                        principal: sourceAccount,
                    });
                }
                // deploy the event bus permissions before the source stack
                sourceStack.addDependency(eventBusPolicyStack);
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${this.node.uniqueId}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            sqsParameters: targetProps.sqsParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: inputProps && inputProps.inputTemplate !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
}
exports.Rule = Rule;
//# sourceMappingURL=data:application/json;base64,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