"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsTask = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const ecs = require("../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster
 */
class EcsTask {
    constructor(props) {
        this.props = props;
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = props.taskCount !== undefined ? props.taskCount : 1;
        if (props.role) {
            const role = props.role;
            this.createEventRolePolicyStatements().forEach(role.addToPolicy.bind(role));
            this.role = role;
        }
        else {
            this.role = util_1.singletonEventRole(this.taskDefinition, this.createEventRolePolicyStatements());
        }
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                this.taskDefinition.node.addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of EventBridge events
     */
    bind(_rule, _id) {
        const arn = this.cluster.clusterArn;
        const role = this.role;
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            id: '',
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
    createEventRolePolicyStatements() {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the EventBridge Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        return policyStatements;
    }
}
exports.EcsTask = EcsTask;
//# sourceMappingURL=data:application/json;base64,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