"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateRedirectResponse = exports.validateFixedResponse = exports.ApplicationListenerRule = exports.ContentType = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * The content type for a fixed response
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule
 */
class ApplicationListenerRule extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.legacyConditions = {};
        this.conditions = props.conditions || [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (this.conditions.length === 0 && !props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'conditions\', \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * Add a non-standard condition to this rule
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field, values) {
        if (values === undefined) {
            delete this.legacyConditions[field];
            return;
        }
        this.legacyConditions[field] = values;
    }
    /**
     * Add a non-standard condition to this rule
     */
    addCondition(condition) {
        this.conditions.push(condition);
    }
    /**
     * Configure the action to perform for this rule
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            this.node.addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * Add a TargetGroup to load balance to
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * Validate the rule
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        const legacyConditionFields = Object.keys(this.legacyConditions);
        if (legacyConditionFields.length === 0 && this.conditions.length === 0) {
            return ['Listener rule needs at least one condition'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const legacyConditions = Object.entries(this.legacyConditions).map(([field, values]) => {
            return { field, values };
        });
        const conditions = this.conditions.map(condition => condition.renderRawCondition());
        return [
            ...legacyConditions,
            ...conditions,
        ];
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,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