"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirelensLogRouter = exports.obtainDefaultFluentBitECRImage = exports.FirelensConfigFileType = exports.FirelensLogRouterType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * Firelens log router type, fluentbit or fluentd.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    /**
     * fluentbit
     */
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    /**
     * fluentd
     */
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * Firelens configuration file type, s3 or file path.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    /**
     * s3
     */
    FirelensConfigFileType["S3"] = "s3";
    /**
     * fluentd
     */
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * Firelens log router
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * Constructs a new instance of the FirelensLogRouter class.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const options = props.firelensConfig.options;
        if (options) {
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    configFileType,
                    configFileValue: options.configFileValue,
                },
            };
            // grant s3 access permissions
            if (configFileType === FirelensConfigFileType.S3) {
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [options.configFileValue],
                }));
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetBucketLocation',
                    ],
                    resources: [options.configFileValue.split('/')[0]],
                }));
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * Render this container definition to a CloudFormation object
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
exports.FirelensLogRouter = FirelensLogRouter;
//# sourceMappingURL=data:application/json;base64,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