"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArtifactPath = exports.Artifact = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const validation = require("./validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * Add arbitrary extra payload to the artifact under a given key.
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Retrieve the metadata stored in this artifact under the given key.
     * If there is no metadata stored under the given key,
     * null will be returned.
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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