"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Spacer = exports.Column = exports.Row = void 0;
const widget_1 = require("./widget");
// This file contains widgets that exist for layout purposes
/**
 * A widget that contains other widgets in a horizontal row
 *
 * Widgets will be laid out next to each other
 */
class Row {
    constructor(...widgets) {
        /**
         * Relative position of each widget inside this row
         */
        this.offsets = [];
        this.widgets = widgets;
        this.width = 0;
        this.height = 0;
        let x = 0;
        let y = 0;
        for (const widget of widgets) {
            // See if we need to horizontally wrap to add this widget
            if (x + widget.width > widget_1.GRID_WIDTH) {
                y = this.height;
                x = 0;
            }
            this.offsets.push({ x, y });
            this.width = Math.max(this.width, x + widget.width);
            this.height = Math.max(this.height, y + widget.height);
            x += widget.width;
        }
    }
    position(x, y) {
        for (let i = 0; i < this.widgets.length; i++) {
            this.widgets[i].position(x + this.offsets[i].x, y + this.offsets[i].y);
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Row = Row;
/**
 * A widget that contains other widgets in a vertical column
 *
 * Widgets will be laid out next to each other
 */
class Column {
    constructor(...widgets) {
        this.widgets = widgets;
        // There's no vertical wrapping so this one's a lot easier
        this.width = Math.max(...this.widgets.map(w => w.width));
        this.height = sum(...this.widgets.map(w => w.height));
    }
    position(x, y) {
        let widgetY = y;
        for (const widget of this.widgets) {
            widget.position(x, widgetY);
            widgetY += widget.height;
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Column = Column;
/**
 * A widget that doesn't display anything but takes up space
 */
class Spacer {
    constructor(props = {}) {
        this.width = props.width || 1;
        this.height = props.height || 1;
    }
    position(_x, _y) {
        // Don't need to do anything, not a physical widget
    }
    toJson() {
        return [];
    }
}
exports.Spacer = Spacer;
/**
 * Return the sum of a list of numbers
 */
function sum(...xs) {
    let ret = 0;
    for (const x of xs) {
        ret += x;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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