"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmRule = exports.AlarmState = void 0;
/**
 * Enumeration indicates state of Alarm used in building Alarm Rule.
 */
var AlarmState;
(function (AlarmState) {
    /**
     * State indicates resource is in ALARM
     */
    AlarmState["ALARM"] = "ALARM";
    /**
     * State indicates resource is not in ALARM
     */
    AlarmState["OK"] = "OK";
    /**
     * State indicates there is not enough data to determine is resource is in ALARM
     */
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
/**
 * Enumeration of supported Composite Alarms operators.
 */
var Operator;
(function (Operator) {
    Operator["AND"] = "AND";
    Operator["OR"] = "OR";
    Operator["NOT"] = "NOT";
})(Operator || (Operator = {}));
/**
 * Class with static functions to build AlarmRule for Composite Alarms.
 */
class AlarmRule {
    /**
     * function to join all provided AlarmRules with AND operator.
     *
     * @param operands IAlarmRules to be joined with AND operator.
     */
    static allOf(...operands) {
        return this.concat(Operator.AND, ...operands);
    }
    /**
     * function to join all provided AlarmRules with OR operator.
     *
     * @param operands IAlarmRules to be joined with OR operator.
     */
    static anyOf(...operands) {
        return this.concat(Operator.OR, ...operands);
    }
    /**
     * function to wrap provided AlarmRule in NOT operator.
     *
     * @param operand IAlarmRule to be wrapped in NOT operator.
     */
    static not(operand) {
        return new class {
            renderAlarmRule() {
                return `(NOT (${operand.renderAlarmRule()}))`;
            }
        };
    }
    /**
     * function to build TRUE/FALSE intent for Rule Expression.
     *
     * @param value boolean value to be used in rule expression.
     */
    static fromBoolean(value) {
        return new class {
            renderAlarmRule() {
                return `${String(value).toUpperCase()}`;
            }
        };
    }
    /**
     * function to build Rule Expression for given IAlarm and AlarmState.
     *
     * @param alarm IAlarm to be used in Rule Expression.
     * @param alarmState AlarmState to be used in Rule Expression.
     */
    static fromAlarm(alarm, alarmState) {
        return new class {
            renderAlarmRule() {
                return `${alarmState}(${alarm.alarmArn})`;
            }
        };
    }
    /**
     * function to build Rule Expression for given Alarm Rule string.
     *
     * @param alarmRule string to be used in Rule Expression.
     */
    static fromString(alarmRule) {
        return new class {
            renderAlarmRule() {
                return alarmRule;
            }
        };
    }
    static concat(operator, ...operands) {
        return new class {
            renderAlarmRule() {
                const expression = operands
                    .map(operand => `${operand.renderAlarmRule()}`)
                    .join(` ${operator} `);
                return `(${expression})`;
            }
        };
    }
}
exports.AlarmRule = AlarmRule;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxhcm0tcnVsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFsYXJtLXJ1bGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0E7O0dBRUc7QUFDSCxJQUFZLFVBYVg7QUFiRCxXQUFZLFVBQVU7SUFDbEI7O09BRUc7SUFDSCw2QkFBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCx1QkFBUyxDQUFBO0lBQ1Q7O09BRUc7SUFDSCxxREFBdUMsQ0FBQTtBQUMzQyxDQUFDLEVBYlcsVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUFhckI7QUFDRDs7R0FFRztBQUNILElBQUssUUFJSjtBQUpELFdBQUssUUFBUTtJQUNULHVCQUFXLENBQUE7SUFDWCxxQkFBUyxDQUFBO0lBQ1QsdUJBQVcsQ0FBQTtBQUNmLENBQUMsRUFKSSxRQUFRLEtBQVIsUUFBUSxRQUlaO0FBQ0Q7O0dBRUc7QUFDSCxNQUFhLFNBQVM7SUFDbEI7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxRQUFzQjtRQUN6QyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLEtBQUssQ0FBQyxHQUFHLFFBQXNCO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLEdBQUcsUUFBUSxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQW1CO1FBQ2pDLE9BQU8sSUFBSTtZQUNBLGVBQWU7Z0JBQ2xCLE9BQU8sU0FBUyxPQUFPLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQztZQUNsRCxDQUFDO1NBQ0osQ0FBQztJQUNOLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFjO1FBQ3BDLE9BQU8sSUFBSTtZQUNBLGVBQWU7Z0JBQ2xCLE9BQU8sR0FBRyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztZQUM1QyxDQUFDO1NBQ0osQ0FBQztJQUNOLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsS0FBYSxFQUFFLFVBQXNCO1FBQ3pELE9BQU8sSUFBSTtZQUNBLGVBQWU7Z0JBQ2xCLE9BQU8sR0FBRyxVQUFVLElBQUksS0FBSyxDQUFDLFFBQVEsR0FBRyxDQUFDO1lBQzlDLENBQUM7U0FDSixDQUFDO0lBQ04sQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLFNBQWlCO1FBQ3RDLE9BQU8sSUFBSTtZQUNBLGVBQWU7Z0JBQ2xCLE9BQU8sU0FBUyxDQUFDO1lBQ3JCLENBQUM7U0FDSixDQUFDO0lBQ04sQ0FBQztJQUNPLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBa0IsRUFBRSxHQUFHLFFBQXNCO1FBQy9ELE9BQU8sSUFBSTtZQUNBLGVBQWU7Z0JBQ2xCLE1BQU0sVUFBVSxHQUFHLFFBQVE7cUJBQ3RCLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEdBQUcsT0FBTyxDQUFDLGVBQWUsRUFBRSxFQUFFLENBQUM7cUJBQzlDLElBQUksQ0FBQyxJQUFJLFFBQVEsR0FBRyxDQUFDLENBQUM7Z0JBQzNCLE9BQU8sSUFBSSxVQUFVLEdBQUcsQ0FBQztZQUM3QixDQUFDO1NBQ0osQ0FBQztJQUNOLENBQUM7Q0FDSjtBQTVFRCw4QkE0RUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJQWxhcm0sIElBbGFybVJ1bGUgfSBmcm9tICcuL2FsYXJtLWJhc2UnO1xuLyoqXG4gKiBFbnVtZXJhdGlvbiBpbmRpY2F0ZXMgc3RhdGUgb2YgQWxhcm0gdXNlZCBpbiBidWlsZGluZyBBbGFybSBSdWxlLlxuICovXG5leHBvcnQgZW51bSBBbGFybVN0YXRlIHtcbiAgICAvKipcbiAgICAgKiBTdGF0ZSBpbmRpY2F0ZXMgcmVzb3VyY2UgaXMgaW4gQUxBUk1cbiAgICAgKi9cbiAgICBBTEFSTSA9ICdBTEFSTScsXG4gICAgLyoqXG4gICAgICogU3RhdGUgaW5kaWNhdGVzIHJlc291cmNlIGlzIG5vdCBpbiBBTEFSTVxuICAgICAqL1xuICAgIE9LID0gJ09LJyxcbiAgICAvKipcbiAgICAgKiBTdGF0ZSBpbmRpY2F0ZXMgdGhlcmUgaXMgbm90IGVub3VnaCBkYXRhIHRvIGRldGVybWluZSBpcyByZXNvdXJjZSBpcyBpbiBBTEFSTVxuICAgICAqL1xuICAgIElOU1VGRklDSUVOVF9EQVRBID0gJ0lOU1VGRklDSUVOVF9EQVRBJ1xufVxuLyoqXG4gKiBFbnVtZXJhdGlvbiBvZiBzdXBwb3J0ZWQgQ29tcG9zaXRlIEFsYXJtcyBvcGVyYXRvcnMuXG4gKi9cbmVudW0gT3BlcmF0b3Ige1xuICAgIEFORCA9ICdBTkQnLFxuICAgIE9SID0gJ09SJyxcbiAgICBOT1QgPSAnTk9UJ1xufVxuLyoqXG4gKiBDbGFzcyB3aXRoIHN0YXRpYyBmdW5jdGlvbnMgdG8gYnVpbGQgQWxhcm1SdWxlIGZvciBDb21wb3NpdGUgQWxhcm1zLlxuICovXG5leHBvcnQgY2xhc3MgQWxhcm1SdWxlIHtcbiAgICAvKipcbiAgICAgKiBmdW5jdGlvbiB0byBqb2luIGFsbCBwcm92aWRlZCBBbGFybVJ1bGVzIHdpdGggQU5EIG9wZXJhdG9yLlxuICAgICAqXG4gICAgICogQHBhcmFtIG9wZXJhbmRzIElBbGFybVJ1bGVzIHRvIGJlIGpvaW5lZCB3aXRoIEFORCBvcGVyYXRvci5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGFsbE9mKC4uLm9wZXJhbmRzOiBJQWxhcm1SdWxlW10pOiBJQWxhcm1SdWxlIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuY29uY2F0KE9wZXJhdG9yLkFORCwgLi4ub3BlcmFuZHMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBmdW5jdGlvbiB0byBqb2luIGFsbCBwcm92aWRlZCBBbGFybVJ1bGVzIHdpdGggT1Igb3BlcmF0b3IuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gb3BlcmFuZHMgSUFsYXJtUnVsZXMgdG8gYmUgam9pbmVkIHdpdGggT1Igb3BlcmF0b3IuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBhbnlPZiguLi5vcGVyYW5kczogSUFsYXJtUnVsZVtdKTogSUFsYXJtUnVsZSB7XG4gICAgICAgIHJldHVybiB0aGlzLmNvbmNhdChPcGVyYXRvci5PUiwgLi4ub3BlcmFuZHMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBmdW5jdGlvbiB0byB3cmFwIHByb3ZpZGVkIEFsYXJtUnVsZSBpbiBOT1Qgb3BlcmF0b3IuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gb3BlcmFuZCBJQWxhcm1SdWxlIHRvIGJlIHdyYXBwZWQgaW4gTk9UIG9wZXJhdG9yLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgbm90KG9wZXJhbmQ6IElBbGFybVJ1bGUpOiBJQWxhcm1SdWxlIHtcbiAgICAgICAgcmV0dXJuIG5ldyBjbGFzcyBpbXBsZW1lbnRzIElBbGFybVJ1bGUge1xuICAgICAgICAgICAgcHVibGljIHJlbmRlckFsYXJtUnVsZSgpOiBzdHJpbmcge1xuICAgICAgICAgICAgICAgIHJldHVybiBgKE5PVCAoJHtvcGVyYW5kLnJlbmRlckFsYXJtUnVsZSgpfSkpYDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogZnVuY3Rpb24gdG8gYnVpbGQgVFJVRS9GQUxTRSBpbnRlbnQgZm9yIFJ1bGUgRXhwcmVzc2lvbi5cbiAgICAgKlxuICAgICAqIEBwYXJhbSB2YWx1ZSBib29sZWFuIHZhbHVlIHRvIGJlIHVzZWQgaW4gcnVsZSBleHByZXNzaW9uLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUJvb2xlYW4odmFsdWU6IGJvb2xlYW4pOiBJQWxhcm1SdWxlIHtcbiAgICAgICAgcmV0dXJuIG5ldyBjbGFzcyBpbXBsZW1lbnRzIElBbGFybVJ1bGUge1xuICAgICAgICAgICAgcHVibGljIHJlbmRlckFsYXJtUnVsZSgpOiBzdHJpbmcge1xuICAgICAgICAgICAgICAgIHJldHVybiBgJHtTdHJpbmcodmFsdWUpLnRvVXBwZXJDYXNlKCl9YDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogZnVuY3Rpb24gdG8gYnVpbGQgUnVsZSBFeHByZXNzaW9uIGZvciBnaXZlbiBJQWxhcm0gYW5kIEFsYXJtU3RhdGUuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gYWxhcm0gSUFsYXJtIHRvIGJlIHVzZWQgaW4gUnVsZSBFeHByZXNzaW9uLlxuICAgICAqIEBwYXJhbSBhbGFybVN0YXRlIEFsYXJtU3RhdGUgdG8gYmUgdXNlZCBpbiBSdWxlIEV4cHJlc3Npb24uXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tQWxhcm0oYWxhcm06IElBbGFybSwgYWxhcm1TdGF0ZTogQWxhcm1TdGF0ZSk6IElBbGFybVJ1bGUge1xuICAgICAgICByZXR1cm4gbmV3IGNsYXNzIGltcGxlbWVudHMgSUFsYXJtUnVsZSB7XG4gICAgICAgICAgICBwdWJsaWMgcmVuZGVyQWxhcm1SdWxlKCk6IHN0cmluZyB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIGAke2FsYXJtU3RhdGV9KCR7YWxhcm0uYWxhcm1Bcm59KWA7XG4gICAgICAgICAgICB9XG4gICAgICAgIH07XG4gICAgfVxuICAgIC8qKlxuICAgICAqIGZ1bmN0aW9uIHRvIGJ1aWxkIFJ1bGUgRXhwcmVzc2lvbiBmb3IgZ2l2ZW4gQWxhcm0gUnVsZSBzdHJpbmcuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gYWxhcm1SdWxlIHN0cmluZyB0byBiZSB1c2VkIGluIFJ1bGUgRXhwcmVzc2lvbi5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcoYWxhcm1SdWxlOiBzdHJpbmcpOiBJQWxhcm1SdWxlIHtcbiAgICAgICAgcmV0dXJuIG5ldyBjbGFzcyBpbXBsZW1lbnRzIElBbGFybVJ1bGUge1xuICAgICAgICAgICAgcHVibGljIHJlbmRlckFsYXJtUnVsZSgpOiBzdHJpbmcge1xuICAgICAgICAgICAgICAgIHJldHVybiBhbGFybVJ1bGU7XG4gICAgICAgICAgICB9XG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgc3RhdGljIGNvbmNhdChvcGVyYXRvcjogT3BlcmF0b3IsIC4uLm9wZXJhbmRzOiBJQWxhcm1SdWxlW10pOiBJQWxhcm1SdWxlIHtcbiAgICAgICAgcmV0dXJuIG5ldyBjbGFzcyBpbXBsZW1lbnRzIElBbGFybVJ1bGUge1xuICAgICAgICAgICAgcHVibGljIHJlbmRlckFsYXJtUnVsZSgpOiBzdHJpbmcge1xuICAgICAgICAgICAgICAgIGNvbnN0IGV4cHJlc3Npb24gPSBvcGVyYW5kc1xuICAgICAgICAgICAgICAgICAgICAubWFwKG9wZXJhbmQgPT4gYCR7b3BlcmFuZC5yZW5kZXJBbGFybVJ1bGUoKX1gKVxuICAgICAgICAgICAgICAgICAgICAuam9pbihgICR7b3BlcmF0b3J9IGApO1xuICAgICAgICAgICAgICAgIHJldHVybiBgKCR7ZXhwcmVzc2lvbn0pYDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfTtcbiAgICB9XG59XG4iXX0=