"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * A set of group metrics
 */
class GroupMetrics {
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        metrics === null || metrics === void 0 ? void 0 : metrics.forEach(metric => this._metrics.add(metric));
    }
    /**
     * Report all group metrics.
     */
    static all() {
        return new GroupMetrics();
    }
}
exports.GroupMetrics = GroupMetrics;
/**
 * Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 */
class GroupMetric {
    constructor(name) {
        this.name = name;
    }
}
exports.GroupMetric = GroupMetric;
/**
 * The minimum size of the Auto Scaling group
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * The maximum size of the Auto Scaling group
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * The number of instances that the Auto Scaling group attempts to maintain
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * The number of instances that are running as part of the Auto Scaling group
 * This metric does not include instances that are pending or terminating
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * The number of instances that are pending
 * A pending instance is not yet in service, this metric does not include instances that are in service or terminating
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * The number of instances that are in a Standby state
 * Instances in this state are still running but are not actively in service
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * The number of instances that are in the process of terminating
 * This metric does not include instances that are in service or pending
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * The total number of instances in the Auto Scaling group
 * This metric identifies the number of instances that are in service, pending, and terminating
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
class AutoScalingGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
    }
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
    addUserData(..._commands) {
        // do nothing
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            this.node.addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.anyValue({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props);
        this.spotPrice = props.spotPrice;
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
                this.osType = ec2.OperatingSystemType.UNKNOWN;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        // undefined is treated as 'true'
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => {
            var _a;
            return ({
                granularity: '1Minute',
                metrics: ((_a = group._metrics) === null || _a === void 0 ? void 0 : _a.size) !== 0 ? [...group._metrics].map(m => m.name) : undefined,
            });
        });
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launched
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: config.suspendProcesses !== undefined ? config.suspendProcesses :
            // Recommended list of processes to suspend from here:
            // https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
            [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
                ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS],
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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