"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const fs_1 = require("fs");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
;
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
class Schema {
    constructor(options) {
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
    }
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api) {
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: core_1.Lazy.stringValue({ produce: () => this.definition }),
            });
        }
        return this.schema;
    }
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.addToSchema(core_1.Lazy.stringValue({ produce: () => type.toString() }));
        return type;
    }
}
exports.Schema = Schema;
//# sourceMappingURL=data:application/json;base64,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